package cn.myapps.designtime.api.controller;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.api.ApiConfig;
import cn.myapps.designtime.api.service.ApiDesignTimeService;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import java.util.Iterator;
import java.util.List;

import static com.jayway.jsonpath.JsonPath.parse;

@Api(tags = "API中心操作模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class ApiDesignTimeController extends AbstractDesignTimeController {

    private ApiDesignTimeService apiService;

    public ApiDesignTimeController() {
        apiService = DesignTimeServiceManager.apiDesignTimeService();
    }

    /**
     * 获取api列表
     * @param applicationId
     *      软件id
     * @param name
     *      api名称
     * @param pageNo
     *      页数
     * @param linesPerPage
     *      每页显示条数
     * @return
     */
    @GetMapping(path = "/{applicationId}/apis")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取api列表", notes = "获取api列表")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
        @ApiImplicitParam(name = "name", value = "名称", required = false, paramType = "query", dataType = "string"),
        @ApiImplicitParam(name = "requestType", value = "请求类型", required = false, paramType = "query", dataType = "string"),
        @ApiImplicitParam(name = "status", value = "状态", required = false, paramType = "query", dataType = "string"),
        @ApiImplicitParam(name = "pageNo", value = "页码", required = false, paramType = "query", dataType = "int"),
        @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = false, paramType = "query", dataType = "int")
    })
    public Resource getApis(@PathVariable String applicationId,@RequestParam(required = false) String name, @RequestParam(required = false) String requestType, @RequestParam(required = false) String status, @RequestParam(required = false, defaultValue = "1") int pageNo, @RequestParam(required = false, defaultValue = "10") int linesPerPage) {
        try {
            DataPackage<ApiConfig> result = apiService.query(applicationId, name, requestType, "", status, pageNo, linesPerPage);
            return success("ok", result);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取api详情
     * @param eipId eipid
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/apis/{apiId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取api详情", notes = "获取api详情")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "apiId", value = "主键", required = true, paramType = "path", dataType = "string")
    })
    public Resource getApi(@PathVariable String apiId) throws Exception {
        try {
            ApiConfig apiConfig = apiService.findById(apiId);
            return success("ok", apiConfig);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建api
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/apis")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建api", notes = "新建api")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
        @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource createApi(@PathVariable String applicationId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            ApiConfig apiConfig = (ApiConfig)json2obj(json,ApiConfig.class);
            apiConfig.setApplicationid(applicationId);
            apiConfig.setParentId(applicationId);

            doSaveValidate(apiConfig);
            apiService.saveOrUpdate(apiConfig);

            JSONObject eipJson = new JSONObject();
            eipJson.put("id", apiConfig.getId());
            return success("ok", eipJson);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新api
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/apis/{apiId}")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "更新api", notes = "更新api")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
        @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource updateApi(@PathVariable String applicationId, @PathVariable String apiId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            ApiConfig ac = (ApiConfig)json2obj(json,ApiConfig.class);

            ApiConfig oldApiConfig = apiService.findById(apiId);

            ApiConfig apiConfig = (ApiConfig) BeanUtils.cloneBean(oldApiConfig);
            apiConfig.setApplicationid(applicationId);
            apiConfig.setName(ac.getName());
            apiConfig.setStatus(ac.getStatus());
            apiConfig.setTags(ac.getTags());
            apiConfig.setRequestType(ac.getRequestType());
            apiConfig.setRequestUrl(ac.getRequestUrl());
            apiConfig.setEncode(ac.getEncode());
            apiConfig.setResponseScript(ac.getResponseScript());
            apiConfig.setResponseType(ac.getResponseType());

            doSaveValidate(apiConfig);
            apiService.saveOrUpdate(apiConfig);
            return success("ok", null);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除api（可批量）
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/apis")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除api", notes = "删除api（可批量）")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doDeleteEips(@RequestBody String content) throws Exception {
        try {
            List<String> ids = parse(content).json();
            String[] idArray = {};
            idArray = ids.toArray(idArray);
            apiService.delete(idArray);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     * @throws Exception
     */
    private void doSaveValidate(ApiConfig ac) throws Exception {
        List<ApiConfig> list = apiService.list(ac.getParentId(),ac.getName());
        if(StringUtils.isBlank(ac.getId())) {
            //新建
            for (Iterator<ApiConfig> iterator = list.iterator(); iterator.hasNext();) {
                ApiConfig apiConfig = iterator.next();
                if(apiConfig.getName().equals(ac.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }else {
            //更新
            for (Iterator<ApiConfig> iterator = list.iterator(); iterator.hasNext();) {
                ApiConfig apiConfig = iterator.next();
                if(apiConfig.getId().equals(ac.getId())) {
                    continue;
                }
                if(apiConfig.getName().equals(ac.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }
    }
}
