package cn.myapps.destop.tray.fs;

import static java.nio.file.LinkOption.NOFOLLOW_LINKS;
import static java.nio.file.StandardWatchEventKinds.ENTRY_CREATE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_DELETE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_MODIFY;
import static java.nio.file.StandardWatchEventKinds.OVERFLOW;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.HashMap;
import java.util.Map;

import cn.myapps.destop.tray.SystemContacts;

/**
 * office文档仓库文件监控服务
 * @author Happy
 *
 */
public class OfficeRespWatchService {
	
    private final WatchService watcher;
    private final Map<WatchKey,Path> keys;
    private final boolean recursive;//是否递归目录
    private final WatchListener watchListener;
    
    
    public static void start(){
       
        	new Thread(new Runnable() {
				public void run() {
					 try {
						Path dir = Paths.get(SystemContacts.WATCH_PATH);
						new OfficeRespWatchService(dir, false).processEvents();
					 } catch (IOException e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
						}
				}
			}).start();
		
    }
    
    
    /**
     * 为参数设置的目录注册文件监控服务
     * @param dir
     * 		目录
     * @param recursive
     * 		是否递归目录
     * @throws IOException
     */
    OfficeRespWatchService(Path dir, boolean recursive) throws IOException {
    	if(!dir.toFile().exists()){
    		dir.toFile().mkdirs();
    	}
        this.watcher = FileSystems.getDefault().newWatchService();
        
        this.keys = new HashMap<WatchKey,Path>();
        
        this.recursive = recursive;
        
        this.watchListener = new StandardWatchListener();

        if (recursive) {
        	
            System.out.format("正在扫描目录： %s ...\n", dir);
            
            registerAll(dir);
            
            System.out.println("完成目录扫描并成功注册监听服务！");
            
        } else {
        	
            register(dir);
            
        }

    }

    /**
     * 为参数给定的目录注册监听服务
     * @param dir
     * @throws IOException
     */
    private void register(Path dir) throws IOException {
    	
        WatchKey key = dir.register(watcher, ENTRY_CREATE, ENTRY_DELETE, ENTRY_MODIFY);
        keys.put(key, dir);
    }

    
    /**
     * 为参数给定的目录（包含所有子目录）注册监听服务
     * @param start
     * 		起始目录
     * @throws IOException
     */
    private void registerAll(final Path start) throws IOException {
        Files.walkFileTree(start, new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs)
                throws IOException
            {
                register(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    /**
     * 事件处理
     */
    void processEvents() {
    	
        for (;;) {//这是个死循环

            WatchKey key;
            
            try {
                key = watcher.take();
            } catch (InterruptedException x) {
                return;
            }

            Path dir = keys.get(key);
            
            if (dir == null) {
            	
                System.err.println("WatchKey 不合法！");
                
                continue;
            }

            for (WatchEvent<?> event: key.pollEvents()) {
            	
                WatchEvent.Kind kind = event.kind();

                if (kind == OVERFLOW) {//OVERFLOW 事件处理
                	
                    continue;
                    
                }

                WatchEvent<Path> ev = (WatchEvent<Path>)event;
                Path name = ev.context();
                Path fullname = dir.resolve(name);
                
                if(kind == ENTRY_CREATE){//创建
                	watchListener.fileCreated(fullname.toString(), name.toString());
                }else if(kind == ENTRY_MODIFY){//修改
                	watchListener.fileModified(fullname.toString(), name.toString());
                }else if(kind == ENTRY_DELETE){//删除
                	watchListener.fileDeleted(fullname.toString(), name.toString());
                }

                if (recursive && (kind == ENTRY_CREATE)) {//递归监控子目录模式下，有文件夹创建，为新建的文件夹注册监控服务
                    try {
                        if (Files.isDirectory(fullname, NOFOLLOW_LINKS)) {
                            registerAll(fullname);
                        }
                    } catch (IOException x) {
                    }
                }
            }

            boolean valid = key.reset();
            if (!valid) {
                keys.remove(key);

                // 所有目录都无法访问 退出循环
                if (keys.isEmpty()) {
                    break;
                }
            }
        }
    }
    
    public static void main(String[] args) {
    	Path dir = Paths.get(SystemContacts.WATCH_PATH);
        
        try {
			new OfficeRespWatchService(dir, false).processEvents();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

}
