package cn.myapps.common;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.util.SequenceGenerator;

/**
 * 抽象的设计时Service实现,提供可复用的基础增删查改服务
 * @author Happy
 *
 */
public abstract class AbstractDesignTimeService<T extends DesignTimeSerializable> {


	protected abstract CommonDAO<T> getDao();

	/**
	 * 深度查询
	 * @param parentId
	 * @param fileSuffix
	 * @return
	 */

	public List<T> deepQueryAll(String parentId, String fileSuffix) throws Exception {
		return getDao().deepQueryAll(parentId, fileSuffix);
	}


	/**
	 * 根据uri查找实例
	 * @param uri
	 * @return
	 * @throws Exception
	 */
	public T findByUri(String uri) throws Exception {
		return getDao().findByUri(uri);
	}

	/**
	 * 根据id查找实例
	 * @param id
	 * @return
	 * @throws Exception
	 */
	public T findById(String id) throws Exception {
		return getDao().findById(id);
	}

	/**
	 * 保存
	 * @param object
	 * @throws Exception
	 */
	public void save(T object) throws Exception {
		if(StringUtils.isBlank(object.getId())){
			object.setId(SequenceGenerator.getDesignTimeSequence());
		}
		getDao().save(object);
	}

	/**
	 * 保存或更新
	 * <p>如果对象已经持久化，则执行更新操作，否则执行创建操作</p>
	 * @param object
	 * @throws Exception
	 */
	public void saveOrUpdate(T object) throws Exception{
		if(StringUtils.isBlank(object.getId())){
			object.setId(SequenceGenerator.getDesignTimeSequence());
			save(object);
		}else{
			update(object);
		}
	}

	/**
	 * 更新
	 * @param object
	 * @throws Exception
	 */
	public void update(T object) throws Exception{
		getDao().save(object);
	}

	/**
	 * 从文件系统中批量删除文件
	 * @param id
	 */
	public void delete(String id) throws Exception {
		getDao().delete(id);
	}

	/**
	 * 从文件系统中批量删除文件
	 * @param ids
	 */
	public void delete(String[] ids) throws Exception  {
		for(String id : ids){
			getDao().delete(id);
		}
	}

	/**
	 * 查询符合条件的对象集合
	 * @param parentId
	 * 		父id
	 * @param name
	 * 		文件名关键字
	 * @param page
	 * 		当前页数
	 * @param lines
	 * 		每页行数
	 * @return
	 * @throws Exception
	 */
	public DataPackage<T> query(String parentId,String name,int page,int lines) throws Exception {
		DataPackage<T> dp = new DataPackage<T>();

		try {
			List<T> list = getDao().queryByName(parentId, getPathSuffix(), getFileSuffix(), name);

			//判断页数过大的处理
			if((int) Math.ceil((double) list.size() / (double) lines) < page) {
				page = 1;
			}

			dp.setRowCount(list.size());
			dp.setPageNo(page);
			dp.setLinesPerPage(lines);
			dp.setDatas(listPage(list, page,lines));
		} catch (Exception e) {
			e.printStackTrace();
			throw e;
		}

		return dp;
	}

	/**
	 * 查询符合条件的对象集合
	 * @param parentId
	 * 		父id
	 * @param searchword
	 * 		文件名或者描述关键字
	 * @param page
	 * 		当前页数
	 * @param lines
	 * 		每页行数
	 * @return
	 * @throws Exception
	 */
	public DataPackage<T> queryByNameOrDescript(String parentId, String searchword, int page, int lines) throws Exception {
		DataPackage<T> dp = new DataPackage<T>();
		try {
			List<T> list = getDao().queryByNameOrDescript(parentId, getPathSuffix(), getFileSuffix(), searchword);

			//判断页数过大的处理
			if((int) Math.ceil((double) list.size() / (double) lines) < page) {
				page = 1;
			}

			dp.setRowCount(list.size());
			dp.setPageNo(page);
			dp.setLinesPerPage(lines);
			dp.setDatas(listPage(list, page,lines));
		} catch (Exception e) {
			throw e;
		}

		return dp;
	}

	/**
	 * 查询符合条件的对象集合
	 * @param parentId
	 * 		父id
	 * @param name 名称
	 *
	 * @param description  描述
	 * @param page
	 * 		当前页数
	 * @param lines
	 * 		每页行数
	 * @return
	 * @throws Exception
	 */
//	public DataPackage<T> queryByNameAndDescript(String parentId, String name,String description, int page, int lines) throws Exception {
//		DataPackage<T> dp = new DataPackage<T>();
//		try {
//			List<T> list = getDao().queryByNameAndDescript(parentId, getPathSuffix(), getFileSuffix(), name, description);
//
//			//判断页数过大的处理
//			if((int) Math.ceil((double) list.size() / (double) lines) < page) {
//				page = 1;
//			}
//
//			dp.setRowCount(list.size());
//			dp.setPageNo(page);
//			dp.setLinesPerPage(lines);
//			dp.setDatas(listPage(list, page,lines));
//		} catch (Exception e) {
//			e.printStackTrace();
//			throw e;
//		}
//
//		return dp;
//	}

	public List<T> queryAccurateByName(String parentId, String name) throws Exception{
		return getDao().queryAccurateByName(parentId, getPathSuffix(), getFileSuffix(), name);
	}

	private List<T> listPage(List<T> data,int page,int lines){
		/** 总数据条数 */
		int totalCount = data.size();
		/** 总页数 */
		int totalPage = (totalCount + lines - 1) / lines;
		/** 上一页 */
		int lastPage = page-1>1? page-1:1;
		/** 下一页 */
		int nextPage = page>=totalPage? totalPage: page + 1;

		int fromIndex = (page - 1) * lines;
		if (fromIndex >= data.size()) {
			return Collections.emptyList();//空数组
		}
		if(fromIndex<0){
			return Collections.emptyList();//空数组
		}
		int toIndex = page * lines;
		if (toIndex >= data.size()) {
			toIndex = data.size();
		}
		List<T> newList = new ArrayList<T>(data.subList(fromIndex, toIndex));
		return newList;
	}

	/**
	 * 查询符合条件的对象集合
	 * @param parentId
	 * 		文件存放的目录
	 * @param name
	 * 		文件名关键字
	 * @return
	 * @throws Exception
	 */
	public List<T> list(String parentId,String name) throws Exception {
		return getDao().queryByName(parentId, getPathSuffix(), getFileSuffix(), name);  // 模糊匹配
	}

	/**
	 * 查询符合条件的对象集合
	 * @param parentId
	 * 		文件存放的目录
	 * @param name
	 * 		文件名关键字
	 * @return
	 * @throws Exception
	 */
	public List<T> list(String parentId, String pathSuffix, String fileSuffix, String name) throws Exception {
		return getDao().queryByName(parentId, pathSuffix, fileSuffix, name);
	}

	/**
	 * 根据名称获取直属下级 父类
	 * @param id
	 * @param name
	 * @return
	 * @throws Exception
	 */
	public List<T> getSubordinate(String id,String name) throws Exception {
		return getDao().querySubordinate(id, name);
	}

	/**
	 * 根据名称获取直属下级 子类
	 * @param id
	 * @param name
	 * @return
	 * @throws Exception
	 */
	public List<T> getSubordinateChildren(String id,String name) throws Exception {
		return getDao().querySubordinate(id,getPathSuffix(),getFileSuffix(), name);
	}

	/**
	 * 根据上级id获取直属下级
	 * @param id
	 * @param name
	 * @return
	 * @throws Exception
	 */
	public List<T> getChildsByParentId(String id,String pathSuffix, String fileSuffix,String name) throws Exception {
		return getDao().queryByParentId(id,pathSuffix,fileSuffix, name);
	}

	/**
	 * 根据上级id获取直属下级
	 * @param id
	 * @param name
	 * @return
	 * @throws Exception
	 */
	public List<T> getChildsByParentId(String id,String name) throws Exception {
		return getDao().queryByParentId(id,getPathSuffix(),getFileSuffix(), name);
	}

	protected abstract String getPathSuffix();

	protected abstract String getFileSuffix();



}
