package com.bcxin.tenant.backend.tasks;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.Infrastructures.components.CacheProvider;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.components.models.DataCacheItem;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.backend.V5AppConfig;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

@Component
public class V5ScheduleTask extends TaskAbstract{
    private final CacheProvider cacheProvider;
    private final V5AppConfig appConfig;
    private final HttpClient httpClient;
    private final JsonProvider jsonProvider;
    private final Logger logger = LoggerFactory.getLogger(V5ScheduleTask.class);

    public V5ScheduleTask(CacheProvider cacheProvider, V5AppConfig appConfig, HttpClient httpClient, JsonProvider jsonProvider) {
        this.cacheProvider = cacheProvider;
        this.appConfig = appConfig;
        this.httpClient = httpClient;
        this.jsonProvider = jsonProvider;
    }

    @Override
    protected void runCore() {
        Collection<String> appIds =
                this.cacheProvider.get("V5.Apps", () -> {
                    try {
                        String url = appConfig.getSiteEndPoint("/build-metas/apps");
                        HttpGet httpGet = new HttpGet(url);
                        HttpResponse response = this.httpClient.execute(httpGet);
                        String responseBody = EntityUtils.toString(response.getEntity());
                        Collection<JSONObject> result = this.jsonProvider.toObjects(JSONObject.class, responseBody);

                        Collection<String> responseAppIds = result.stream().map(ii -> {
                            return ii.getString("id");
                        }).collect(Collectors.toList());

                        return DataCacheItem.create(responseAppIds);
                    } catch (Exception ex) {
                        ex.printStackTrace();
                        logger.error("获取Apps信息异常", ex);
                        throw new BadTenantException("获取Apps信息异常", ex);
                    }
                }).getData();

        appIds.forEach(appId -> {
            try {
                executeAppTasks(appId);
            } catch (Exception ex) {
                logger.error("执行App{}的任务异常:{}", appId, ex);
            }
        });
    }


    private void executeAppTasks(String appId) {
        Collection<Map<String, String>> triggerIdMaps =
                this.cacheProvider.get("V5.App(%s).triggers", () -> {
                    try {
                        String url = appConfig.getSiteEndPoint(String.format("/build-metas/apps/%s/jobs", appId));
                        HttpGet httpGet = new HttpGet(url);
                        HttpResponse response = this.httpClient.execute(httpGet);
                        String responseBody = EntityUtils.toString(response.getEntity());
                        Collection<JSONObject> result = this.jsonProvider.toObjects(JSONObject.class, responseBody);

                        Collection<Map<String, String>> responseAppIds = result.stream().map(ii -> {
                            return Collections.singletonMap(ii.getString("id"), ii.getString("appId"));
                        }).collect(Collectors.toList());

                        return DataCacheItem.create(responseAppIds);
                    } catch (Exception ex) {
                        ex.printStackTrace();
                        logger.error("获取App{}.Trigger信息异常：{}", appId, ex);
                        throw new BadTenantException("获取Apps信息异常", ex);
                    }
                }).getData();

        triggerIdMaps.forEach(item -> {
            try {
                String selectedTriggerId = item.keySet().stream().findFirst().get();
                String selectedAppId = item.get(selectedTriggerId);

                String url = appConfig.getSiteEndPoint(String.format("/build-metas/apps/%s/jobs/%s/execute", selectedAppId, selectedTriggerId));
                HttpPost httpPost = new HttpPost(url);
                HttpResponse response = this.httpClient.execute(httpPost);
                String responseBody = EntityUtils.toString(response.getEntity());
                logger.info("执行App{}.Trigger:{}成功过:{}", selectedAppId, selectedTriggerId,responseBody);
                System.err.println(String.format("执行App%s.Trigger:%s成功过:%s", selectedAppId, selectedTriggerId,responseBody));
            } catch (Exception ex) {
                ex.printStackTrace();
                logger.error("获取App{}.Trigger信息异常：{}", appId, ex);
                throw new BadTenantException("获取Apps信息异常", ex);
            }
        });
    }

    @Override
    public String getTaskName() {
        return "v5.schedule.tasks";
    }
}
