package com.bcxin.tenant.backend;

import com.bcxin.tenant.backend.tasks.TaskAbstract;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;

import java.util.Collection;
import java.util.stream.Collectors;

@Configuration
@EnableScheduling
public class ScheduleJobConfig implements ApplicationContextAware {

    private ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Scheduled(fixedDelay = 23 * 60 * 60 * 1000)
    public void executeAppJob() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("v5.app");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    @Scheduled(fixedDelay = 2  * 60 * 1000)
    public void executeDataExchangeJob() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("v5.sync.data.tasks.normal");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    @Scheduled(fixedDelay = 20 * 60 * 1000)
    public void executeFailedDataExchangeJob() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("v5.sync.data.tasks.failed");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    /**
     * ִ执行员工是否投保定时任务
     */
    @Scheduled(fixedDelay = 120 * 60 * 1000)
    public void executeEmployeeIsSureChangeJob() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("employee.data");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    /**
     * 补充证书信息
     */
    @Scheduled(fixedDelay = 60 * 60 * 1000)
    public void executeEmployeeCredentialSyncJob() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("employee.credential");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    /**
     * 湖南审计日志上传
     */
    @Scheduled(cron = "${hunan.logReport}")
    public void executeHuNanLogReportTask() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("log.event");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    /**
     * description：定时删除部门数据、离职部门人员
     * author：linchunpeng
     * date：2024/12/13
     */
    @Scheduled(cron = "${scheduled.cron.delete-department}")
    public void executeDepartmentDeleteTask() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("department.delete.task");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    @Bean
    public TaskScheduler taskScheduler() {
        ThreadPoolTaskScheduler taskScheduler = new ThreadPoolTaskScheduler();
        taskScheduler.setPoolSize(50);
        return taskScheduler;
    }

    protected Collection<TaskAbstract> getTasks(String prefixBeanName) {
        Collection<TaskAbstract> tasks =
                this.applicationContext.getBeansOfType(TaskAbstract.class)
                        .values().stream().filter(ii -> ii.getTaskName().startsWith(prefixBeanName))
                        .collect(Collectors.toList());

        return tasks;
    }
}
