package com.bcxin.rest.web.apis.controllers;

import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.components.CacheV2Provider;
import com.bcxin.api.interfaces.rbacs.RbacCategoryRpcProvider;
import com.bcxin.api.interfaces.rbacs.responses.CategoryDetailGetResponse;
import com.bcxin.api.interfaces.rbacs.responses.ResourceGetResponse;
import com.bcxin.api.interfaces.tenants.EmployeeRpcProvider;
import com.bcxin.api.interfaces.tenants.OrganizationRpcProvider;
import com.bcxin.api.interfaces.tenants.UserRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.EmployeeCriteria;
import com.bcxin.api.interfaces.tenants.requests.employees.BatchPermissionRequest;
import com.bcxin.api.interfaces.tenants.responses.AppRolePermissionResponse;
import com.bcxin.api.interfaces.tenants.responses.EmployeeGetResponse;
import com.bcxin.api.interfaces.tenants.responses.MyOrganizationProfileGetResponse;
import com.bcxin.rest.web.apis.responses.EmployeeAppRoleGetResponse;
import com.bcxin.rest.web.apis.utils.CacheUtil;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/tenant/organizations/{organizationId}")
public class MyAppController extends ControllerAbstract{
    private final RbacCategoryRpcProvider categoryRpcProvider;
    private final OrganizationRpcProvider organizationRpcProvider;
    private final EmployeeRpcProvider employeeRpcProvider;
    private final CacheV2Provider cacheV2Provider;
    private final UserRpcProvider userRpcProvider;

    public MyAppController(RbacCategoryRpcProvider categoryRpcProvider,
                           OrganizationRpcProvider organizationRpcProvider,
                           EmployeeRpcProvider employeeRpcProvider,
                           CacheV2Provider cacheV2Provider,
                           UserRpcProvider userRpcProvider) {
        this.categoryRpcProvider = categoryRpcProvider;
        this.organizationRpcProvider = organizationRpcProvider;
        this.employeeRpcProvider = employeeRpcProvider;
        this.cacheV2Provider = cacheV2Provider;
        this.userRpcProvider = userRpcProvider;
    }

    @GetMapping("/my-apps")
    public ResponseEntity getMyApp(@PathVariable String organizationId) {
        Collection<String> appIds =
                this.organizationRpcProvider.getApps(organizationId);

        Collection<CategoryDetailGetResponse> responses =
                this.categoryRpcProvider.getByIds(appIds);

        if(TenantContext.getInstance().getUserContext()!=null &&
                !TenantContext.getInstance().getUserContext().isDomainAdmin()) {
            AppRolePermissionResponse appRoles =
                    this.employeeRpcProvider.getPermissions(organizationId, getSelectedEmployeeId());
            for (CategoryDetailGetResponse cr : responses) {
                for (ResourceGetResponse rr : cr.getResponses()) {
                    if (!appRoles.getApps().stream().anyMatch(ix -> ix.getRoles().stream().anyMatch(it -> it.getId().equals(rr.getId())))) {
                        rr.assignPermissionStatus(false);
                    }
                }
            }
        }

        return this.ok(responses);
    }

    @PostMapping("/employees/batch-roles")
    public ResponseEntity post_my_permissions(@PathVariable String organizationId,
                                              @RequestBody BatchPermissionRequest request) {
        request.setFromAdmin(this.isCurrentOrgDomainOrDepartAdmin());
        this.employeeRpcProvider.batchPermissions(organizationId, request);

        return this.ok();
    }

    @GetMapping("/employees/{employeeId}/roles")
    public ResponseEntity get_my_permissions(@PathVariable String organizationId,
                                             @PathVariable String employeeId) {
        AppRolePermissionResponse appRoles =
                this.employeeRpcProvider.getPermissions(organizationId, employeeId);

        return this.ok(appRoles);
    }


    @PostMapping("/employee-permissions")
    public ResponseEntity employee_permissions(@PathVariable String organizationId,
                                             @RequestBody EmployeeCriteria criteria) {
        return search_permissions(organizationId, criteria);
    }

    @PostMapping("/permissions/search")
    public ResponseEntity search_permissions(@PathVariable String organizationId,
                                             @RequestBody EmployeeCriteria criteria) {
        Pageable<EmployeeGetResponse> employeeGetResponsePageable = this.employeeRpcProvider.find(organizationId, criteria);
        Collection<String> employeeIds = employeeGetResponsePageable.getData().stream()
                .map(ii -> ii.getId()).collect(Collectors.toList());

        Collection<AppRolePermissionResponse> appRolePermissionResponses =
                this.employeeRpcProvider.getBatchPermissions(organizationId, employeeIds);

        Collection<EmployeeAppRoleGetResponse> employeeAppRoleGetResponses =
                employeeGetResponsePageable.getData().stream().map(ii -> {
                    Optional<AppRolePermissionResponse> selectedAppRoleOptional =
                            appRolePermissionResponses.stream().filter(apr -> apr.getEmployeeId().equals(ii.getId()))
                                    .findFirst();

                    return EmployeeAppRoleGetResponse.create(
                            ii.getId(),
                            ii.getName(), ii.getTelephone(), ii.getDepartName(),
                            ii.getOccupationType(),
                            selectedAppRoleOptional.isPresent() ? selectedAppRoleOptional.get() : null);
                }).collect(Collectors.toList());


        return this.ok(Pageable.create(employeeGetResponsePageable.getPageIndex(), employeeGetResponsePageable.getPageSize(), employeeGetResponsePageable.getTotalCount(),
                employeeAppRoleGetResponses));
    }
}
