package com.bcxin.rest.web.apis.controllers;

import cn.hutool.core.util.StrUtil;
import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.components.CacheV2Provider;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.components.UniqueVerificationCodeGenerator;
import com.bcxin.Infrastructures.components.models.VerificationCode;
import com.bcxin.Infrastructures.exceptions.ForbidTenantException;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.api.interfaces.commons.OperateLogRpcProvider;
import com.bcxin.api.interfaces.identities.IdentityRpcProvider;
import com.bcxin.api.interfaces.identities.requests.BindWechatRequest;
import com.bcxin.api.interfaces.identities.requests.ChangeUserNameRequest;
import com.bcxin.api.interfaces.tenants.EmployeeRpcProvider;
import com.bcxin.api.interfaces.tenants.OrganizationRpcProvider;
import com.bcxin.api.interfaces.tenants.UserRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.operatelog.SearchOperateLogRequest;
import com.bcxin.api.interfaces.tenants.requests.organizations.CreateOrganizationRequest;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.*;
import com.bcxin.api.interfaces.tenants.responses.*;
import com.bcxin.rest.web.apis.caches.ExternalMemberInviteCodeCache;
import com.bcxin.rest.web.apis.requests.GenerateExternalInviteCodeRequest;
import com.bcxin.rest.web.apis.responses.ApiDepartmentTreeGetResponse;
import com.bcxin.rest.web.apis.utils.CacheUtil;
import io.swagger.annotations.*;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 获取当前用户信息
 */
@Api("当前用户信息")
@RestController
@RequestMapping("/tenant/users/current")
public class CurrentUserController extends ControllerAbstract {
    private final UserRpcProvider userRpcProvider;
    private final EmployeeRpcProvider employeeRpcProvider;
    private final IdentityRpcProvider identityRpcProvider;
    private final CacheV2Provider cacheV2Provider;
    private final OperateLogRpcProvider operateLogRpcProvider;

    private final OrganizationRpcProvider organizationRpcProvider;

    private final RedisTemplate redisTemplate;
    private final JsonProvider jsonProvider;

    private final UniqueVerificationCodeGenerator uniqueVerificationCodeGenerator;

    private final boolean updateNicknameHeadPhoto;
    private final String envRegionCode;

    public CurrentUserController(UserRpcProvider userRpcProvider,
                                 EmployeeRpcProvider employeeRpcProvider,
                                 IdentityRpcProvider identityRpcProvider,
                                 CacheV2Provider cacheV2Provider,
                                 OperateLogRpcProvider operateLogRpcProvider,
                                 OrganizationRpcProvider organizationRpcProvider,
                                 RedisTemplate redisTemplate,
                                 JsonProvider jsonProvider,
                                 UniqueVerificationCodeGenerator uniqueVerificationCodeGenerator,
                                 @Value("${tenant-user-config.update-nickname-headphoto:false}") Boolean updateNicknameHeadPhoto,
                                 @Value("${env.region-code:110000}") String envRegionCode) {
        this.userRpcProvider = userRpcProvider;
        this.employeeRpcProvider = employeeRpcProvider;
        this.identityRpcProvider = identityRpcProvider;
        this.cacheV2Provider = cacheV2Provider;
        this.operateLogRpcProvider = operateLogRpcProvider;
        this.organizationRpcProvider = organizationRpcProvider;
        this.redisTemplate = redisTemplate;
        this.jsonProvider = jsonProvider;
        this.uniqueVerificationCodeGenerator = uniqueVerificationCodeGenerator;
        this.updateNicknameHeadPhoto = updateNicknameHeadPhoto;
        this.envRegionCode = envRegionCode;
    }

    /**
     * @return
     */
    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiResponses({
            @ApiResponse(code = 200, message = "返回当前租户的基本信息",
                    response = UserProfileGetResponse.class,
                    examples = @Example({
                            @ExampleProperty(mediaType = "id", value = "xxxx"),
                            @ExampleProperty(mediaType = "name", value = "张三"),
                            @ExampleProperty(mediaType = "lonLatJson", value = "{lat:11.222,lon:2323232}"),
                            @ExampleProperty(mediaType = "birthdate", value = "2021-01-01"),
                            @ExampleProperty(mediaType = "checkedStatus", value = "34323"),
                            @ExampleProperty(mediaType = "authenticateStatus", value = "34323"),
                            @ExampleProperty(mediaType = "authenticatedResult", value = "34323"),
                            @ExampleProperty(mediaType = "stature", value = "1.8"),
                            @ExampleProperty(mediaType = "userType", value = "xxx")
                    })),
            @ApiResponse(code = 404, message = "找不到租户信息", response = Valid.class)
    })
    @ApiOperation("获取当前租户(不是公司的职员)的profile")
    @GetMapping("/profile")
    public ResponseEntity<UserProfileGetResponse> profile(HttpServletResponse servletResponse) {

        UserProfileGetResponse response = this.cacheV2Provider.get(CacheUtil.getCurrentProfileCacheKey(), () -> {
            return this.userRpcProvider.get(this.getCurrentUserId());
        }, 60 * 60 * 24);
        response.setUpdateNicknameHeadPhoto(updateNicknameHeadPhoto);
        response.setEnvRegionCode(envRegionCode);
        return this.ok(response);
    }

    @GetMapping("/detail")
    public ResponseEntity<UserDetailGetResponse> detail(HttpServletResponse servletResponse) {
        UserDetailGetResponse response = this.userRpcProvider.getDetail(this.getCurrentUserId());
        response.setUpdateNicknameHeadPhoto(updateNicknameHeadPhoto);
        response.setEnvRegionCode(envRegionCode);
        return this.ok(response);
    }

    @GetMapping("/credentials")
    public ResponseEntity<Collection<CurrentCredentialResponse>> credentials(QueryCredentialRequest queryRequest) {
        queryRequest.setUserId(this.getCurrentUserId());
        List<CurrentCredentialResponse> response = this.userRpcProvider.getCredentials(queryRequest);
        return this.ok(response);
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("获取当前租户的组织列表")
    @ApiResponses({
            @ApiResponse(code = 200, message = "返回当前租户的组织列表", response = ApiDepartmentTreeGetResponse.class, responseContainer = "list"),
            @ApiResponse(code = 404, message = "找不到租户信息", response = Valid.class)
    })
    @GetMapping("/organizations")
    public ResponseEntity<Collection<MyOrganizationProfileGetResponse>> myOrgans(HttpServletResponse servletResponse) {
        Collection<MyOrganizationProfileGetResponse> response =
                this.userRpcProvider.getById(
                        this.getCurrentUserId()
                );

        return this.ok(response);
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("获取当前的团队信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "返回当前租户 + 团队列表", response = ApiDepartmentTreeGetResponse.class, responseContainer = "list")
    })
    @GetMapping("/teams")
    public ResponseEntity<Collection<MyTeamGetResponse>> myTeams() {
        Collection<MyTeamGetResponse> response =
                this.userRpcProvider.getMyTeamsByUserId(
                        this.getCurrentUserId()
                );

        return this.ok(response);
    }

    @GetMapping("/organizations/{organizationId}/departs")
    public ResponseEntity myOrganDepart(@PathVariable String organizationId) {
        Collection<MyImDepartGetResponse> responses =
                this.employeeRpcProvider.getMyImDeparts(organizationId, this.getCurrentUserId());

        return this.ok(responses.stream().sorted((t1, t2) -> t2.getDisplayOrder() - t1.getDisplayOrder()).collect(Collectors.toList()));
    }

    @GetMapping("/organizations/{organizationId}/departs/{departId}/contacts")
    public ResponseEntity myOrganContact(@PathVariable String organizationId, @PathVariable String departId) {
        Collection<MyImContactGetResponse> responses =
                this.employeeRpcProvider.getMyImContacts(IMContactCriteria.create(0,9999,organizationId,
                        departId,this.getCurrentUserId()
                ));

        return this.ok(responses);
    }


    @PostMapping("/organizations/{organizationId}/contacts/search")
    public ResponseEntity myOrganContact(@PathVariable String organizationId, @RequestBody IMContactCriteria request) {
        if(StrUtil.isEmpty(request.getKeyword())){
            return this.error("请输入要搜索的人员信息!");
        }
        request.setOrganizationId(organizationId);
        request.setUserId(this.getCurrentUserId());
        Collection<MyImContactGetResponse> responses = this.employeeRpcProvider.getMyImContactSearch(request);

        return this.ok(responses);
    }

    @PutMapping("/sync-location")
    public ResponseEntity SyncLocation(@RequestBody SyncTenantUserLocationRequest request) {
        /*
        this.userRpcProvider.update(this.getCurrentUserId(), request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        */
        return this.ok();
    }

    @PutMapping("/sync-device")
    public ResponseEntity SyncLocation(@RequestBody SyncCidRequest request) {
        this.userRpcProvider.updateDeviceId(this.getCurrentUserId(), request.getCid());
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("上传证件信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class),
            @ApiResponse(code = 400, message = "如果有保安员的职业身份, 则必须上传身份证且大于18岁", response = Valid.class)
    })
    @PostMapping("/update-credential")
    public ResponseEntity UpdateCredential(@RequestBody UpdateCredentialRequest request) {
        this.userRpcProvider.updateCredential(this.getCurrentUserId(), request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("针对百课堂的假身份核验")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class),
            @ApiResponse(code = 400, message = "如果有保安员的职业身份, 则必须上传身份证且大于18岁", response = Valid.class)
    })
    @PostMapping("/bkt/check-status")
    public ResponseEntity doPostReCheckStatus(@RequestBody UpdateCredentialRequest request) {
        this.userRpcProvider.updateCredentialByBkt(this.getCurrentUserId(), request);
        this.cacheV2Provider.del(CacheUtil.getCheckStatusForBkt(this.getCurrentUserId()));

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("查看针对百课堂的身份核验")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class)
    })
    @GetMapping("/bkt/check-status")
    public ResponseEntity getCheckStatusByBkt() {
        boolean status = this.cacheV2Provider.get(CacheUtil.getCheckStatusForBkt(this.getCurrentUserId()), () -> {
            return this.userRpcProvider.getCheckStatusByBkt(this.getCurrentUserId());
        });

        return this.ok(status);
    }

    @ApiOperation(value = "更改身份证正反面信息-只针对已核验用户可用", response = Void.class)
    @PostMapping("/upload-credential-image")
    public ResponseEntity change_credential( @RequestBody UpdateCredentialImageRequest request){
        this.userRpcProvider.changeCredential(this.getCurrentUserId(),request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }


    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("上传一寸免冠照片")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/upload-photo")
    public ResponseEntity updatePhotos(@RequestBody UpdateTenantPhotoRequest request) {
        this.userRpcProvider.updatePhoto(this.getCurrentUserId(), request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("绑定微信")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/wechat/binding")
    public ResponseEntity bindWechat(@RequestBody BindWechatRequest request) {
        this.identityRpcProvider.bindWechat(request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }
    @ApiOperation("判断是否绑定unionid")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/wechat/unionid")
    public ResponseEntity unionid() {
        boolean result = this.identityRpcProvider.isBindUnionId(this.getCurrentUserId());
        return this.ok(result);
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("解除绑定微信")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/wechat/unbinding")
    public ResponseEntity unbindWechat() {
        this.identityRpcProvider.unbindWechat(this.getCurrentUserId());
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("重新进行实名认证")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/re-real-name-authenticate")
    public ResponseEntity reRealNameAuthenticate(@RequestParam(name = "checkAuthenticate",defaultValue = "false") boolean checkAuthenticate) {
        this.userRpcProvider.reRealNameAuthenticate(this.getCurrentUserId(),checkAuthenticate);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("创建企业信息;创建完成之后; 如果返回成功; 那么")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/organizations")
    public ResponseEntity doCreateOrganization(@Validated @RequestBody CreateOrganizationRequest request) {
        this.organizationRpcProvider.create(request);

        return this.ok();
    }

    @ApiOperation("获取我申请的组织列表")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @GetMapping("/my-registration-organizations")
    public ResponseEntity getMyRegistrationOrgans(@RequestParam(value = "pageIndex",required = false,defaultValue = "1") int pageIndex,
                                                  @RequestParam(value = "pageSize",required = false,defaultValue = "10") int pageSize
                                                  ) {
        Collection<MyRegistrationOrganizationResponse> organs =
                this.organizationRpcProvider.getMyRegistrationOrganizations(pageIndex, pageSize);

        return this.ok(organs);
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("刷新当前企业的邀请码/二维码")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/organizations/{organizationId}/generate-invite-codes")
    public ResponseEntity<ExternalMemberInviteCodeResponse> generateCode(
            @PathVariable String organizationId,
            @RequestBody GenerateExternalInviteCodeRequest request
            ) throws IOException {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("当前用户无效");
        }

        if (!StringUtils.hasLength(userModel.getOrganId())) {
            throw new ForbidTenantException("非团体禁止生成二维码操作");
        }

        ExternalMemberInviteCodeCache codeCache = ExternalMemberInviteCodeCache.create(
                userModel, request.getGroupId(), request.getGroupName()
        );

        String key = userModel.getOrganId();
        if (key.contains("--__")) {
            key = key.substring(0, key.indexOf("--__"));
        }

        String cacheKey = String.format("%s:%s", key, request.getGroupId());

        VerificationCode inviteCode = uniqueVerificationCodeGenerator.execute(cacheKey);
        String jsonData = this.jsonProvider.getJson(codeCache);
        this.redisTemplate.opsForValue().set(
                ExternalMemberInviteCodeCache.getInviteCodeCacheKey(inviteCode.getCode()),
                jsonData, 5, TimeUnit.DAYS
        );

        return this.ok(
                ExternalMemberInviteCodeResponse.create(inviteCode.getCode(), inviteCode.getRetryIndex())
        );
    }

    @ApiOperation(value = "修改昵称/头像")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/update/nick-head-photo")
    public ResponseEntity updateNickHeadPhoto(@RequestBody UpdateNickHeadPhotoRequest request) {
        if (StringUtils.isEmpty(request.getNick()) && StringUtils.isEmpty(request.getHeadPhoto())) {
            throw new UnAuthorizedTenantException("参数不能为空");
        }
        this.userRpcProvider.updateNickHeadPhoto(this.getCurrentUserId(), request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }


    @ApiOperation(value = "获取我的操作日志", response = OperateLogResponse.class)
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/operate/log")
    public ResponseEntity operateLog(@RequestBody SearchOperateLogRequest request) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("当前用户无效");
        }
        Pageable<OperateLogResponse> page = operateLogRpcProvider.getListByUserId(request, userModel.getId());
        return this.ok(page);
    }


       @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("修改当前的用户名信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PutMapping("/change-user-name")
    public void changeUserName(@RequestBody ChangeUserNameRequest request) {
        this.identityRpcProvider.changedUserName(request);
    }
}
