package com.bcxin.rest.web.apis.controllers;

import cn.hutool.core.collection.ListUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.metadata.fill.FillConfig;
import com.bcxin.Infrastructures.Pageable;
import com.bcxin.api.interfaces.commons.CommonImportResponse;
import com.bcxin.api.interfaces.rbacs.custom.RbacCustomRoleManagerProvider;
import com.bcxin.api.interfaces.rbacs.custom.request.*;
import com.bcxin.api.interfaces.rbacs.custom.response.RbacCustomRoleResponse;
import com.bcxin.api.interfaces.rbacs.custom.response.RbacUserResponse;
import com.bcxin.api.interfaces.tenants.requests.employees.BatchImportRoleMemberRequest;
import com.bcxin.infrastructure.offices.utils.TitleHandler;
import com.bcxin.rest.web.apis.ExtractDataComponent;
import com.bcxin.rest.web.apis.requests.BatchImportDataRequest;
import com.bcxin.rest.web.apis.utils.FileUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.ClassPathResource;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.io.*;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@RestController
@RequestMapping("/tenant/organizations/{organizationId}/role")
@Api(value = "自定义角色管理api", tags = "自定义角色管理api")
@AllArgsConstructor
public class RoleManageController extends ControllerAbstract {
    private final RbacCustomRoleManagerProvider rbacCustomRoleManagerProvider;
    private final ExtractDataComponent extractDataComponent;


    @ApiOperation(value = "获取企业自定义角色列表", response = RbacCustomRoleResponse.class)
    @PostMapping("/search")
    public ResponseEntity<List<RbacCustomRoleResponse>> getRbacCustomRoleList(@PathVariable String organizationId, @Valid @RequestBody RbacCustomRoleSearchRequest request) {
        request.setOrganizationId(organizationId);
        List<RbacCustomRoleResponse> list = this.rbacCustomRoleManagerProvider.findOrgRoleList(request);
        return this.ok(list);
    }


    @ApiOperation(value = "分页查询用户职员列表", response = RbacUserResponse.class)
    @PostMapping("/user/page")
    public ResponseEntity<Pageable<RbacUserResponse>> getRbacUserDtoList(@PathVariable String organizationId, @Valid @RequestBody RbacUserSearchRequest request) {
        request.setOrganizationId(organizationId);
        Pageable<RbacUserResponse> page = this.rbacCustomRoleManagerProvider.getRbacUserDtoList(request);
        return this.ok(page);
    }

    @ApiOperation(value = "企业新增自定义角色", response = String.class)
    @PostMapping("/add")
    public ResponseEntity<String> addRole(@PathVariable String organizationId, @Valid @RequestBody RbacCustomRoleAddRequest request) {
        request.setOrganizationId(organizationId);
        this.rbacCustomRoleManagerProvider.addRole(request);
        return this.ok("操作成功");
    }

    @ApiOperation(value = "企业编辑角色", response = String.class)
    @PostMapping("/edit")
    public ResponseEntity<String> editRole(@PathVariable String organizationId, @Valid @RequestBody RbacCustomRoleEditRequest request) {
        request.setOrganizationId(organizationId);
        this.rbacCustomRoleManagerProvider.editRole(request);
        return this.ok("操作成功");
    }

    @ApiOperation(value = "根据id查询角色详情", response = RbacCustomRoleResponse.class)
    @PostMapping("/{roleId}")
    public ResponseEntity<RbacCustomRoleResponse> findById(@PathVariable String organizationId, @PathVariable Long roleId) {
        return this.ok(this.rbacCustomRoleManagerProvider.findById(roleId));
    }

    @ApiOperation(value = "根据id复制角色", response = String.class)
    @PostMapping("/copy")
    public ResponseEntity<String> copyRole(@PathVariable String organizationId, @Valid @RequestBody RbacCustomRoleAddRequest request) {
        this.rbacCustomRoleManagerProvider.copyRole(request);
        return this.ok("操作成功");
    }

    @ApiOperation(value = "根据id列表删除角色", response = String.class)
    @PostMapping("/delete")
    public ResponseEntity<String> deleteRole(@PathVariable String organizationId, @Valid @RequestBody IdListRequest request) {
        this.rbacCustomRoleManagerProvider.deleteRole(request);
        return this.ok("操作成功");
    }

    @ApiOperation(value = "根据角色id列表、职员id列表，批量设置用户角色关联", response = String.class)
    @PostMapping("/add/roles/employees")
    public ResponseEntity<String> addRolesAndUsers(@PathVariable String organizationId, @Valid @RequestBody RbacCustomRoleIdListEmployeeIdListRequest request) {
        this.rbacCustomRoleManagerProvider.addUsersAndRoles(request);
        return this.ok("操作成功");
    }

//    @ApiOperation(value = "根据角色id、职员id列表，批量设置用户关联", response = String.class)
//    @PostMapping("/add/role/employees")
//    public ResponseEntity<String> addRoleAndUsers(@PathVariable String organizationId, @Valid @RequestBody RbacCustomRoleIdEmployeeIdListRequest request) {
//        this.rbacCustomRoleManagerProvider.addRoleAndUsers(request);
//        return this.ok("操作成功");
//    }
//
//    @ApiOperation(value = "根据职员id、角色id列表，批量设置角色关联", response = String.class)
//    @PostMapping("/add/employee/roles")
//    public ResponseEntity<String> addUserAndRoles(@PathVariable String organizationId, @Valid @RequestBody RbacCustomEmployeeIdRoleIdListRequest request) {
//        this.rbacCustomRoleManagerProvider.addUserAndRoles(request);
//        return this.ok("操作成功");
//    }

    @ApiOperation(value = "根据角色id、职员id列表，批量删除用户关联", response = String.class)
    @PostMapping("/delete/role/employees")
    public ResponseEntity<String> deleteRoleAndUsers(@PathVariable String organizationId, @Valid @RequestBody RbacCustomRoleIdEmployeeIdListRequest request) {
        this.rbacCustomRoleManagerProvider.deleteRoleAndUsers(request);
        return this.ok("操作成功");
    }

    @ApiOperation(value = "根据职员id、角色id列表，批量删除用户关联", response = String.class)
    @PostMapping("/delete/employee/roles")
    public ResponseEntity<String> deleteUserAndRoles(@PathVariable String organizationId, @Valid @RequestBody RbacCustomEmployeeIdRoleIdListRequest request) {
        this.rbacCustomRoleManagerProvider.deleteUserAndRoles(request);
        return this.ok("操作成功");
    }

    @ApiOperation(value = "下载批量导入成员模板")
    @PostMapping("/download/import/member/template")
    public void downloadImportMemberTemplate(@PathVariable String organizationId, HttpServletResponse response)  {
        try {
            String filePath = "templates/import_role_member_template.xlsx";
            response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
            FileUtils.setAttachmentResponseHeader(response, "批量导入成员模板.xlsx");
            Map<Integer, List<String>> dropDownMap = new HashMap<>();
            InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(filePath);
            EasyExcel.write(response.getOutputStream())
                    .withTemplate(inputStream).
                    registerWriteHandler(new TitleHandler(dropDownMap)).
                    sheet().
                    doWrite(ListUtil.empty());
        }  catch (IOException e) {
            log.error("下载批量导入成员模板异常，{}", e.getMessage(), e);
        }
    }


    @ApiOperation(value = "批量导入成员", response = CommonImportResponse.class)
    @PostMapping("/{roleId}/import/member")
    public void importMember(@PathVariable String organizationId, @PathVariable Long roleId,
                                       @RequestBody BatchImportDataRequest request,
                                       HttpServletRequest servletRequest, HttpServletResponse servletResponse) {
        try {
            Collection<BatchImportRoleMemberRequest> batchImportRoleMemberRequests =
                    extractDataComponent.extract(2, request.getPath(), this::translate2BatchImportRoleMemberRequests);

            List<Map<String, String>>  resultList = this.rbacCustomRoleManagerProvider.importMember(organizationId, roleId, batchImportRoleMemberRequests);

            servletResponse.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
            FileUtils.setAttachmentResponseHeader(servletResponse, "批量导入成员结果.xlsx");

            InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream("templates/批量导入角色成员.xlsx");
            ExcelWriter excelWriter = EasyExcel.write(servletResponse.getOutputStream())
                    .withTemplate(inputStream).build();
            FillConfig fillConfig = FillConfig.builder().forceNewRow(false).build();
            excelWriter.fill(resultList, fillConfig, EasyExcel.writerSheet().build());
            excelWriter.finish();
            servletResponse.flushBuffer();
        } catch (IOException e) {
            log.error("生成批量导入成员结果文件异常， {}", e.getMessage(), e);
        }
    }

    /**
     * description：批量导入角色成员-转换对象
     * author：linchunpeng
     * date：2024/2/6
     */
    private Collection<BatchImportRoleMemberRequest> translate2BatchImportRoleMemberRequests(Collection<Map<Integer, String>> excelMapValues) {
        if (CollectionUtils.isEmpty(excelMapValues)) {
            return Collections.emptyList();
        }
        Collection<BatchImportRoleMemberRequest> requests = excelMapValues.parallelStream().map(ii -> {
            String name = StringUtils.trimWhitespace(ii.get(0));
            String telephone = StringUtils.trimWhitespace(ii.get(1));
            return BatchImportRoleMemberRequest.create(name, telephone);
        }).collect(Collectors.toList());
        return requests;
    }
}
