package com.bcxin.rest.web.apis.controllers;

import cn.hutool.core.collection.ListUtil;
import com.alibaba.excel.EasyExcel;
import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.components.CacheV2Provider;
import com.bcxin.Infrastructures.components.ExcelProvider;
import com.bcxin.Infrastructures.components.RetryProvider;
import com.bcxin.Infrastructures.components.WebFileProvider;
import com.bcxin.Infrastructures.enums.EmploymentStatus;
import com.bcxin.Infrastructures.exceptions.DataOperationTenantException;
import com.bcxin.Infrastructures.exceptions.ForbidTenantException;
import com.bcxin.Infrastructures.utils.AuthUtil;
import com.bcxin.Infrastructures.utils.URLUtil;
import com.bcxin.api.interfaces.commons.CommonImportResponse;
import com.bcxin.api.interfaces.commons.responses.CompanyResponse;
import com.bcxin.api.interfaces.identities.IdentityRpcProvider;
import com.bcxin.api.interfaces.identities.requests.GetCredentialRequest;
import com.bcxin.api.interfaces.tenants.*;
import com.bcxin.api.interfaces.tenants.criterias.EmployeeCriteria;
import com.bcxin.api.interfaces.tenants.criterias.EmployeeLeaveCriteria;
import com.bcxin.api.interfaces.tenants.requests.employees.*;
import com.bcxin.api.interfaces.tenants.requests.inviteAttendSite.InviteAttendSiteRequest;
import com.bcxin.api.interfaces.tenants.requests.inviteDepart.InviteDepartRequest;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.UpdateCredentialRequest;
import com.bcxin.api.interfaces.tenants.responses.*;
import com.bcxin.infrastructure.offices.utils.TitleHandler;
import com.bcxin.rest.web.apis.ExtractDataComponent;
import com.bcxin.rest.web.apis.requests.BatchImportDataRequest;
import com.bcxin.rest.web.apis.utils.CacheUtil;
import com.bcxin.rest.web.apis.utils.FileUtils;
import io.swagger.annotations.*;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/tenant/organizations/{organizationId}/employees")
public class EmployeeController extends ControllerAbstract {
    private final EmployeeRpcProvider employeeRpcProvide;
    private final WebFileProvider webFileProvider;
    private final ExcelProvider excelProvider;
    private final ExtractDataComponent extractDataComponent;
    private final OrganizationRpcProvider organizationRpcProvider;

    private final InviteDepartRpcProvider inviteDepartRpcProvider;



    private final InviteAttendSiteRpcProvider inviteAttendSiteRpcProvider;
    private final CacheV2Provider cacheV2Provider;
    private final IdentityRpcProvider identityRpcProvider;
    private final UserRpcProvider userRpcProvider;
    private final RetryProvider retryProvider;

    private final boolean updateNicknameHeadPhoto;
    private final String envRegionCode;

    public EmployeeController(EmployeeRpcProvider employeeRpcProvide,
                              WebFileProvider webFileProvider,
                              ExcelProvider excelProvider,
                              ExtractDataComponent extractDataComponent,
                              InviteDepartRpcProvider inviteDepartRpcProvider,
                              InviteAttendSiteRpcProvider inviteAttendSiteRpcProvider,
                              CacheV2Provider cacheV2Provider,
                              IdentityRpcProvider identityRpcProvider,
                              UserRpcProvider userRpcProvider,
                              OrganizationRpcProvider organizationRpcProvider,
                              RetryProvider retryProvider,
                              @Value("${tenant-user-config.update-nickname-headphoto:false}") Boolean updateNicknameHeadPhoto,
                              @Value("${env.region-code:110000}") String envRegionCode) {
        this.employeeRpcProvide = employeeRpcProvide;
        this.webFileProvider = webFileProvider;
        this.excelProvider = excelProvider;
        this.extractDataComponent = extractDataComponent;
        this.inviteDepartRpcProvider = inviteDepartRpcProvider;
        this.inviteAttendSiteRpcProvider = inviteAttendSiteRpcProvider;
        this.cacheV2Provider = cacheV2Provider;
        this.identityRpcProvider = identityRpcProvider;
        this.userRpcProvider = userRpcProvider;
        this.organizationRpcProvider = organizationRpcProvider;
        this.retryProvider = retryProvider;
        this.updateNicknameHeadPhoto = updateNicknameHeadPhoto;
        this.envRegionCode = envRegionCode;
    }
    @ApiOperation(value = "劳动合同下载", response = Void.class)
    @RequestMapping("/contractDownloadLocal/{id}")
    public ResponseEntity contractDownloadLocal(@PathVariable String organizationId, @PathVariable String id,HttpServletResponse response) throws IOException {
           //获取合同文件路径
        String path ="";
        //下载合同文件
        FileUtils.downloadLocal(path,response);
        return this.ok();
    }
    @ApiOperation(value = "转移职员到其他部门", response = Void.class)
    @PostMapping("/department-movement")
    public ResponseEntity post_department_movement(@PathVariable String organizationId,
                                                   @RequestBody BatchMoveEmployeesRequest request) {
        this.employeeRpcProvide.batchMove(organizationId, request);

        return this.ok();
    }

    @ApiOperation(value = "新增企业职员信息", response = String.class)
    @PostMapping
    public ResponseEntity post_employees(@PathVariable String organizationId,
                                         @Valid @RequestBody EmployeeRequest request) {
        request.setOperator(AuthUtil.getCurrentOperator());
        request.setAreaCode(AuthUtil.getAreaCode());
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());

        String employeeId = this.employeeRpcProvide.create(organizationId, request);
        return this.ok(employeeId);
    }

    @ApiOperation(value = "导出员工导入模板")
    @RequestMapping("/exportEmployeeTemp")
    public void exportEmployeeTemp(@PathVariable String organizationId,HttpServletResponse response) throws Exception {
        List<String> departs = inviteDepartRpcProvider.getDeparts(organizationId);
        CompanyResponse companyResponse = organizationRpcProvider.get(organizationId);
        Map<Integer, List<String>> dropDownMap = new HashMap<>();
        dropDownMap.put(2, departs);
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        FileUtils.setAttachmentResponseHeader(response, "员工导入模板.xlsx");

        String fileName = "templates/V5_employee_temp_2023.xlsx";
        if (companyResponse != null) {
            if (companyResponse.getInstitutionalCode().endsWith("07")
                    || companyResponse.getInstitutionalCode().endsWith("08")
                    || companyResponse.getInstitutionalCode().endsWith("09")) {
                //监管组织
                fileName = "templates/V5_jg_employee_temp_2023.xlsx";
            } else if ("02,04,06,07".contains(companyResponse.getIndustryCode())) {
                //群防群治行业、内保行业、行业主管部门
                String code = companyResponse.getIndustryCode();
                if (companyResponse.getIndustryCode().equals("07")) {
                    //行业主管部门与内保行业公用一个模板
                    code = "04";
                }
                fileName = "templates/V5_"+code+"_employee_temp_2023.xlsx";
            }
        }

        try (InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(fileName)) {
            EasyExcel.write(response.getOutputStream())
                    .withTemplate(inputStream).
                    registerWriteHandler(new TitleHandler(dropDownMap)).
                    sheet().
                    doWrite(ListUtil.empty());
        }
    }

    @ApiOperation(value = "批量导入保安员", response = CommonImportResponse.class)
    @PostMapping("/batch-import")
    public ResponseEntity bathImport(
            HttpServletRequest servletRequest,
            @PathVariable String organizationId,
            @RequestBody BatchImportDataRequest request) {
        CompanyResponse companyResponse = organizationRpcProvider.get(organizationId);
        Collection<BatchImportEmployeeRequest> batchImportEmployeeRequests =
                extractDataComponent.extract(2, request.getPath(), excelMapValues -> {
                    return translate2BatchImportEmployeeRequests(excelMapValues, companyResponse.getIndustryCode(), companyResponse.getInstitutionalCode());
                });

        CommonImportResponse response =
                this.employeeRpcProvide
                        .batchImport(organizationId, companyResponse.getInstitutionalCode(), request.getPath(), batchImportEmployeeRequests);

        response.buildDownloadUrl(
                String.format("%s/download/resources/%s/employee", servletRequest.getContextPath(), response.getResourceId())
        );

        CacheUtil.clearDepartments(cacheV2Provider, organizationId, false);
        return this.ok(response);
    }

    @ApiOperation(value = "批量离职", response = Void.class)
    @PostMapping("/batch-leave")
    public ResponseEntity batchLeave(@PathVariable String organizationId,
                                     @RequestBody BatchLeaveEmployeeRequest request) {
        this.employeeRpcProvide.batchLeave(organizationId, request);
        CacheUtil.clearDepartments(cacheV2Provider, organizationId, false);
        return this.ok();
    }

    @ApiOperation(value = "excel批量离职导入", response = Void.class)
    @PostMapping("/batch-leave-excel")
    public ResponseEntity batchLeaveExcel(HttpServletRequest servletRequest,
                                          @PathVariable String organizationId,
                                          @RequestBody BatchImportDataRequest request) {
        Collection<BatchLeaveEmployeeExcelRequest> batchLeaveEmployeeExcelRequests =
                extractDataComponent.extract(2, request.getPath(), this::translate2BatchImportEmployeeExcelRequests);

        CommonImportResponse response =
                employeeRpcProvide.batchLeaveExcel(organizationId, request.getPath(), batchLeaveEmployeeExcelRequests);

        response.buildDownloadUrl(
                String.format("%s/download/resources/%s/employee_leave", servletRequest.getContextPath(), response.getResourceId())
        );

        CacheUtil.clearDepartments(cacheV2Provider, organizationId, false);
        return this.ok(response);
    }

    @ApiOperation(value = "复职", response = Void.class)
    @PostMapping("/go-back")
    public ResponseEntity goBack(@PathVariable String organizationId,
                                 @RequestBody BackEmployeeRequest request) {
        this.employeeRpcProvide.back(organizationId, request);
        CacheUtil.clearDepartments(cacheV2Provider, organizationId, false);

        return this.ok();
    }

    @ApiOperation(value = "批量设置上级", response = Void.class)
    @PostMapping("/batch-assign-superior")
    public ResponseEntity batchAssignSuperior(@PathVariable String organizationId,
                                              @RequestBody BatchAssignEmployeeSuperiorRequest request) {
        this.employeeRpcProvide.batchAssignSuperiors(organizationId, request);

        return this.ok();
    }


    @ApiOperation(value = "获取已离职的职员列表")
    @PostMapping("/search-leave")
    public ResponseEntity get_leave_employees(@PathVariable String organizationId,
                                              @RequestBody EmployeeLeaveCriteria criteria) {
        Pageable<EmployeeLeaveGetResponse> data = this.employeeRpcProvide.findLeaves(organizationId, criteria);

        return this.ok(data);
    }


    @ApiOperation(value = "导出已离职的职员列表")
    @RequestMapping("/exportLeaveExcel")
    public void exportLeaveExcel(@PathVariable String organizationId,@RequestBody EmployeeLeaveCriteria criteria, HttpServletResponse response) throws Exception {
        CompanyResponse companyResponse = organizationRpcProvider.get(organizationId);
        String filePath = this.employeeRpcProvide.createLeaveExcel(organizationId,criteria,companyResponse!=null?companyResponse.getIndustryCode():null);
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        FileUtils.setAttachmentResponseHeader(response, "离职职员.xlsx");
        FileUtils.writeBytes(filePath, response.getOutputStream(),retryProvider);
        FileUtils.deleteFile(filePath);
    }

    @ApiOperation(value = "导出离职证明")
    @RequestMapping("/exportLeaveWord/{id}")
    public void exportLeaveWord(@PathVariable String organizationId,@PathVariable String id,HttpServletResponse response) throws Exception {
        String filePath = this.employeeRpcProvide.createLeaveWord(organizationId,id);
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        FileUtils.setAttachmentResponseHeader(response, "离职证明.doc");
        FileUtils.writeBytes(filePath, response.getOutputStream(),retryProvider);
        FileUtils.deleteFile(filePath);
    }


    @ApiOperation(value = "获取企业职员列表", response = EmployeeGetResponse.class)
    @PostMapping("/search")
    public ResponseEntity get_employees(@PathVariable String organizationId, @Valid @RequestBody EmployeeCriteria criteria) {
        Pageable<EmployeeGetResponse> data = this.employeeRpcProvide.find(organizationId, criteria);

        return this.ok(data);
    }

    @ApiOperation(value = "获取企业职员列表统计")
    @PostMapping("/searchReport")
    public ResponseEntity get_employeesReport(@PathVariable String organizationId, @Valid @RequestBody EmployeeCriteria criteria) {
        EmployeeReportResponse data = this.employeeRpcProvide.findReport(organizationId, criteria);
        return this.ok(data);
    }


    @ApiOperation(value = "管理员获取企业职员列表", response = EmployeeGetResponse.class)
    @PostMapping("/manage/search")
    public ResponseEntity get_manageEmployees(@PathVariable String organizationId, @Valid @RequestBody EmployeeCriteria criteria) {
        Pageable<EmployeeGetResponse> data = this.employeeRpcProvide.find(null, criteria);

        return this.ok(data);
    }

    @ApiOperation(value = "获取企业职员导出字段", response = ColumnResponse.class)
    @PostMapping("/getExportColumns")
    public ResponseEntity getExportColumns() {
        List<ColumnResponse> data = this.employeeRpcProvide.getExportColumns();
        return this.ok(data);
    }

    @ApiOperation(value = "导出企业职员列表")
    @RequestMapping("/exportExcel")
    public void exportExcel(@PathVariable String organizationId,@RequestBody EmployeeCriteria criteria, HttpServletResponse response) throws Exception {
        String filePath = this.employeeRpcProvide.createExcel(organizationId,criteria);
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        FileUtils.setAttachmentResponseHeader(response, "企业职员信息.xlsx");
        FileUtils.writeBytes(filePath, response.getOutputStream(),retryProvider);
        FileUtils.deleteFile(filePath);
    }


    @ApiOperation(value = "获取职员/用户的基本信息", response = EmployeeDetailGetResponse.class)
    @GetMapping("/{id}")
    public ResponseEntity<EmployeeDetailGetResponse> get(@PathVariable String organizationId, @PathVariable String id) {
        EmployeeDetailGetResponse detailGetResponse = this.employeeRpcProvide.getByOrganIdAndId(organizationId, id);
        detailGetResponse.getUserInfo().setUpdateNicknameHeadPhoto(updateNicknameHeadPhoto);
        detailGetResponse.getUserInfo().setEnvRegionCode(envRegionCode);
        return this.ok(detailGetResponse);
    }

    @ApiOperation(value = "修改职员信息", response = Void.class)
    @PutMapping("/{id}")
    public ResponseEntity put(@PathVariable String organizationId, @PathVariable String id, @RequestBody UpdateEmployeeRequest request) {
        this.employeeRpcProvide.update(organizationId, id, request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }


    @ApiOperation(value = "新增邀请入职部门", response = Valid.class)
    @PostMapping("add-invite-depart")
    public ResponseEntity addInviteDepart(@PathVariable String organizationId,
                               @Valid @RequestBody InviteDepartRequest request) {
        String inviteDepartId = this.inviteDepartRpcProvider.create(organizationId, request);

        CacheUtil.clearDepartments(cacheV2Provider, organizationId, false);
        return this.ok(inviteDepartId);
    }


    @ApiOperation(value = "新增邀请入职驻勤点", response = Valid.class)
    @PostMapping("add-invite-attend-site")
    public ResponseEntity addInviteAttendSite(@PathVariable String organizationId,
                                              @Valid @RequestBody InviteAttendSiteRequest request) {
        String inviteDepartId = this.inviteAttendSiteRpcProvider.create(organizationId, request);

        CacheUtil.clearDepartments(cacheV2Provider, organizationId, false);
        return this.ok(inviteDepartId);
    }


    @ApiOperation(value = "重置密码为后六位")
    @PostMapping("/{employeeId}/resetPassword")
    public ResponseEntity resetPassword(@PathVariable String organizationId,@PathVariable String employeeId) {

        /**
         * 当前操作者, 可以为当前用户/也可以是固定后台用户
         */
        TenantUserContext.UserModel currentUser = TenantContext.getInstance().getUserContext().get();
        if (currentUser == null) {
            throw new ForbidTenantException("必须有人来重置密码");
        }
        boolean authFlag = false;
        Collection<MyOrganizationProfileGetResponse> response = this.cacheV2Provider.get(CacheUtil.getCurrentOrganizationsCacheKey(), () -> {
            return this.userRpcProvider.getById(this.getCurrentUserId());
        }, 5 * 60 * 24);
        for (MyOrganizationProfileGetResponse myOrganizationProfileGetResponse : response) {
            if(myOrganizationProfileGetResponse.getEmployeeId().equals(this.getSelectedEmployeeId())){
                if(myOrganizationProfileGetResponse.isDepartAdmin() || myOrganizationProfileGetResponse.isDomainAdmin()){
                    authFlag = true;
                }
                break;
            }
        }
        if(!authFlag){
            throw new ForbidTenantException("组织管理员或者部门管理员才能进行此操作");
        }

        EmployeeBasicResponse employee = employeeRpcProvide.getEmployeeBasic(organizationId, employeeId);
        ResetPassWordRequest request = new ResetPassWordRequest();
        request.setLoginName(employee.getIdNum());
        request.setNewPassword("");
        //存在用户并且不是离职状态
        if (employee != null && employee.getStatus() != EmploymentStatus.OffJob){
            identityRpcProvider.resetPassword(request);
        }

        return this.ok();

    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("获取证书信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class),
    })
    @PostMapping("/{employeeId}/credentials")
    public ResponseEntity<Pageable<CredentialResponse>> getCredential(@PathVariable  String organizationId,@PathVariable String employeeId,@RequestBody GetCredentialRequest request) {
        Pageable<CredentialResponse> credentials = this.userRpcProvider.getCredentials(employeeId,organizationId, request);
        return this.ok(credentials);
    }
    @ApiOperation("获取心理测评记录信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class),
    })
    @PostMapping("/{employeeId}/xlcps")
    public ResponseEntity<List<XlcpReportResponse>> xlcps(@PathVariable  String organizationId,@PathVariable String employeeId) {
        List<XlcpReportResponse> credentials = this.userRpcProvider.xlcps(employeeId,organizationId);
        return this.ok(credentials);
    }
    @ApiOperation("心理测评记录是否在期限内")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class),
    })
    @PostMapping("/{employeeId}/xlcp-valid-or-not")
    public ResponseEntity<Boolean> xlcpValidOrNot(@PathVariable  String organizationId,@PathVariable String employeeId) {
        return this.ok(this.userRpcProvider.xlcpValidOrNot(organizationId,employeeId,this.getCurrentUserId()));
    }


    @ApiOperation(value = "变更用户证件信息", response = Void.class)
    @PostMapping("/{id}/edit-credential")
    public ResponseEntity put_credential(@PathVariable String id, @RequestBody UpdateCredentialRequest request) {
        this.userRpcProvider.updateCredential(id, request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }

    @ApiOperation(value = "获取公司管理员",responseContainer = "List", response = OrganizationAdminResponse.class)
    @GetMapping("/admins")
    public ResponseEntity<Collection<OrganizationAdminResponse>> getOrganizationAdmin(@PathVariable String organizationId) {
        return this.ok(employeeRpcProvide.getOrganizationAdministrators(organizationId));
    }

    /**
     * 转换为对象
     * 先跳过前两行
     *
     * @param excelMapValues
     * @return
     */
    private Collection<BatchImportEmployeeRequest> translate2BatchImportEmployeeRequests(
            Collection<Map<Integer, String>> excelMapValues, String industryCode, String institutionalCode) {
        if (CollectionUtils.isEmpty(excelMapValues)) {
            return Collections.emptyList();
        }

        Collection<BatchImportEmployeeRequest> requests = excelMapValues.parallelStream().map(ii -> {
            String name = StringUtils.trimWhitespace(ii.get(0));
            String telephone = StringUtils.trimWhitespace(ii.get(1));
            String departNames = StringUtils.trimWhitespace(ii.get(2));
            String occupationType = StringUtils.trimWhitespace(ii.get(3));
            if (!institutionalCode.endsWith("07")
                    && !institutionalCode.endsWith("08")
                    && !institutionalCode.endsWith("09")
                    &&"04,06,07".contains(industryCode)) {
                //群防群治行业、内保行业、行业主管部门，不是监管组织
                String position = StringUtils.trimWhitespace(ii.get(4));
                String hiredDate = StringUtils.trimWhitespace(ii.get(5));
                String credentialType = StringUtils.trimWhitespace(ii.get(6));
                String credentialNumber = StringUtils.trimWhitespace(ii.get(7));
                String nation = StringUtils.trimWhitespace(ii.get(8));
                String education = StringUtils.trimWhitespace(ii.get(9));
                String politicsStatus = StringUtils.trimWhitespace(ii.get(10));
                String militaryStatus = StringUtils.trimWhitespace(ii.get(11));
                String maritalStatus = StringUtils.trimWhitespace(ii.get(12));
                String emergencyContact = StringUtils.trimWhitespace(ii.get(13));
                String emergencyPhone = StringUtils.trimWhitespace(ii.get(14));
                String address = StringUtils.trimWhitespace(ii.get(15));
                String householdType = StringUtils.trimWhitespace(ii.get(16));
                if (StringUtils.hasLength(name) && (StringUtils.hasLength(telephone) || StringUtils.hasLength(credentialNumber) )) {
                    return BatchImportEmployeeRequest.create(name, telephone, departNames, occupationType, position, hiredDate,
                            credentialType, credentialNumber, nation, education, politicsStatus, militaryStatus,
                            maritalStatus, emergencyContact, emergencyPhone, address, householdType);
                }
            } else {
                //其他行业
                String hiredDate = StringUtils.trimWhitespace(ii.get(4));
                String credentialType = StringUtils.trimWhitespace(ii.get(5));
                String credentialNumber = StringUtils.trimWhitespace(ii.get(6));
                String nation = StringUtils.trimWhitespace(ii.get(7));
                String education = StringUtils.trimWhitespace(ii.get(8));
                String politicsStatus = StringUtils.trimWhitespace(ii.get(9));
                String militaryStatus = StringUtils.trimWhitespace(ii.get(10));
                String maritalStatus = StringUtils.trimWhitespace(ii.get(11));
                String emergencyContact = StringUtils.trimWhitespace(ii.get(12));
                String emergencyPhone = StringUtils.trimWhitespace(ii.get(13));
                String address = StringUtils.trimWhitespace(ii.get(14));
                String householdType = StringUtils.trimWhitespace(ii.get(15));
                if (StringUtils.hasLength(name) && (StringUtils.hasLength(telephone) || StringUtils.hasLength(credentialNumber) )) {
                    return BatchImportEmployeeRequest.create(name, telephone, departNames, occupationType, null, hiredDate,
                            credentialType, credentialNumber, nation, education, politicsStatus, militaryStatus,
                            maritalStatus, emergencyContact, emergencyPhone, address, householdType);
                }
            }
            return null;
        }).filter(ii -> ii != null).collect(Collectors.toList());

        return requests;
    }

    /**
     * 通过excel批了离职
     * 转换为对象
     * 先跳过前两行
     *
     * @param excelMapValues
     * @return
     */
    private Collection<BatchLeaveEmployeeExcelRequest> translate2BatchImportEmployeeExcelRequests(
            Collection<Map<Integer, String>> excelMapValues) {
        if (CollectionUtils.isEmpty(excelMapValues)) {
            return Collections.emptyList();
        }

        Collection<BatchLeaveEmployeeExcelRequest> requests = excelMapValues.parallelStream().map(ii -> {
            String username = StringUtils.trimWhitespace(ii.get(0));
            String idNum = StringUtils.trimWhitespace(ii.get(1));
            String leaveData = ii.get(2);
            String remark = StringUtils.trimWhitespace(ii.get(3));

            if (StringUtils.hasLength(username) && StringUtils.hasLength(idNum)&& StringUtils.hasLength(leaveData)) {
                BatchLeaveEmployeeExcelRequest batchLeaveEmployeeExcelRequest = new BatchLeaveEmployeeExcelRequest();

                batchLeaveEmployeeExcelRequest.setLeaveDateText(leaveData);
                batchLeaveEmployeeExcelRequest.setUsername(username);
                batchLeaveEmployeeExcelRequest.setIdNum(idNum);
                batchLeaveEmployeeExcelRequest.setRemark(remark);

                return batchLeaveEmployeeExcelRequest;
            }

            return null;
        }).filter(ii -> ii != null).collect(Collectors.toList());

        return requests;
    }
}
