import loadjs from "./loadjs/loadjs";

export default function useExtendPrintApi(formCreate) {
    formCreate.extendApi((api, h, a) => {
        function disableImageSmoothing(ctx) {
            ctx.imageSmoothingEnabled = false;
            ctx.mozImageSmoothingEnabled = false;
            ctx.webkitImageSmoothingEnabled = false;
            ctx.msImageSmoothingEnabled = false;
        }

        function print(options) {
            const config = {
                left: 20,
                right: 20,
                top: 20,
                bottom: 20,
                width: 780,
                format: 'a4',
                ...options || {},
            }
            return new Promise(resolve => {
                if (!loadjs.isDefined('html2canvas')) {
                    loadjs.loadNpm('html2canvas@1.4.1/dist/html2canvas.min.js', 'html2canvas');
                }
                if (!loadjs.isDefined('jspdf')) {
                    loadjs.loadNpm('jspdf@3.0.1/dist/jspdf.umd.js', 'jspdf');
                }
                loadjs.ready(['html2canvas', 'jspdf'], () => {
                    const el = config.el || api.formEl().$el;
                    el.classList.add('fc-print-form');
                    window.html2canvas(el, {
                        allowTaint: true,
                        useCORS: true,
                    }).then((canvas) => {
                        const pdf = new window.jspdf.jsPDF({
                            orientation: 'p',
                            unit: 'pt',
                            format: (config.format || 'a4').toLowerCase(),
                        });

                        disableImageSmoothing(canvas.getContext('2d'));

                        const {
                            left: marginLeft,
                            right: marginRight,
                            top: marginTop,
                            bottom: marginBottom,
                        } = config;

                        const pageWidth = pdf.internal.pageSize.getWidth();
                        const pageHeight = pdf.internal.pageSize.getHeight();
                        const contentWidth = pageWidth - marginLeft - marginRight;
                        const contentHeight = pageHeight - marginTop - marginBottom;

                        const scaledHeight = (canvas.height * contentWidth) / canvas.width;

                        if (scaledHeight <= contentHeight) {
                            pdf.addImage(
                                canvas.toDataURL('image/jpeg'),
                                'JPEG',
                                marginLeft,
                                marginTop,
                                contentWidth,
                                scaledHeight
                            );
                        } else {
                            let remainingHeight = scaledHeight;
                            let page = 0;
                            const clipHeight = canvas.width * contentHeight / contentWidth;

                            while (remainingHeight > 0) {
                                const tempCanvas = document.createElement('canvas');
                                const tempCtx = tempCanvas.getContext('2d');
                                disableImageSmoothing(tempCtx);

                                const offsetY = page * clipHeight;
                                const actualClipHeight = Math.min(clipHeight, canvas.height - offsetY);

                                tempCanvas.width = canvas.width;
                                tempCanvas.height = actualClipHeight;

                                tempCtx.drawImage(
                                    canvas,
                                    0, offsetY, canvas.width, actualClipHeight,
                                    0, 0, canvas.width, actualClipHeight
                                );

                                const imageHeight = (actualClipHeight / canvas.height) * scaledHeight;

                                pdf.addImage(
                                    tempCanvas.toDataURL('image/jpeg'),
                                    'JPEG',
                                    marginLeft,
                                    marginTop,
                                    contentWidth,
                                    imageHeight
                                );

                                remainingHeight -= contentHeight;
                                if (remainingHeight > 0) {
                                    pdf.addPage();
                                    page++;
                                }
                            }
                        }
                        return resolve(pdf);
                    }).finally(() => {
                        el.classList.remove('fc-print-form');
                    });
                });
            });
        }

        return {
            exportPdf(config) {
                print(config).then(pdf => {
                    window.open(URL.createObjectURL(pdf.output('blob')));
                })
            },
            print(config) {
                print(config).then(pdf => {
                    const frame = (document.createElement('iframe'));
                    frame.style.width = '0';
                    frame.style.position = 'absolute';
                    frame.style.height = '0';
                    frame.style.border = 'none';
                    frame.onload = function () {
                        setTimeout(() => {
                            frame.contentWindow.print();
                        }, 100);
                    }
                    document.body.appendChild(frame);
                    frame.src = URL.createObjectURL(pdf.output('blob'));
                })
            }
        }
    });
}