package com.bcxin.runtime.apis.controllers;

import cn.hutool.core.io.FileUtil;
import cn.myapps.authtime.common.dao.PersistenceUtils;
import cn.myapps.common.Environment;
import cn.myapps.common.model.application.Application;
import cn.myapps.common.model.datasource.DataSource;
import cn.myapps.common.util.PropertiesConfig;
import cn.myapps.designtime.common.cache.DesignTimeSerializableCache;
import cn.myapps.util.file.ZipUtil;
import com.bcxin.runtime.apis.components.MappingSqlValueTranslator;
import com.bcxin.runtime.apis.configs.RegionConfig;
import com.bcxin.runtime.apis.dtos.*;
import com.bcxin.runtime.apis.exceptions.ChangeLogBadException;
import com.bcxin.runtime.apis.requests.DownChangelogRequest;
import com.bcxin.runtime.apis.responses.ChangeLogResponse;
import com.bcxin.runtime.apis.responses.DownloadChangelogResponse;
import com.bcxin.runtime.domain.constants.FieldNames;
import com.bcxin.saas.core.components.JsonProvider;
import com.bcxin.saas.core.exceptions.SaasBadException;
import com.bcxin.saas.core.exceptions.SaasForbidException;
import com.bcxin.saas.core.exceptions.SaasNofoundException;
import com.bcxin.saas.core.utils.ExceptionUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.sql.ResultSetMetaData;
import java.sql.Timestamp;
import java.sql.Types;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@RestController
@RequestMapping("/v3/extends/change-logs")
public class ChangelogController extends SecurityControllerAbstract {
    private final JdbcTemplate jdbcTemplate;
    private final MappingSqlValueTranslator mappingSqlValueTranslator;
    private final PropertiesConfig propertiesConfig;
    private final RegionConfig regionConfig;
    private final JsonProvider jsonProvider;
    private final String INSERT_CHANGE_LOG = "INSERT INTO sync_change_logs(id,name,path,total,current,params)values(?,?,?,?,?,?)";
    private final String UPDATE_CHANGE_LOG = "UPDATE sync_change_logs SET current=current+?,result=?,last_updated_time=? WHERE ID=?";

    private final String DELETE_CHANGE_LOG = "delete from sync_change_logs where id=?";
    private final String GET_CHANGE_LOG = "select id,name,path,createdTime,total,current,result,last_updated_time from sync_change_logs where id=?";
    private final String GET_ALL_CHANGE_LOG = "select id,name,path,createdTime,total,current,result from sync_change_logs order by createdTime desc limit 30 ";
    private final String LOG_OUTPUT_ZIP_DIRECTORY="/uploads/export_cdc/%s";

    /**
     * 只获取通过摆渡服务同步的数据
     */
    private final String SYNC_CHANGE_LOG_SQL_TEMPLATE="" +
            "select app.app_id,f.name,sf.filter, f.table_name,sf.is_online,sf.config,st.config as target_config,st.url\n" +
            "from meta_apps app join meta_forms f on app.id=f.application_meta_id\n" +
            "join sync_meta_forms sf on f.id=sf.form_meta_id\n" +
            "join sync_meta_form_targets st on st.id=sf.target_meta_id\n" +
            "where sf.is_online=1 and st.url like '%v2/ftp%' ";
    private final String SYNC_DYNAMIC_DATA_MAP_TEMPLATES="SELECT template,mapkey,region FROM dynamic_data_map_templates";

    public ChangelogController(JdbcTemplate jdbcTemplate,
                               MappingSqlValueTranslator mappingSqlValueTranslator,
                               PropertiesConfig propertiesConfig,
                               RegionConfig regionConfig, JsonProvider jsonProvider) {
        this.jdbcTemplate = jdbcTemplate;
        this.mappingSqlValueTranslator = mappingSqlValueTranslator;
        this.propertiesConfig = propertiesConfig;
        this.regionConfig = regionConfig;
        this.jsonProvider = jsonProvider;
    }

    /**
     * 下载增量数据包和文件包
     * @param request
     * @return
     * @throws Exception
     */
    @PostMapping
    @Async
    public CompletableFuture<ResponseEntity<DownloadChangelogResponse>> post(
            @RequestBody DownChangelogRequest request, HttpServletRequest servletRequest) throws Exception {
        Map<String, String> detailContainer = new ConcurrentHashMap<>();
        try {
            request.validate(servletRequest);
            RegionConfig.RegionConfigItem regionConfigItem = this.regionConfig.getSelectedRegionConfigItem(request.getRegionCode());
            if (regionConfigItem == null) {
                throw new SaasForbidException(String.format("找不到%s对应的区域信息", request.getRegionCode()));
            }

            SimpleDateFormat yyMMddDateFormat = new SimpleDateFormat("yyyyMMdd");
            String rootDirName = String.format("cdc_%s_%s", request.getRegionCode(), yyMMddDateFormat.format(request.getFromTime()));
            Long requestId = Long.parseLong((new SimpleDateFormat("yyyyMMddhhmmssSSS").format(new Date())));
            String relativePath =
                    String.format(LOG_OUTPUT_ZIP_DIRECTORY, requestId).concat(File.separator)
                            .concat(rootDirName);
            String rootPath =
                    propertiesConfig.getStorageroot()
                            .concat(File.separator)
                            .concat(relativePath);

            Collection<SyncChangeLogTableDto> tables = getChangelogTables(request).stream().filter(ii -> {
                if (CollectionUtils.isEmpty(request.getTables())) {
                    return true;
                }

                return request.matched(ii);
            }).collect(Collectors.toList());

            String selectedNames = "所有";
            if (!CollectionUtils.isEmpty(request.getTables())) {
                selectedNames = tables.stream().map(ii -> ii.getName()).limit(10).collect(Collectors.joining("-"));
                if (tables.size() > 10) {
                    selectedNames += "...";
                }
            }

            try {
                Object[] param = new Object[6];
                param[0] = requestId;
                param[1] = String.format("{rn:\"%s\",ft:\"%s\",et:\"%s\",sn:\"%s\"}",
                        request.getRegionName(),
                        yyMMddDateFormat.format(request.getFromTime()),
                        yyMMddDateFormat.format(request.getEndTime()),
                        selectedNames);
                param[2] = relativePath;
                param[3] = tables.size();
                param[4] = 0;
                param[5] = this.jsonProvider.getJson(request);

                this.jdbcTemplate.update(INSERT_CHANGE_LOG, param);
            } catch (Exception ex) {
                ex.printStackTrace();
            }

            CountDownLatch countDownLatch = new CountDownLatch(1);
            Executors.newSingleThreadExecutor().execute(() -> {
                try {
                    this.executeCdcCapture(regionConfigItem, requestId, tables, request, servletRequest, rootPath, rootDirName, yyMMddDateFormat);
                } catch (Exception e) {
                    e.printStackTrace();

                    throw new SaasBadException("执行差异化日志异常", e);
                } finally {
                    countDownLatch.countDown();
                }
            });

            countDownLatch.await(30, TimeUnit.SECONDS);

            DownloadChangelogResponse downloadChangelogResponse = DownloadChangelogResponse.create(
                    String.format("%s/%s", this.propertiesConfig.getStorageroot(), relativePath),
                    relativePath,
                    detailContainer,
                    null,
                    "系统只等待30秒, 其他等待系统执行结果;几分钟后进行下载");

            return CompletableFuture.completedFuture(ResponseEntity.ok(downloadChangelogResponse));
        } catch (Exception ex) {
            return CompletableFuture.completedFuture(ResponseEntity.badRequest().body(DownloadChangelogResponse.create(
                    "异常",
                    "异常",
                    detailContainer,
                    Collections.singleton(ex),
                    ex.toString())));
        }
    }

    @GetMapping("/download/{zipId}")
    public void download(@PathVariable("zipId") String zipId, HttpServletResponse response) throws Exception {
        String realPath = propertiesConfig.getStorageroot().concat(String.format(LOG_OUTPUT_ZIP_DIRECTORY, zipId));
        File dir = new File(realPath);
        if (!dir.exists()) {
            response.setStatus(HttpStatus.NOT_FOUND.value());
            return;
        }

        //如果同步的数据拉取未完成，不允许下载
        if(!isChangeLogsFinished(zipId)){
            response.setStatus(HttpStatus.INSUFFICIENT_STORAGE.value());
            return;
        }

        Collection<String> allFilePathes =
                Arrays.stream(dir.listFiles()).map(ii -> ii.getAbsolutePath()).collect(Collectors.toList());
        if (!allFilePathes.stream().anyMatch(ii -> ii.endsWith(".zip"))) {
            String[] pathes =
                    allFilePathes.stream().filter(ii -> !ii.endsWith(".zip"))
                            .flatMap(ii -> {
                                File selectedFile = new File(ii);
                                if (selectedFile.isDirectory()) {
                                    return Arrays.stream(selectedFile.listFiles()).map(fi -> fi.getAbsolutePath());
                                } else {
                                    return Stream.of(ii);
                                }
                            })
                            .collect(Collectors.toList()).toArray(new String[]{});
            ZipUtil.compressFiles(zipId, pathes, realPath);
        }

        responseWithFile(zipId, realPath, response);
    }

    @GetMapping
    @Async
    public CompletableFuture<ResponseEntity<Collection<ChangeLogResponse>>> getAll() {
        Collection<ChangeLogResponse> data =
                this.jdbcTemplate.query(GET_ALL_CHANGE_LOG, (rs, rnum) -> {
                    String id = rs.getString("id");
                    String name = rs.getString("name");
                    String path = rs.getString("path");

                    Long total = rs.getLong("total");
                    Long current = rs.getLong("current");
                    String result = rs.getString("result");
                    Timestamp datetime = rs.getTimestamp("createdTime");


                    return ChangeLogResponse.create(id, name, path, total, current, result, datetime);
                });

        return CompletableFuture.completedFuture(ResponseEntity.ok(data));
    }

    @GetMapping("/{id}")
    @Async
    public CompletableFuture<ResponseEntity<ChangeLogResponse>> get(@PathVariable String id) {
        Object[] param = new Object[1];
        param[0] = id;

        ChangeLogResponse logResponse =
                this.jdbcTemplate.queryForObject(GET_CHANGE_LOG, param, (rs, rNum) -> {
                    String rid = rs.getString("id");
                    String name = rs.getString("name");
                    String path = rs.getString("path");

                    Long total = rs.getLong("total");
                    Long current = rs.getLong("current");
                    String result = rs.getString("result");
                    Timestamp datetime = rs.getTimestamp("createdTime");

                    return ChangeLogResponse.create(id, name, path,total,current,result, datetime);
                });

        return CompletableFuture.completedFuture(ResponseEntity.ok(logResponse));
    }

    @DeleteMapping("/{id}")
    @Async
    public CompletableFuture<ResponseEntity> delete(@PathVariable String id) throws ExecutionException, InterruptedException {
        Object[] param = new Object[1];
        param[0] = id;

        CompletableFuture<ResponseEntity<ChangeLogResponse>> completableFuture = get(id);
        ChangeLogResponse body = completableFuture.get().getBody();
        if (body == null) {
            return CompletableFuture.completedFuture(ResponseEntity.notFound().build());
        }

        if (FileUtil.exist(body.getPath())) {
            FileUtil.del(body.getPath());
        }

        this.jdbcTemplate.update(DELETE_CHANGE_LOG, param);

        return CompletableFuture.completedFuture(ResponseEntity.ok("删除成功"));
    }


    @GetMapping("/template/download")
    public void download(HttpServletResponse response) throws Exception {
        Collection<SyncChangeLogTableDto> changeLogTables = getChangelogTables(new DownChangelogRequest());
        Long requestId = (new Date()).getTime();
        String rootPath = String.format("%s/uploads/templates/%s", this.propertiesConfig.getStorageroot(), requestId);

        /**
         * 避免重复的文件列表
         */
        Collection<String> templateSqlPaths = new HashSet<>();
        changeLogTables.forEach(tb -> {
            if (!tb.getTableName().toLowerCase(Locale.ROOT).startsWith("tlk")) {
                System.err.println(String.format("忽略该表的配置信息:%s;", tb.toString()));
                return;
            }

            String templateSQL = String.format("select column_name from information_schema.`COLUMNS` col where table_name='%s'", tb.getTableName());

            Collection<String> columnNames =
                    this.jdbcTemplate.query(templateSQL, (rs, rowNum) -> {
                        return rs.getString("column_name");
                    });

            StringBuilder sb = new StringBuilder(String.format("insert into %s(", tb.getTableName()));
            StringBuilder colsName = new StringBuilder();
            StringBuilder colsValue = new StringBuilder();

            StringBuilder colsValueUpdate = new StringBuilder();
            /**
             * 忽略一个表存在多个库的情况 比如:
             * select * from information_schema.`COLUMNS` col where table_name='tlk_ShareholderInformation'
             */
            for (String col : columnNames.stream().distinct().collect(Collectors.toList())) {
                if (colsName.length() > 0) {
                    colsName.append(",");
                    colsValue.append(",");
                }
                colsName.append(col);
                String colValue = col;

                if ("domainid".equalsIgnoreCase(colValue)) {
                    colValue = "dynamic_domain_id";
                } else if ("item_domain_id".equalsIgnoreCase(colValue)) {
                    colValue = "domainid";
                }

                colsValue.append(String.format("'#{%s}'", colValue));

                String colIgnore = col.toLowerCase(Locale.ROOT);

                if ((!colIgnore.equalsIgnoreCase("id") && colIgnore.startsWith("item_")) ||
                        col.equalsIgnoreCase("LASTMODIFIED")) {
                    if (colsValueUpdate.length() > 0) {
                        colsValueUpdate.append(",");
                    }

                    colsValueUpdate.append(String.format("%s='#{%s}'", col, col));
                }
            }
            sb.append(colsName);
            sb.append(") values(");
            sb.append(colsValue);

            if (StringUtils.hasLength(colsValueUpdate)) {
                sb.append(String.format(") ON DUPLICATE KEY UPDATE %s;", colsValueUpdate));
            } else {
                sb.append(") ");
            }

            String path = String.format("%s/%s.sql", rootPath, tb.getTableName());
            FileUtil.writeUtf8String(sb.toString(), path);

            templateSqlPaths.add(path);
        });

        ZipUtil.compressFiles("增量数据模板", templateSqlPaths.toArray(new String[]{}), rootPath);

        responseWithFile("增量数据模板:" + requestId, rootPath, response);
    }

    @GetMapping("/meta/tables")
    public ResponseEntity<Collection<SyncChangeLogTableDto>> getAllTables() {
        Collection<SyncChangeLogTableDto> tables = getChangelogTables(new DownChangelogRequest());

        return ResponseEntity.ok(tables.stream().map(ii -> {
            String formattedTableName = DownChangelogRequest.getAppTableName(ii.getAppId(), ii.getTableName());
            return SyncChangeLogTableDto.create(ii.getName(), formattedTableName,ii.getFilter(), ii.getAppId(), ii.getConfig(),
                    ii.getTarget_config(), ii.getUrl(), ii.isOnline());
        }).collect(Collectors.toList()));
    }

    @PostMapping("/meta/tables/clear")
    public ResponseEntity clearTables() {
        Collection<ChangeLogTableMapDto> x = _changeLogTableMaps;
        if (_changeLogTableMaps != null) {
            _changeLogTableMaps.clear();
        }

        if (_syncChangeLogTables != null) {
            _syncChangeLogTables.clear();
        }

        return ResponseEntity.ok(x);
    }

    /**
     * 解析changeLog的result字符串，判断数据拉取过程中是否有表格发生错误。
     * @param id 对应的changeLog的id
     * @return 状态码200。同时返回数据获取过程中出现错误的表格名称。
     */
    @GetMapping("/download/error/{id}")
    public ResponseEntity getErrorTables(@PathVariable String id){
        Object[] param = new Object[1];
        param[0] = id;
        Map<String, Object> resultMap = this.jdbcTemplate.queryForMap(GET_CHANGE_LOG, param);
        String result = (String) resultMap.get("result");
        Pattern pattern = Pattern.compile("Error for table=([^\\(]+)");
        Matcher matcher = pattern.matcher(result);
        List<String> tables = new ArrayList<>();
        while(matcher.find()){
            tables.add(matcher.group(1));
        }

        return  ResponseEntity.ok(tables);
    }

    private void executeCdcCapture(RegionConfig.RegionConfigItem regionConfigItem,
                                   Long requestId,
                                   Collection<SyncChangeLogTableDto> tables,
                                   DownChangelogRequest request,
                                   HttpServletRequest servletRequest,
                                   String rootPath,
                                   String rootDirName,
                                   SimpleDateFormat yyMMddDateFormat
    ) throws Exception {
        System.err.println("begin to execute executeCdcCapture");
        Map<String, String> detailContainer = new ConcurrentHashMap<>();

        StopWatch watch = new StopWatch();
        Collection<Exception> changeLogBadExceptions = new ArrayList<>();
        Collection<String> storedPaths = Collections.synchronizedList(new ArrayList<>());
        watch.start("开始抓取数据表的变更需信息");

        CountDownLatch countDownLatch = new CountDownLatch(tables.size());

        AtomicInteger successAtomicCount = new AtomicInteger(0);
        AtomicInteger processedAtomicIndex = new AtomicInteger(0);
        StringBuffer processedMessage = new StringBuffer();
        AtomicInteger processedTotalCount = new AtomicInteger(0);
        AtomicInteger executeIndex = new AtomicInteger(0);
        int initThread = 5;
        if (tables.size() > 8) {
            initThread = 8;
        }
        ExecutorService executorService = Executors.newFixedThreadPool(initThread);
        tables.stream()
                .forEach(tb -> {
                    executorService.execute(() -> {
                        Collection<Map<String, Object>> data = null;
                        StopWatch subStopWatch = new StopWatch();
                        int totalCount = 0;
                        StringBuilder sqlBuilderTracking = new StringBuilder();

                        try {
                            subStopWatch.start(String.format("开始获取模板信息:%s", tb.getTableName()));
                            SyncChangeLogTableDto table = tb;
                            Collection<String> sqlTemplate = table.getTemplates();//getSqlTemplateByMapping(table);
                            subStopWatch.stop();

                            subStopWatch.start(String.format("开始获取模板信息:%s", tb.getTableName()));
                            if (CollectionUtils.isEmpty(sqlTemplate)) {
                                throw new SaasBadException(String.format("数据(mapKey=%s)找不到对应模板信息", table.getMapKey()));
                            }
                            subStopWatch.stop();

                            subStopWatch.start(String.format("开始执行数据抓取:%s", tb.getTableName()));
                            /**
                             * store
                             */
                            Collection<String> paths = null;
                            int skipOffset = 0;


                            boolean isContinue = false;
                            do {
                                ChangeLogDataResult result = getChangelogData(
                                        tb, skipOffset,
                                        request
                                );
                                data = result.getData();
                                skipOffset = result.getOffset();
                                totalCount += data.size();
                                isContinue = !CollectionUtils.isEmpty(data) && data.size()>=result.getPageSize();

                                sqlBuilderTracking.append(result.getTrackingBuilder());
                                detailContainer.put(tb.getTableName(), String.format("总数量:%s", data.size()));

                                if (!CollectionUtils.isEmpty(data)) {
                                    /**
                                     * SQL
                                     */
                                    Collection<TranslateSqlResult> sqls = translate2Sql(
                                            ChangeLogContext.create(tb, sqlTemplate, request, regionConfigItem), data);
                                    processedTotalCount.addAndGet(data.size());

                                    /**
                                     * store
                                     */
                                    paths = this.store(tb, rootPath, sqls);
                                    if (!CollectionUtils.isEmpty(paths)) {
                                        storedPaths.addAll(paths);
                                    }
                                }
                            } while (isContinue);

                            successAtomicCount.incrementAndGet();

                            subStopWatch.stop();
                            processedMessage.append(String.format("[cost:%s s]-Done for table=%s(数量=%s);",
                                    subStopWatch.getTotalTimeSeconds(),
                                    tb.getTableName(),
                                    totalCount));

                            System.err.println(String.format("%s: done for table=%s", new Date(), tb.getTableName()));
                        } catch (Exception ex) {
                            if (subStopWatch.isRunning()) {
                                subStopWatch.stop();
                            }

                            processedMessage.append(String.format("[cost:%s s]-Error for table=%s(数量=%s;消息=%s);",
                                    subStopWatch.getTotalTimeSeconds(),
                                    tb.getTableName(),
                                    totalCount,
                                    ex.getMessage())
                            );
                            detailContainer.put(tb.getTableName(), String.format("异常:%s", ex.toString()));
                            changeLogBadExceptions.add(ex);
                            ex.printStackTrace();
                        } finally {
                            processedMessage.append(sqlBuilderTracking);

                            processedAtomicIndex.incrementAndGet();
                            if (processedAtomicIndex.get() % 5 == 0) {
                                executeIndex.incrementAndGet();
                                synchronized (countDownLatch) {
                                    try {
                                        Object[] params = new Object[4];
                                        params[0] = 5;
                                        params[1] = processedMessage.toString();
                                        params[2] =LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
                                        params[3] = requestId;
                                        this.jdbcTemplate.update(UPDATE_CHANGE_LOG, params);
                                    } catch (Exception ex) {
                                        ex.printStackTrace();
                                    }
                                }
                            }

                            countDownLatch.countDown();
                        }
                    });
                });
        watch.stop();

        System.err.println("waiting for all done");
        countDownLatch.await();

        try {
            Object[] params = new Object[4];
            params[0] = (tables.size() - executeIndex.get() * 5);
            params[1] = String.format("[完成:总执行次数=%s]:%s", processedTotalCount, processedMessage);
            params[2] =LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
            params[3] = requestId;
            this.jdbcTemplate.update(UPDATE_CHANGE_LOG, params);
        } catch (Exception ex) {
            ex.printStackTrace();
        }

        if (successAtomicCount.get() == 0 || storedPaths.size() == 0) {
            System.err.println("not success atomic count data");

            throw new SaasNofoundException(String.format("找不到任何条件(FromTime=%s;RegionCode=%s)的数据",
                    request.getFromTime(), request.getRegionCode()));
        }
    }

    private void responseWithFile(String title,String realPath, HttpServletResponse response) throws IOException {
        File dir = new File(realPath);
        File[] files = dir.listFiles();
        if(files==null) {
            response.setStatus(HttpStatus.NOT_FOUND.value());
            response.getWriter().write(String.format("找不到要下载的文件(%s)", title));
            return;
        }

        Optional<File> fileOptional =
                Arrays.stream(files).filter(ii -> ii.getName().endsWith(".zip"))
                        .findFirst();
        if (!fileOptional.isPresent()) {
            response.setStatus(HttpStatus.NOT_FOUND.value());
            response.getWriter().write(String.format("%s 底下找不符合的打包文件, 请重新生成", title));
            return;
        }

        File selectedFile = fileOptional.get();
        String encoding = Environment.getInstance().getEncoding();
        response.setContentType("application/x-download; charset=" + encoding + "");
        response.setHeader("Content-Disposition", "attachment;filename=\"" + java.net.URLEncoder.encode(selectedFile.getName(), encoding) + "\"");

        try (ServletOutputStream outputStream = response.getOutputStream()) {
            try (BufferedInputStream reader = new BufferedInputStream(new FileInputStream(selectedFile))) {
                byte[] buffer = new byte[4096];
                int i = -1;
                while ((i = reader.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, i);
                }
            }

            outputStream.flush();
        }
    }

    private static Collection<ChangeLogTableMapDto> _changeLogTableMaps;
    private Collection<ChangeLogTableMapDto> getAllChangeLogTemplates() {
        _changeLogTableMaps = new ArrayList<>();
        if (CollectionUtils.isEmpty(_changeLogTableMaps)) {
            _changeLogTableMaps = jdbcTemplate.query(SYNC_DYNAMIC_DATA_MAP_TEMPLATES, (rs, rowNum) -> {
                String template = rs.getString("template");
                String mapKey = rs.getString("mapkey");
                String region = rs.getString("region");

                return ChangeLogTableMapDto.create(mapKey, template,region);
            });
        }

        if(_changeLogTableMaps==null) {
            _changeLogTableMaps = new ArrayList<>();
        }

        return _changeLogTableMaps;
    }

    private static Collection<SyncChangeLogTableDto> _syncChangeLogTables;
    private Collection<SyncChangeLogTableDto> getChangelogTables(DownChangelogRequest request) {
        _syncChangeLogTables = new ArrayList<>();
        if (CollectionUtils.isEmpty(_syncChangeLogTables)) {
            _syncChangeLogTables = jdbcTemplate.query(SYNC_CHANGE_LOG_SQL_TEMPLATE, (rs, rowNum) -> {
                String name = rs.getString("name");
                String tableName = rs.getString("table_name");
                String appId = rs.getString("app_id");
                String config = rs.getString("config");
                String target_config = rs.getString("target_config");
                String url = rs.getString("url");
                String filter = rs.getString("filter");
                boolean is_online = rs.getBoolean("is_online");

                return SyncChangeLogTableDto.create(name, tableName, filter, appId, config, target_config, url, is_online);
            });

            Collection<ChangeLogTableMapDto> templates = getAllChangeLogTemplates();
            _syncChangeLogTables = _syncChangeLogTables.stream()
                    .filter(ii -> templates.stream().anyMatch(ix -> ix.getMapKey().equalsIgnoreCase(ii.getMapKey())))
                    .collect(Collectors.toList());


            if (!CollectionUtils.isEmpty(_syncChangeLogTables)) {
                for (SyncChangeLogTableDto syncChangeLogTableDto : _syncChangeLogTables) {
                    Collection<ChangeLogTableMapDto> selectedTemplates = templates.stream().filter(ii -> ii.getMapKey()
                                    .equalsIgnoreCase(syncChangeLogTableDto.getMapKey()) &&
                                    (!StringUtils.hasLength(ii.getRegion()) || !StringUtils.hasLength(request.getRegionCode())
                                            || ii.getRegion().contains(request.getRegionCode())))
                            .collect(Collectors.toList());
                    syncChangeLogTableDto.assignTemplates(selectedTemplates);
                }
            }
        }

        return _syncChangeLogTables;
    }

    private ChangeLogDataResult getChangelogData(SyncChangeLogTableDto table,int skipOffset,DownChangelogRequest request) {
        String appId = table.getAppId();
        ChangeLogDataResult data = getData(table,skipOffset, getSelectedDataSource(appId),request);

        return data;
    }

    private static final Map<String,DataSource> _selectedDataSourceByAppId=new ConcurrentHashMap<>();
    private DataSource getSelectedDataSource(String appId) {
        if(isObpmDataSource(appId)) {
            return null;
        }

        DataSource dataSource = _selectedDataSourceByAppId.get(appId);
        if (dataSource == null) {
            Application application = (Application) DesignTimeSerializableCache.get(appId);
            if(application==null) {
                return null;
            }
            dataSource = application.getDataSourceDefine();
            if(dataSource==null) {
                return null;
            }
            _selectedDataSourceByAppId.put(appId, dataSource);
        }

        return dataSource;
    }

    private static final Map<String,JdbcTemplate> _selectedJdbcTemplateByDsId=new ConcurrentHashMap<>();
    private ChangeLogDataResult getData(SyncChangeLogTableDto table,int skipOffset, DataSource dataSource, DownChangelogRequest request) {
        String currentSql = null;
        String dsIdentity = null;
        try {
            JdbcTemplate selectedJdbcTemplate = jdbcTemplate;
            if (!isObpmDataSource(table.getAppId())) {
                if (dataSource == null) {
                    throw new SaasBadException(String.format("找不到(appId=%s;tableName=%s;config=%s;)的dataSource数据源-v2",
                            table.getAppId(), table.getTableName(),
                            table.getConfig())
                    );
                }

                dsIdentity = dataSource.getIdentityUri();
                selectedJdbcTemplate =
                        _selectedJdbcTemplateByDsId.get(dataSource.getIdentityUri());
                if (selectedJdbcTemplate == null) {
                    javax.sql.DataSource ds = PersistenceUtils.getDataSource(dataSource);
                    if (ds == null) {
                        throw new SaasBadException(String.format("(%s)的数据源无效", table.getTableName()));
                    }
                    selectedJdbcTemplate = new JdbcTemplate(ds);
                    _selectedJdbcTemplateByDsId.put(dataSource.getIdentityUri(), selectedJdbcTemplate);
                }
            }

            String regionId = this.regionConfig.getRegionId(request.getRegionCode());
            String sql = table.getSql(regionId, request.getFromTime(), request.getEndTime());

            Collection<Map<String, Object>> mergedResults = new ArrayList<>();
            int offset = skipOffset;
            int take = 2000;
            Collection<Map<String, Object>> currentPageData = null;
            int index = 0;
            StringBuilder sb = new StringBuilder();
            boolean isError = false;
            do {
                try {
                    isError = false;
                    currentSql = String.format("%s limit %s,%s", sql, offset, take);
                    currentPageData = selectedJdbcTemplate.query(currentSql, rse -> {
                        Collection<Map<String, Object>> dsx = new ArrayList<>();
                        ResultSetMetaData metaData = rse.getMetaData();
                        int columnCount = metaData.getColumnCount();
                        while (rse.next()) {
                            Map<String, Object> mp = new HashMap<>();

                            for (int ci = 0; ci < columnCount; ci++) {
                                String column = null;
                                int ciIndex = ci + 1;
                                int columnType = metaData.getColumnType(ciIndex);
                                try {
                                    column = metaData.getColumnLabel(ciIndex);
                                } catch (Exception ex) {
                                    column = metaData.getColumnName(ciIndex);
                                }

                                Object value = null;
                                try {
                                    value = rse.getObject(column);
                                    if (value instanceof Boolean && "bit".equalsIgnoreCase(metaData.getColumnTypeName(ciIndex))) {
                                        value = ((Boolean) value) ? "1" : "0";
                                    }
                                } catch (Exception ex) {
                                    value = String.format("%s:exception=%s", rse.getString(column), ex.toString());
                                    System.err.println(String.format("column=%s;value=%s;", column, value));
                                    ex.printStackTrace();

                                    if (ex.toString().contains("0000-00-00")) {
                                        value = null;
                                    }
                                }

                                mp.put(column, value);
                            }

                            dsx.add(mp);
                        }

                        return dsx;
                    });
                    if (index < 1) {
                        sb.append(String.format("size=%s;sql=%s;", currentPageData.size(), currentSql));
                    }

                    mergedResults.addAll(currentPageData);
                    offset += take;

                    /**
                     * 1万条数据存储一次到文件
                     */
                    if (isReachMaxCount(mergedResults)) {
                        break;
                    }

                    index++;
                } catch (Exception ex) {
                    if (ex.toString().contains("time")) {
                        if (take > 500) {
                            take = take - 100;
                        }
                    }

                    isError = true;
                    index++;

                    sb.append(String.format("exception=%s;size=%s;sql=%s;", ex, currentPageData == null ? "NULL" : currentPageData.size(), currentSql));

                    ex.printStackTrace();
                }
            }
            while (!CollectionUtils.isEmpty(currentPageData) && currentPageData.size() >= take && index < 1_000);

            ChangeLogDataResult dataResult = ChangeLogDataResult.create(offset, take, mergedResults);
            dataResult.appendTracking(String.format("[总共执行次数=%s]跟踪记录:%s;", index, sb));

            return dataResult;
        } catch (Exception ex) {
            throw new ChangeLogBadException(table.getTableName(),
                    String.format("数据获取异常:%s【数据源=%s】;sql=%s;", ex.getMessage(), dsIdentity, currentSql),
                    ex);
        }
    }

    private Collection<TranslateSqlResult> translate2Sql(
            ChangeLogContext changeLogContext, Collection<Map<String,Object>> data) {
        if (CollectionUtils.isEmpty(data)) {
            return Collections.emptyList();
        }

        /**
         insert into dynamic_data_map_templates
         (mapkey,template)VALUES
         ('tenant-app-securityman','insert into tlk_securityman (
         id,formid,created,LASTMODIFIED,domainid,item_pirture,item_leavedate,item_deptid,item_phone,item_jobtype,item_deptname,item_userid,item_securityname,item_isinmycompany,item_companyname,item_iscertified,item_lastauthtime,item_documenttype,item_documentid,item_entrydate,item_domain_id,item_insure,item_birthdate,item_sex,item_nationality,item_education,item_height,item_bloodtype,item_nation,item_politicaloutlook,
         item_maritalstatus,item_homeaddress,item_householdtype,item_emergencycontact,item_emergencyphone,item_licenselevel,item_isveteran,item_workyears,item_diseasehistory,item_idfacephoto,item_idnationphoto,item_idvalidations,item_idvalidatione,item_idaddress,item_headphoto,item_shiming,item_beizhu,item_registerpoliceidindex
         )VALUES(
         ''#{id}'',''#{formid}'',''#{created}'',''#{LASTMODIFIED}'',''#{domain_id}'',''#{item_pirture}'',''#{item_leavedate}'',''#{item_deptid}'',''#{item_phone}'',''#{item_jobtype}'',''#{item_deptname}'',''#{item_userid}'',''#{item_securityname}'',''#{item_isinmycompany}'',''#{item_companyname}'',''#{item_iscertified}'',''#{item_lastauthtime}'',''#{item_documenttype}'',''#{item_documentid}'',''#{item_entrydate}'',''#{domain_id}'',
         ''#{item_insure}'',''#{item_birthdate}'',''#{item_sex}'',''#{item_nationality}'',''#{item_education}'',''#{item_height}'',''#{item_bloodtype}'',''#{item_nation}'',''#{item_politicaloutlook}'',''#{item_maritalstatus}'',''#{item_homeaddress}'',''#{item_householdtype}'',''#{item_emergencycontact}'',''#{item_emergencyphone}'',''#{item_licenselevel}'',''#{item_isveteran}'',''#{item_workyears}'',''#{item_diseasehistory}'',
         ''#{id_face_photo}'',''#{id_header_photo}'',''#{item_idvalidations}'',''#{item_idvalidatione}'',''#{item_idaddress}'',''#{item_headphoto}'',''#{item_shiming}'',''#{item_beizhu}'',''#{item_registerpoliceidindex}''
         )ON DUPLICATE KEY UPDATE item_pirture=''#{item_pirture}'',item_leavedate=''#{item_leavedate}'',item_deptid=''#{item_deptid}'',item_phone=''#{item_phone}'',
         item_jobtype=''#{item_jobtype}'',item_deptname=''#{item_deptname}'',
         item_userid=''#{item_userid}'',item_securityname=''#{item_securityname}'',item_isinmycompany=''#{item_isinmycompany}'',item_companyname=''#{item_companyname}'',item_iscertified=''#{item_iscertified}'',
         item_lastauthtime=''#{item_lastauthtime}'',item_documenttype=''#{item_documenttype}'',item_documentid=''#{item_documentid}'',item_entrydate=''#{item_entrydate}'',
         item_domain_id=''#{item_domain_id}'',item_insure=''#{item_insure}'',item_birthdate=''#{item_birthdate}'',item_sex=''#{item_sex}'',item_nationality=''#{item_nationality}'',item_education=''#{item_education}'',
         item_height=''#{item_height}'',item_bloodtype=''#{item_bloodtype}'',item_nation=''#{item_nation}'',item_politicaloutlook=''#{item_politicaloutlook}'',item_maritalstatus=''#{item_maritalstatus}'',
         item_homeaddress=''#{item_homeaddress}'',item_householdtype=''#{item_householdtype}'',item_emergencycontact=''#{item_emergencycontact}'',
         item_emergencyphone=''#{item_emergencyphone}'',item_licenselevel=''#{item_licenselevel}'',
         item_isveteran=''#{item_isveteran}'',item_workyears=''#{item_workyears}'',item_diseasehistory=''#{item_diseasehistory}'',item_idfacephoto=''#{item_idfacephoto}'',item_idnationphoto=''#{item_idnationphoto}'',
         item_idvalidations=''#{item_idvalidations}'',item_idvalidatione=''#{item_idvalidatione}'',item_idaddress=''#{item_idaddress}'',item_headphoto=''#{item_headphoto}'',
         item_registerpoliceidindex=''#{item_registerpoliceidindex}''');
         */
        Collection<TranslateSqlResult> sqlResult = new ArrayList<>();
        data.stream().forEach(dt -> {
            Collection<String> paths = new HashSet<>();
            Map<String, Object> mapValues = changeLogContext.getParams(dt);

            String translatedSql =
                    changeLogContext.getSqlTemplates().stream().map(tmp -> {
                        try {
                            String mapResult = tmp;
                            for (Object key : mapValues.keySet()) {
                                if (key != null) {
                                    StringBuilder sbKey = new StringBuilder();
                                    sbKey.append(String.format("key=%s;FieldNames.isFile=%s;", key, FieldNames.isFile(String.valueOf(key))));
                                    Object originalValue = null;
                                    try {
                                        originalValue = mapValues.get(key);
                                        sbKey.append(String.format("originalValue=%s;", originalValue));
                                        if (originalValue != null && FieldNames.isFile(String.valueOf(key))) {
                                            Collection<String> extractPaths = FieldNames.extractPathValues(
                                                    this.regionConfig.getSourceSite(),
                                                    String.valueOf(originalValue));
                                            if (!CollectionUtils.isEmpty(extractPaths)) {
                                                for (String singlePath : extractPaths) {
                                                    if (!StringUtils.hasLength(singlePath)) {
                                                        continue;
                                                    }

                                                    /**
                                                     * 针对OBS挂在的, 不需要添加obs的功能
                                                     */
                                                    if (StringUtils.hasLength(this.regionConfig.getSourceSite()) &&
                                                            this.regionConfig.getSourceSite().contains("cn-north-1.myhuaweicloud.com")) {
                                                        singlePath = singlePath.replace(
                                                                String.format("%s/obpm", this.regionConfig.getSourceSite()),
                                                                this.regionConfig.getSourceSite());
                                                    }

                                                    /**
                                                     * 互联网的obs不需要添加/obpm的路径
                                                     */
                                                    if (singlePath.contains("02obs-file-system-obpm-uploads.obs.cn-north-1.myhuaweicloud.com")) {
                                                        if (singlePath.contains("/obpm/uploads/")) {
                                                            singlePath = singlePath.replace("/obpm/uploads/", "/");
                                                        }
                                                    }

                                                    paths.add(singlePath);
                                                }
                                            }
                                        }

                                        mapResult = mappingSqlValueTranslator.get(changeLogContext, mapResult, String.valueOf(key), originalValue);
                                    } catch (Exception ex) {
                                        throw new SaasBadException(String.format("字段信息异常:key=%s;value=%s;详情=%s", key, originalValue, ExceptionUtils.getStackMessage(ex)));
                                    }
                                }
                            }

                            if(StringUtils.hasLength(mapResult)) {
                                mapResult = mapResult.trim();
                                if (mapResult.endsWith(";")) {
                                    mapResult = mapResult.substring(0, mapResult.length() - 1);
                                }
                            }

                            return mapResult;
                        } catch (Exception ex) {
                            ex.printStackTrace();
                            return String.format("[%s];ex_Sql=%s;详细=%s;",
                                    mapValues.containsKey("id") ? "" : mapValues.get("id"),
                                    tmp, ExceptionUtils.getStackMessage(ex));
                        }
                    }).collect(Collectors.joining(";"));

            sqlResult.add(TranslateSqlResult.create(translatedSql, paths));
        });

        return sqlResult;
    }

    private Collection<String> store(SyncChangeLogTableDto table,String rootPath,
                                     Collection<TranslateSqlResult> sqlResults) {
        if (CollectionUtils.isEmpty(sqlResults)) {
            return Collections.emptyList();
        }
        StringBuilder builder = new StringBuilder();
        sqlResults.forEach(sqlResult -> {
            String sql = sqlResult.getSql();
            if (StringUtils.hasLength(sql)) {
                sql = sql.trim();
                if (sql.endsWith(";")) {
                    sql = sql.substring(0, sql.length() - 1);
                }

                builder.append(String.format("%s;", sql));
            }
        });

        String storePath = rootPath.concat(File.separator).concat(table.getAppId());

        Collection<String> pathResult = new ArrayList<>();
        String sql_path = String.format("%s%s_%s.sql",
                storePath,
                table.getTableName(),
                table.getAppId()
        );
        //FileUtil.writeUtf8String(builder.toString(), sql_path);
        FileUtil.appendString(builder.toString(), sql_path, StandardCharsets.UTF_8);

        pathResult.add(sql_path);
        Collection<String> filePaths = sqlResults.stream().flatMap(ii -> ii.getPaths().stream())
                .distinct()
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(filePaths)) {
            String file_path = sql_path.replace(".sql", "_file.txt");
            //FileUtil.writeLines(filePaths, file_path, "utf-8");
            FileUtil.appendLines(filePaths, file_path, "utf-8");
            pathResult.add(file_path);
        }

        return pathResult;
    }

    private boolean isObpmDataSource(String appId) {
        return appId.contains("tenant");
    }

    private String getDbName(String url) {
        if (!StringUtils.hasLength(url)) {
            return null;
        }

        if(!url.contains("?")) {
            return url;
        }

        String prefix = url.substring(0, url.indexOf("?"));
        return prefix.substring(url.lastIndexOf("/"));
    }

    private boolean isReachMaxCount(Collection<Map<String, Object>> data) {
        if (CollectionUtils.isEmpty(data)) {
            return false;
        }

        return data.size() >= 10_000;
    }

    /**
     *判断同步数据是否已经拉取完成。同时预留120S的时间确保文件上传OBS
     * @param zipId: 拉取Id
     * @return false, 数据拉取未完成.
    */
    private boolean isChangeLogsFinished(String zipId){
        Object[] param = new Object[1];
        param[0] = zipId;
        Map<String, Object> queryResult = this.jdbcTemplate.queryForMap(GET_CHANGE_LOG, param);
        long current = ((Integer)(queryResult.get("current"))).longValue();
        long total = ((Integer)(queryResult.get("total"))).longValue();
        LocalDateTime lastUpdatedTime = (LocalDateTime) queryResult.get("last_updated_time");

        return current == total && (Duration.between(lastUpdatedTime, LocalDateTime.now()).getSeconds() > 180);
    }
}
