package com.bcxin.tenant.domain.v5.services.impls;

import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.enums.CredentialType;
import com.bcxin.Infrastructures.enums.EmploymentStatus;
import com.bcxin.Infrastructures.enums.MemberType;
import com.bcxin.Infrastructures.enums.TrueFalseStatus;
import com.bcxin.Infrastructures.exceptions.NotAllowedTenantException;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.api.interfaces.identities.requests.ChangeBasicIdentityRequest;
import com.bcxin.tenant.domain.entities.EmployeeEntity;
import com.bcxin.tenant.domain.entities.TenantUserCredentialsEntity;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.readers.dtos.EmployeeDepartIdDto;
import com.bcxin.tenant.domain.v5.entities.TDepartmentUserEntity;
import com.bcxin.tenant.domain.v5.entities.TUserEntity;
import com.bcxin.tenant.domain.v5.enums.IdCardType;
import com.bcxin.tenant.domain.v5.repositories.TDepartmentRepository;
import com.bcxin.tenant.domain.v5.repositories.TDepartmentUserRepository;
import com.bcxin.tenant.domain.v5.repositories.TUserRepository;
import com.bcxin.tenant.domain.v5.services.TUserService;
import com.bcxin.tenant.domain.v5.services.commands.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

@Service
public class TUserServiceImpl implements TUserService {
    private final static Logger logger  = LoggerFactory.getLogger(TUserServiceImpl.class);
    private final TUserRepository userRepository;
    private final TDepartmentUserRepository departmentUserRepository;
    private final TDepartmentRepository tDepartmentRepository;
    private final TenantDbReader dbReader;
    private final UnitWork unitWork;

    public TUserServiceImpl(TUserRepository userRepository,
                            TDepartmentUserRepository departmentUserRepository,
                            TDepartmentRepository tDepartmentRepository,
                            TenantDbReader dbReader,
                            UnitWork unitWork) {
        this.userRepository = userRepository;
        this.departmentUserRepository = departmentUserRepository;
        this.tDepartmentRepository = tDepartmentRepository;
        this.dbReader = dbReader;
        this.unitWork = unitWork;
    }

    @Override
    public void create(SyncUserCommand command) {
        TUserEntity tUser =
                TUserEntity.create(command.getId(), command.getName(),
                        command.getLoginNo(), command.getDomainId(),
                        command.getTelephone()
                );

        tUser.changeNumber(command.getCredentialType() == null ? "" : command.getCredentialType().name(),
                command.getCredentialNumber());

        String defaultDepartId = tUser.getDefaultDepartId();
        if (!CollectionUtils.isEmpty(command.getDepartmentIds())) {
            defaultDepartId = command.getDepartmentIds().stream().findFirst().get();
        }

        tUser.change(defaultDepartId, command.getIsDomainAdmin() == TrueFalseStatus.True);

        this.unitWork.executeTran(() -> {
            this.userRepository.save(tUser);
            if (command.getDepartmentIds() != null) {
                Collection<TDepartmentUserEntity> departmentUsers
                        = command.getDepartmentIds().stream().map(ii -> {
                    return TDepartmentUserEntity.create(ii, tUser.getId());
                }).collect(Collectors.toList());

                this.departmentUserRepository.saveAll(departmentUsers);
            }
        });
    }

    @Override
    public void update(SyncUserCommand command) {
        TUserEntity tUser = this.userRepository.getById(command.getId());
        if (tUser == null) {
            this.create(command);
        } else {
            tUser.change(command.getName(), command.getLoginNo());
        }

        this.unitWork.executeTran(() -> {
            this.userRepository.save(tUser);
        });
    }

    @Override
    public void dispatch(SyncUserRoleCommand command) {
        if (!command.isFromAdmin()) {
            throw new NotAllowedTenantException("非组织或者部门管理员不允许执行该操作");
        }
        //初始登录的才校验
        if(TenantContext.getInstance().getUserContext().get() != null){
            if (command.getEmployeeIds().stream().anyMatch(ix -> ix.equalsIgnoreCase(TenantContext.getInstance().getUserContext().get().getEmployeeId())))
            {
                throw new NotAllowedTenantException("不允许更改自己的应用权限");
            }
        }

        command.validate();
        Collection<TDepartmentUserEntity> departmentUserEntities = this.departmentUserRepository
                .getByOrganIdAndUserIds(command.getOrganizationId(), command.getEmployeeIds());

        Collection<EmployeeDepartIdDto> employeeDepartIds =
                this.dbReader.getEmployeeDepartIds(command.getOrganizationId(), command.getEmployeeIds());

        this.unitWork.executeTran(() -> {
            command.getEmployeeIds().forEach(ei -> {
                Optional<String> departIdOptional = employeeDepartIds.stream().filter(ix -> ix.getEmployeeId().equals(ei))
                        .map(ii -> ii.getDepartId())
                        .findFirst();

                if (!departIdOptional.isPresent()) {
                    throw new NotFoundTenantException(String.format("数据异常, 该用户(%s)未分配部门，无法进行角色授权", ei));
                }

                String departId = departIdOptional.get();
                /**
                 * 允许roleids为空，这时候用于清除授权信息
                 */
                Collection<String> roleIds = new ArrayList<>();
                if (!CollectionUtils.isEmpty(command.getRoleIds())) {
                    roleIds.addAll(command.getRoleIds());
                }

                Collection<String> notExistsRoleIds = roleIds.stream().filter(ix ->
                        !departmentUserEntities.stream().anyMatch(ii -> StringUtils.hasLength(ii.getRoleId()) && ii.getRoleId().equals(ix)))
                        .collect(Collectors.toList());

                for (String rId : notExistsRoleIds) {
                    TDepartmentUserEntity tDepartmentUser = TDepartmentUserEntity.create(departId, ei);
                    tDepartmentUser.change(departId, rId);
                    this.departmentUserRepository.save(tDepartmentUser);
                }

                Collection<TDepartmentUserEntity> toBeRemovedEntities =
                        departmentUserEntities.stream().filter(ix ->
                                !roleIds.contains(ix.getRoleId()))
                                .collect(Collectors.toList());
                this.departmentUserRepository.deleteAll(toBeRemovedEntities);
            });
        });
    }

    @Override
    public void dispatch(ClearUserRoleCommand command) {
        command.validate();

        this.unitWork.executeTran(() -> {
            this.departmentUserRepository.deleteByOrganIdAndUserId(command.getOrganizationId(), command.getId());
        });
    }

    @Override
    public void dispatch(UpdateUserCommand command) {
        command.validate();
        EmployeeEntity employee = command.getEmployee();
        String id = employee.getId();
        Optional<TUserEntity> userEntityOptional = this.userRepository.findById(id);
        String defaultDepartId = null;
        if (employee.getDefaultDepartment() != null) {
            defaultDepartId = employee.getDefaultDepartment().getId();
        }

        String organId = employee.getOrganization().getId();
        if (!userEntityOptional.isPresent()) {
            if (command.getEmployee().getStatus() == EmploymentStatus.OnJob) {
                TenantUserCredentialsEntity userCredentials = employee.getTenantUser().getSelectedCredential();
                this.create(
                        SyncUserCommand.create(employee.getId(), employee.getTenantUser().getName(),
                        employee.getTenantUser().getSelectIdNum(),
                        organId, employee.getMasterSlaveType(),
                        StringUtils.hasLength(defaultDepartId) ? Collections.singleton(defaultDepartId) : null,
                        userCredentials == null ? CredentialType.IdCard : userCredentials.getCredentialType(),
                        userCredentials == null ? "" : userCredentials.getNumber(),
                        employee.getDomainAdmin(),
                                employee.getTenantUser().getTelephone()
                ));
            }

            return;
        }

        TUserEntity entity = userEntityOptional.get();
        String finalDefaultDepartId = defaultDepartId;
        this.unitWork.executeTran(() -> {
            if (command.getEmployee().getStatus() == EmploymentStatus.OffJob) {
                //entity.markDeleted();
                entity.markDeleted();
            } else {
                entity.markUnDeleted();
                entity.change(employee.getTenantUser().getName(), employee.getTenantUser().getTelephone());
                entity.change(finalDefaultDepartId, employee.getDomainAdmin() == TrueFalseStatus.True);
                TenantUserCredentialsEntity credentials = employee.getTenantUser().getSelectedCredential();
                if(credentials!=null) {
                    entity.changeNumber(credentials.getCredentialType().name(), credentials.getNumber());
                }

                Collection<TDepartmentUserEntity> departmentUsers = departmentUserRepository.getByIds(organId, Collections.singleton(employee.getId()));
                if (departmentUsers.size() > 0) {
                    departmentUsers.stream().forEach(di -> {
                        di.assignDepartId(finalDefaultDepartId);
                    });
                } else {
                    departmentUsers.add(TDepartmentUserEntity.create(finalDefaultDepartId, employee.getId()));
                }

                this.departmentUserRepository.saveAll(departmentUsers);
            }

            this.userRepository.save(entity);
        });
    }

    @Override
    public void changeBasic(ChangeBasicIdentityRequest changeBasicIdentityRequest) {
        /*TUserEntity searchUser = TUserEntity.createTelePhone(changeBasicIdentityRequest.getOldPhone());
        Example<TUserEntity> example = Example.of(searchUser);
        List<TUserEntity> list = this.userRepository.findAll(example);
        if(list.size() > 0){
            for (TUserEntity tUserEntity : list) {
                tUserEntity.change(changeBasicIdentityRequest.getName(),changeBasicIdentityRequest.getTelephone());
            }
            this.userRepository.saveAll(list);
        }*/
        Collection<TUserEntity> list = this.userRepository.getByLoginNo(changeBasicIdentityRequest.getOldPhone());
        if(list.size() > 0){
            for (TUserEntity tUserEntity : list) {
                tUserEntity.change(changeBasicIdentityRequest.getName(),changeBasicIdentityRequest.getTelephone());
            }
            this.userRepository.saveAll(list);
        }
    }

    @Override
    public void dispatch(SyncMemberCommand command) {
        try {
            String defaultDepartId = this.tDepartmentRepository.getTop1ByOrganId(command.getOrganizationId());
            this.unitWork.executeTran(() -> {
                TUserEntity tUser = TUserEntity.create(
                        command.getId(),
                        command.getName(),
                        command.getId(),
                        command.getOrganizationId(),
                        command.getTelephone()
                );

                boolean domainAdmin = command.getMemberType() == MemberType.Master;
                tUser.change(defaultDepartId, domainAdmin);
                tUser.changeNumber(IdCardType.IdCard.getText(), "--");
                this.userRepository.save(tUser);
            });
        } catch (Exception ex) {
            logger.error("同步团队成员(id={},name={},orgId={})发生异常", command.getId(), command.getName(), command.getOrganizationId(), ex);
            //todo: 临时注释; 确保不影响团队API
        }
    }
}
