package com.bcxin.tenant.domain.v5.dispatches.impls;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.bcxin.Infrastructures.components.RetryProvider;
import com.bcxin.Infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.exceptions.NotSupportTenantException;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.domain.v5.dispatches.DataDispatcher;
import com.bcxin.tenant.domain.v5.dispatches.DispatchProperties;
import com.bcxin.tenant.domain.v5.dispatches.requests.DispatchRequestAbstract;
import com.bcxin.tenant.domain.v5.dispatches.requests.DispatchType;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.io.UnsupportedEncodingException;
import java.util.Base64;
import java.util.Collection;

@Component
public class DataDispatcherImpl implements DataDispatcher {
    private final DispatchProperties dispatchProperties;
    private final RetryProvider retryProvider;

    public DataDispatcherImpl(DispatchProperties dispatchProperties,
                              RetryProvider retryProvider) {
        this.dispatchProperties = dispatchProperties;
        this.retryProvider = retryProvider;
    }

    /**
     * 临时不用数据分发
     * @param request
     */
    @Override
    public void dispatch(DispatchRequestAbstract request) {
        if (!StringUtils.hasLength(this.dispatchProperties.getApi()) || true) {
            return;
        }

        Assert.hasLength(this.dispatchProperties.getApi(), "patch.dispatch.api配置无效!");
        Assert.isTrue(this.dispatchProperties.getCertificates().size() > 0, "patch.dispatch.certificates配置无效!");

        retryProvider.execute(() -> {
            try {
                try (CloseableHttpClient httpClient = HttpClients.createDefault()) {
                    HttpRequestBase requestBase = this.getHttpRequestBase(request);
                    HttpResponse response = httpClient.execute(requestBase);

                    int status = response.getStatusLine().getStatusCode();
                    if (status == 401) {
                        throw new UnAuthorizedTenantException("未授权");
                    } else {
                        if (status < 200 || status > 300) {
                            throw new BadTenantException(response.toString());
                        }
                    }
                }
            } catch (Exception ex) {
                throw new BadTenantException(ex.toString());
            }
        }, 2);
    }

    protected HttpRequestBase getHttpRequestBase(DispatchRequestAbstract request)
            throws UnsupportedEncodingException {
        Collection<DispatchProperties.Certificate> certificates = this.dispatchProperties.getCertificates()
                .get(request.getConfigKey());
        DispatchProperties.Certificate certificate = getCertificate(request.getDispatchType(), certificates);

        String clientId = certificate.getClientId();
        String secret = certificate.getSecret();

        String authString = String.format("%s:%s", clientId, secret);
        String authStringEnc = Base64.getEncoder().encodeToString(authString.getBytes("UTF-8"));

        HttpPost post = new HttpPost(dispatchProperties.getApi());
        post.setHeader("Authorization", String.format("Basic %s", authStringEnc));
        post.setHeader("Content-Type", "application/json");
        JSONObject body = new JSONObject();
        body.put("routeKey", certificate.getRouteKey());
        //直接推送数据的方式
        body.put("messageType", 1);
        body.put("data", request.getPostData());
        StringEntity entity = new StringEntity(JSONObject.toJSONString(body, SerializerFeature.WriteMapNullValue), "UTF-8");
        post.setEntity(entity);

        return post;
    }

    private DispatchProperties.Certificate getCertificate(DispatchType dispatchType,
                                                          Collection<DispatchProperties.Certificate> certificates) {
        switch (dispatchType) {
            case Created:
                DispatchProperties.Certificate createdCertificate = certificates.stream()
                        .filter(ii -> ii.getMethod().equals("created"))
                        .findFirst().get();
                if (createdCertificate == null) {
                    throw new ArgumentTenantException("created配置无效!");
                }
                return createdCertificate;
            case Edited:
                DispatchProperties.Certificate editedCertificate = certificates.stream()
                        .filter(ii -> ii.getMethod().equals("edited"))
                        .findFirst().get();
                if (editedCertificate == null) {
                    throw new ArgumentTenantException("edited配置无效!");
                }
                return editedCertificate;
            case Deleted:
                DispatchProperties.Certificate deletedCertificate = certificates.stream()
                        .filter(ii -> ii.getMethod().equals("deleted"))
                        .findFirst().get();
                if (deletedCertificate == null) {
                    throw new ArgumentTenantException("deleted!");
                }

                return deletedCertificate;
            default:
                throw new NotSupportTenantException("");
        }
    }
}
