package com.bcxin.tenant.bcx.domains.services.impls;

import com.bcxin.tenant.bcx.domains.entities.BusOrganRegistryFormsEntity;
import com.bcxin.tenant.bcx.domains.repositories.OrganRegistryFormsRepository;
import com.bcxin.tenant.bcx.domains.repositories.criterias.OrganRegistryFormsSearchCriteria;
import com.bcxin.tenant.bcx.domains.services.MetaPageService;
import com.bcxin.tenant.bcx.domains.services.OrganRegistryFormsService;
import com.bcxin.tenant.bcx.domains.services.commands.DeleteDataCommand;
import com.bcxin.tenant.bcx.domains.services.commands.organregistry.BatchLogicalDeleteOrganRegistryFormsCommand;
import com.bcxin.tenant.bcx.domains.services.commands.organregistry.CreateOrganRegistryFormsCommand;
import com.bcxin.tenant.bcx.domains.services.commands.organregistry.LogicalDeleteOrganRegistryFormsCommand;
import com.bcxin.tenant.bcx.domains.services.commands.organregistry.UpdateOrganRegistryFormsCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.BatchLogicalDeletePageCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.CreateMetaPageCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.LogicalDeletePageCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.UpdateMetaPageCommand;
import com.bcxin.tenant.bcx.domains.utils.BusinessConstants;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.TenantContext;
import com.bcxin.tenant.bcx.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.bcx.infrastructures.UnitWork;
import com.bcxin.tenant.bcx.infrastructures.components.IdWorker;
import com.bcxin.tenant.bcx.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.bcx.infrastructures.utils.StringUtil;
import org.springframework.stereotype.Service;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

/**
 * 企业注册表单服务实现
 */
@Service
public class OrganRegistryFormsServiceImpl implements OrganRegistryFormsService {
    private final OrganRegistryFormsRepository repository;
    private final MetaPageService metaPageService;
    private final UnitWork unitWork;
    private final IdWorker idWorker;

    public OrganRegistryFormsServiceImpl(OrganRegistryFormsRepository repository,
                                       MetaPageService metaPageService,
                                       UnitWork unitWork, 
                                       IdWorker idWorker) {
        this.repository = repository;
        this.metaPageService = metaPageService;
        this.unitWork = unitWork;
        this.idWorker = idWorker;
    }

    @Override
    public void dispatch(CreateOrganRegistryFormsCommand command) {
        command.validate();

        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel==null) {
            throw new UnAuthorizedTenantException();
        }

        this.unitWork.executeTran(() -> {
            // 1. 先生成 pageId
            String pageId = String.valueOf(this.idWorker.getNextId());
            
            // 2. 创建 MetaPage - 字段值由AOP切面自动填充
            CreateMetaPageCommand createPageCommand = new CreateMetaPageCommand(
                    BusinessConstants.SYS_DEFAULT_PROJECT_ID, // projectId 传 null
                    BusinessConstants.SYS_DEFAULT_ORG_REGISTRATION_MODULE_ID, // moduleId 传 null
                    command.getName(),
                    command.getNote(),
                    command.getFormJson(),
                    command.getSlug(),
                    command.getLogo(),
                    command.getPageType(),
                    command.getBusinessType(),
                    command.getTableName(),
                    command.getEnableBPMFlow() != null ? command.getEnableBPMFlow() : false,
                    command.getEnableAnonymousAccess() != null ? command.getEnableAnonymousAccess() : false,
                    false,
                    null,null,null,0
            );
            pageId = this.metaPageService.dispatch(createPageCommand);
            
            // 3. 创建企业注册表单实体 - 字段值由AOP切面自动填充
            String id = String.valueOf(this.idWorker.getNextId());
            BusOrganRegistryFormsEntity entity = BusOrganRegistryFormsEntity.create(
                    id,
                    command.getInstitutionalTypeCode(),
                    command.getIndustryTypeCode(),
                    command.getRegionCode(),
                    pageId, // 使用生成的 pageId
                    userModel.getUserType(),
                    userModel.getOrganizationId(),
                    userModel.getEmployeeId()
            );
            this.repository.insert(entity);
        });
    }

    @Override
    public void dispatch(UpdateOrganRegistryFormsCommand command) {
        command.validate();

        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel==null) {
            throw new UnAuthorizedTenantException();
        }
        
        this.unitWork.executeTran(() -> {
            // 根据ID查询现有实体（包括已删除的）
            BusOrganRegistryFormsEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }

            // 检查是否已删除
            if (entity.isDeleted()) {
                throw new IllegalArgumentException("无法更新已删除的企业注册表单，ID: " + command.getId());
            }

            // 更新企业注册表单字段
            entity.change(
                    command.getInstitutionalTypeCode(),
                    command.getIndustryTypeCode(),
                    command.getRegionCode(),
                    userModel.getUserType(),
                    userModel.getOrganizationId(),
                    userModel.getEmployeeId()
            );

            this.repository.update(entity);
            
            // 如果有表单定义相关字段需要更新，则更新关联的 MetaPage
            if (command.getName() != null || command.getNote() != null || 
                command.getFormJson() != null || command.getSlug() != null ||
                command.getLogo() != null || command.getPageType() != null ||
                command.getBusinessType() != null || command.getTableName() != null ||
                command.getEnableBPMFlow() != null || command.getEnableAnonymousAccess() != null ||
                command.getSettingJson() != null) {
                
                // lastUpdaterId由AOP切面自动填充
                UpdateMetaPageCommand updatePageCommand = UpdateMetaPageCommand.create(
                        entity.getPageId(),
                        command.getName(),
                        command.getNote(),
                        command.getSlug(),
                        command.getLogo(),
                        command.getPageType(),
                        command.getBusinessType(),
                        command.getTableName(),
                        command.getEnableBPMFlow() != null ? command.getEnableBPMFlow() : false,
                        command.getEnableAnonymousAccess() != null ? command.getEnableAnonymousAccess() : false,
                        false,null,null,null,0
                );
                this.metaPageService.dispatch(updatePageCommand);
            }
        });
    }

    @Override
    public void dispatch(DeleteDataCommand command) {
        // 物理删除，暂时不实现
        throw new UnsupportedOperationException("暂不支持物理删除");
    }

    @Override
    public void dispatch(LogicalDeleteOrganRegistryFormsCommand command) {
        command.validate();
        this.unitWork.executeTran(() -> {
            BusOrganRegistryFormsEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }

            if (entity.isDeleted()) {
                throw new NoFoundTenantException();
            }

            // 1. 先删除关联的 MetaPage
            if (entity.getPageId() != null && !entity.getPageId().isEmpty()) {
                LogicalDeletePageCommand deletePageCommand = new LogicalDeletePageCommand(entity.getPageId());
                this.metaPageService.dispatch(deletePageCommand);
            }

            // 2. 再删除企业注册表单
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
            entity.changeMarkAsDeleted(userModel);
            this.repository.update(entity);
        });
    }

    @Override
    public void dispatch(BatchLogicalDeleteOrganRegistryFormsCommand command) {
        command.validate();
        this.unitWork.executeTran(() -> {
            List<String> ids = command.getIds();
            if (ids == null || ids.isEmpty()) {
                return;
            }
            
            // 1. 批量查询所有entities（一次查询）
            List<BusOrganRegistryFormsEntity> entities = this.repository.getByIds(ids);
            
            // 2. 过滤出未删除的entities并收集pageIds
            List<String> pageIds = new ArrayList<>();
            List<String> validIds = new ArrayList<>();
            
            for (BusOrganRegistryFormsEntity entity : entities) {
                if (entity != null && !entity.isDeleted()) {
                    validIds.add(entity.getId());
                    if (entity.getPageId() != null && !entity.getPageId().isEmpty()) {
                        pageIds.add(entity.getPageId());
                    }
                }
            }
            
            // 如果没有有效的ID需要删除，直接返回
            if (validIds.isEmpty()) {
                return;
            }
            
            // 3. 批量删除所有关联的MetaPages（一次操作）
            if (!pageIds.isEmpty()) {
                this.metaPageService.dispatch(new BatchLogicalDeletePageCommand(pageIds));
            }
            
            // 4. 批量更新所有entities为已删除（一次更新）
            Timestamp now = new Timestamp(System.currentTimeMillis());
            this.repository.batchMarkAsDeleted(validIds, now);
        });
    }

    @Override
    public EntityCollection<BusOrganRegistryFormsEntity> search(OrganRegistryFormsSearchCriteria criteria) {
        return this.repository.search(criteria);
    }

    @Override
    public BusOrganRegistryFormsEntity getById(String id) {
        if (StringUtil.isEmpty(id)) {
            return null;
        }
        return this.repository.getById(id);
    }

    @Override
    public BusOrganRegistryFormsEntity findByCondition(String institutionalTypeCode, String industryTypeCode, String regionCode) {
        return this.repository.findByCondition(institutionalTypeCode, industryTypeCode, regionCode);
    }

    /**
     * 根据ID获取实体（包括已删除的）
     */
    private BusOrganRegistryFormsEntity getByIdIncludeDeleted(String id) {
        if (StringUtil.isEmpty(id)) {
            return null;
        }
        // 这里需要实现一个不包含deleted条件的方法，暂时使用getById
        // 实际项目中应该在Repository中添加getByIdIncludeDeleted方法
        return this.repository.getById(id);
    }
}
