package com.bcxin.backend.certificateSignatures.impls;

import cn.hutool.core.codec.Base64Encoder;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.backend.certificateSignatures.CertificateConfigProperty;
import com.bcxin.backend.certificateSignatures.CertificateSignatureProvider;
import com.bcxin.backend.certificateSignatures.impls.values.SignatureData;
import com.bcxin.backend.certificateSignatures.impls.values.SignaturePdfResult;
import com.bcxin.backend.domain.enums.HnCertCatalog;
import com.bcxin.backend.domain.models.Result;
import com.bcxin.backend.domain.models.SignatureQueuesDTO;
import com.bcxin.backend.domain.repositories.SignatureQueuesRepository;
import com.bcxin.backend.domain.services.impls.ConvertServiceImpl;
import com.bcxin.backend.domain.signature.service.impls.HuNanECUtil;
import com.bcxin.backend.domain.utils.FileUtils;
import com.bcxin.backend.domain.utils.JwtUtil;
import com.bcxin.backend.domain.utils.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.*;

@Component
@Qualifier("hunanin")
public class CertificateSignatureProvider_Hunan extends CertificateSignatureProviderAbstract
        implements CertificateSignatureProvider {
    private static final String XML_TEMPLATE =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
                    + "<license>"
                    + "<catalog catalogId=\"%s\" catalogName=\"%s\" templateId=\"%s\" templateName=\"%s\" version=\"%s\"/>"
                    + "<surface>"
                    + "<item alias=\"holderCodeType\" code=\"certificateHolderTypeCode\" name=\"证件类型\" require=\"1\" value=\"\"/>"
                    + "<item alias=\"issueUnitname\" catetype=\"0\" code=\"certificateIssuingAuthorityName\" name=\"颁证单位\" require=\"1\" type=\"banzhengdanwei\" value=\"\"/>"
                    + "<item alias=\"licenseNumber\" catetype=\"0\" code=\"certificateNumber\" name=\"证照编号\" require=\"1\" type=\"string\" value=\"\"/>"
                    + "<item alias=\"issueDate\" catetype=\"0\" code=\"certificateIssuedDate\" dateFormat=\"yyyy-MM-dd\" name=\"颁证时间\" require=\"1\" type=\"riqi\" value=\"\"/>"
                    + "<item alias=\"validTimeBegin\" catetype=\"0\" code=\"certificateEffectiveDate\" dateFormat=\"yyyy-MM-dd\" name=\"有效期（起始）\" require=\"1\" type=\"riqi\" value=\"\"/>"
                    + "<item alias=\"holder\" catetype=\"0\" code=\"certificateHolderName\" name=\"持证者\" require=\"1\" type=\"string\" value=\"\"/>"
                    + "<item alias=\"validTimeEnd\" catetype=\"0\" code=\"certificateExpiringDate\" dateFormat=\"yyyy-MM-dd\" name=\"有效期（截止）\" require=\"1\" type=\"riqi\" value=\"\"/>"
                    + "<item alias=\"holderCode\" catetype=\"0\" code=\"certificateHolderCode\" name=\"证件号码\" require=\"1\" type=\"string\" value=\"\"/>"
                    + "<item alias=\"csny\" catetype=\"1\" code=\"KZ_csny\" dateFormat=\"yyyy-MM-dd\" name=\"出生年月\" require=\"1\" type=\"riqi\" value=\"\"/>"
                    + "<item alias=\"zz\" catetype=\"1\" code=\"KZ_zz\" name=\"住址\" require=\"1\" type=\"string\" value=\"\"/>"
                    + "<item alias=\"zp\" catetype=\"1\" code=\"KZ_zp\" fileName=\"\" name=\"照片\" require=\"1\" type=\"tupian\" value=\"\"/>"
                    + "<item alias=\"MJXM\" catetype=\"1\" code=\"KZ_MJXM\" name=\"民警姓名\" require=\"1\" type=\"string\" value=\"\"/>"
                    + "<item alias=\"MJSFZHM\" catetype=\"1\" code=\"KZ_MJSFZHM\" name=\"民警身份证号码\" require=\"1\" type=\"string\" value=\"\"/>"
                    + "<item alias=\"MJJH\" catetype=\"1\" code=\"KZ_MJJH\" name=\"民警警号\" require=\"0\" type=\"string\" value=\"\"/>"
                    + "<item alias=\"YHID\" catetype=\"1\" code=\"KZ_YHID\" name=\"用户Id\" require=\"1\" type=\"string\" value=\"\"/>"
                    + "<item alias=\"EWM\" catetype=\"1\" code=\"KZ_EWM\" name=\"二维码\" require=\"1\" type=\"p_qrcode\" value=\"\"/>"
                    + "</surface>"
                    + "</license>";

    private static final Logger logger = LoggerFactory.getLogger(CertificateSignatureProvider_Hunan.class);

    @Value("${myapps.supervise.target-app}")
    private String targetApp;

    public CertificateSignatureProvider_Hunan(CertificateConfigProperty configProperty,
                                                SignatureQueuesRepository signatureQueuesRepository,
                                                @Qualifier("primaryJdbcTemplate") JdbcTemplate jdbcTemplate,
                                                ConvertServiceImpl convertService) {
        super(configProperty, signatureQueuesRepository, jdbcTemplate, convertService);
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected SignaturePdfResult generateSignaturePdf(SignatureQueuesDTO queue) throws IOException {
        String tempPDF = configProperty.getTempPDF();
        String cgiApi = configProperty.getCgiApi();

        SignatureData data = JSONObject.parseObject(queue.getData(), SignatureData.class);
        String infile = tempPDF + "/" + DateUtil.today();
        String filename = System.currentTimeMillis() + "hnzgz_queueId_" + queue.getId() + ".ofd";
        String path = Paths.get(infile, filename).normalize().toString();
        String result = "";
        String message = "";
        Map<String, Object> map = getAuthorInfo(data.getAuthor());

        String xml = handleXml(map, data, targetApp + path);// 报文

        Result ret = HuNanECUtil.xmlDataSend(cgiApi, xml, IdUtil.randomUUID(), getToken());

        if (Result.SUCCESS.equals(ret.getRetType())) {
            result = ret.getData() + "";
            message = "资格证已申请，证书编码:" + result;
        } else {
            message = ret.getMsg();
        }

        return SignaturePdfResult.create(result, message);
    }

    @Override
    public void step3_doSignature() {
        Page<SignatureQueuesDTO> pageQueues = signatureQueuesRepository.pageQueues(3, PageRequest.of(0, 300));
        List<SignatureQueuesDTO> queues = pageQueues.getContent();
        logger.error("待签章的队列数据={}", pageQueues.stream().count());

        if (CollectionUtils.isEmpty(queues)) {
            return;
        }

        String tempPDF = configProperty.getTempPDF();

        String infile = tempPDF + "/" + DateUtil.today();
        queues.parallelStream().forEach(queuesDTO -> {
            processQueue(queuesDTO, infile);
        });
    }

    private Map<String,Object> getAuthorInfo(String author) {
        if (StringUtils.isEmpty(author)) {
            return Collections.EMPTY_MAP;
        }

        Map<String, Object> map = this.jdbcTemplate.queryForMap(
                "select b.name,b.number,d.id as author" +
                        " from obpm2_security.tenant_user_credentials b join obpm2_security.tenant_employees d on b.tenant_user_id.tenant_user_id" +
                        " where e.id='" + author + "'");

        return map;
    }

    private String handleXml(Map<String,Object> stringMap, SignatureData data, String previewUrl) {
        logger.warn("====> 电子签章定时任务开始.createPDF.handleXml.previewUrl：{}", previewUrl);
        String urlHead = configProperty.getUrlHead();

        Document document = null;
        try {
            //获取模板数据
            HnCertCatalog cert = HnCertCatalog.getByAreaName(data.getCertificatefrom());
            if (cert == null) {
                return "";
            }
            //填充模版数据
            String xml = String.format(XML_TEMPLATE, cert.getCatalogId(), cert.getCatalogName(), cert.getTemplateId(), cert.getTemplateName(), cert.getVersion());
            document = DocumentHelper.parseText(xml);
            Element rootElement = document.getRootElement();
            Element surface = rootElement.element("surface");
            List<Element> itemElementList = surface.elements("item");

            for (Element item : itemElementList) {
                String alias = item.attributeValue("alias");
                switch (alias) {
                    case "holderCodeType":
                        setItemValue(item, alias, "0"); // 证件类型 保安服务许可证是001
                        break;
                    case "issueUnitname":
                        setItemValue(item, alias, cert.getIssueUnitname()); // 颁证单位
                        break;
                    case "licenseNumber":
                        setItemValue(item, alias, data.getCertificateno()); // 证照编号
                        break;
                    case "issueDate":
                        setItemValue(item, alias, data.getFzDate()); // 颁证时间
                        break;
                    case "validTimeBegin":
                        setItemValue(item, alias, data.getFzDate()); // 有效期（起始）
                        break;
                    case "holder":
                        setItemValue(item, alias, data.getName()); // 持证者
                        break;
                    case "validTimeEnd":
                        setItemValue(item, alias, "长期"); // 有效期（截止）
                        break;
                    case "holderCode":
                        setItemValue(item, alias, data.getIdcardno()); // 证件号码
                        break;
                    case "csny":
                        setItemValue(item, alias, data.getYear() + "-" + data.getMonth() + "-" + data.getDay()); // 出生年月
                        break;
                    case "zz":
                        setItemValue(item, alias, data.getAddress()); // 住址
                        break;
                    case "zp":
                        String imageUrl = data.getHeadphoto();
                        if (imageUrl == null || imageUrl.isEmpty()) {
                            logger.error("图片URL为空");
                            return "";
                        }
                        imageUrl = imageUrl.startsWith("http") ? imageUrl : urlHead + imageUrl;

                        String filename = "";
                        int lastSlashIndex = imageUrl.lastIndexOf('/');
                        if (lastSlashIndex != -1 && lastSlashIndex < imageUrl.length() - 1) {
                            filename = imageUrl.substring(lastSlashIndex + 1);
                        }
                        item.attribute("fileName").setValue(filename);

                        byte[] imageBytes = HttpUtil.downloadBytes(imageUrl);
                        String base64 = Base64Encoder.encode(imageBytes);
                        setItemValue(item, alias, base64); // 照片
                        break;
                    case "MJXM":
                        setItemValue(item, alias, stringMap.get("name") + ""); // 民警姓名
                        break;
                    case "MJSFZHM":
                        setItemValue(item, alias, stringMap.get("number") + ""); // 民警身份证号码
                        break;
                    case "MJJH":
                        setItemValue(item, alias, ""); // 民警警号
                        break;
                    case "YHID":
                        setItemValue(item, alias, stringMap.get("author") + ""); // 用户Id
                        break;
                    case "EWM":
                        String htmlContent = createHtml(data, cert.getIssueUnitname());
                        setItemValue(item, alias, htmlContent); // 二维码
                        break;
                    default:
                        System.err.println("未知的项别名: " + alias);
                        break;
                }
            }
        } catch (DocumentException e) {
            logger.error("handleXMl发生异常:{}", previewUrl, e);
        }

        return document.asXML();
    }

    private void setItemValue(Element item, String alias, String value) {
        if (item.attributeValue("alias").equals(alias)) {
            item.attribute("value").setValue(value);
        }
    }

    public static String createHtml(SignatureData data, String issueUnitname){
        String toDay = data.getFzDate();
        if(org.apache.commons.lang3.StringUtils.isEmpty(toDay)){
            toDay = DateUtil.today();
        }
        String zDay = toDay.split("-")[2].substring(0,2);
        String gender = "";
        // 获取倒数第二位字符
        char c = data.getIdcardno().charAt(data.getIdcardno().length() - 2);
        int genderBit = Integer.parseInt(String.valueOf(c));
        // 判断奇偶数
        if (genderBit % 2 == 0) {
            gender = "女";
        } else {
            gender = "男";
        }
        String htmlContent =
                "证件编号："+data.getCertificateno()+"\n" +
                "姓   名："+data.getName()+"\n" +
                "性   别："+gender+"\n" +
                "出生年月："+data.getYear()+" 年 "+data.getMonth()+" 月 "+data.getDay()+" 日\n" +
                "身份证号："+data.getIdcardno()+"\n" +
                "住   址："+data.getAddress()+"\n" +
                "发证机关："+issueUnitname+"\n" +
                "发证日期："+toDay.split("-")[0]+" 年 "+toDay.split("-")[1]+" 月 "+zDay+" 日\n";
        return htmlContent;
    }

    //token生成jwt字符串存（计算过期时间）
    private static String jwt_token = "";

    private String getToken(){
        String cgiApi = configProperty.getCgiApi();
        String token = "";
        if(org.apache.commons.lang3.StringUtils.isEmpty(jwt_token)||!JwtUtil.isTokenExpired(jwt_token)){
            token = HuNanECUtil.getToken(cgiApi);
            jwt_token = JwtUtil.createJWT(token);
        }else{
            token = JwtUtil.parseData(jwt_token);
        }

        return token;
    }


    /**
     * 执行签章的逻辑
     * @param queuesDTO
     * @param infile
     */
    private void processQueue(SignatureQueuesDTO queuesDTO, String infile) {
        logger.error("====> 电子签章定时任务开始.signQuicklyExCGI.processQueue.Last_processed_result：id={};lastResult={};infile={}",
                queuesDTO.getId(),
                queuesDTO.getLast_processed_result(),
                infile
        );

        if (org.apache.commons.lang3.StringUtils.isEmpty(queuesDTO.getLast_processed_result())) {
            return;
        }

        String cgiApi = configProperty.getCgiApi();
        String rootPath = configProperty.getRootPath();

        Exception exception = null;
        StringBuilder sb = new StringBuilder();
        String traceId = UUID.randomUUID().toString();

        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        try {
            String fileNumber = queuesDTO.getLast_processed_result().split("证书编码:")[1];
            String filename = System.currentTimeMillis() + "hnzgz_queueId_" + queuesDTO.getId() + ".ofd";
            sb.append(String.format("执行的filename=%s;", filename));

            String path = Paths.get(infile, filename).normalize().toString();
            sb.append(String.format("执行的path=%s;", path));
            String base64Str = HuNanECUtil.getFileRequest(cgiApi, fileNumber);
            if (org.apache.commons.lang3.StringUtils.isEmpty(base64Str)) {
                return;
            }
            //创建目录
            FileUtils.mkdirs(rootPath + infile);
            // 将Base64字符串解码为字节数组 存服务器
            byte[] bytes = cn.hutool.core.codec.Base64.decode(base64Str);
            File file = new File(rootPath + path);
            try (FileOutputStream fos = new FileOutputStream(file)) {
                // 将字节数组写入文件
                fos.write(bytes);
            } catch (IOException e) {
                logger.error("{}-执行文件写入异常:{}", traceId, file, e);
                sb.append(String.format("写入文件时发生错误=%s;", e.toString()));
                return;
            }

            String updateSql = "UPDATE tlk_certificate SET ITEM_SIGNATURED_PROCESSED_STATUS=?, ITEM_LAST_SIGNATURED_PROCESSED_TIME=NOW(), ITEM_CERTIFICATEDATE=IFNULL(ITEM_CERTIFICATEDATE, NOW()), ITEM_LAST_SIGNATURED_PROCESSED_RESULT=?, ITEM_ATTACHMENT=? WHERE ID=?";
            jdbcTemplate.update(updateSql, "2", "完成", path, queuesDTO.getBusiness_id());
            sb.append(String.format("updateSql=%s;", updateSql));
            signatureQueuesRepository.updateSignature(1, new Date(), fileNumber, queuesDTO.getId());
        } catch (Exception ex) {
            exception = ex;
        } finally {
            stopWatch.stop();
            logger.error("{}-{}签章的执行: 跟踪信息={};耗时{}秒", traceId, (exception == null ? "完成" : "异常完成"), sb, exception, stopWatch.getTotalTimeSeconds());
        }
    }
}
