package com.bcxin.tenant.data.etc.tasks.jobs;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.event.core.*;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.exceptions.NoSupportEventException;
import com.bcxin.event.core.jdbc.JdbcNameParameterSqlParser;
import com.bcxin.event.core.jdbc.JdbcNameParameterSqlParserImpl;
import com.bcxin.event.core.jdbc.ParseSqlParameter;
import com.bcxin.event.job.core.domain.documents.enums.DocumentType;
import com.bcxin.event.job.core.domain.dtos.RedisConfig;
import com.bcxin.event.job.core.domain.utils.DocumentTypeUtil;
import com.bcxin.flink.streaming.cores.properties.CheckpointConfigProperty;
import com.bcxin.flink.streaming.cores.properties.StreamingConfigConstants;
import com.bcxin.tenant.data.etc.tasks.components.BinlogRawValue;
import com.bcxin.tenant.data.etc.tasks.components.CustomJdbcOutputFormat;
import com.bcxin.tenant.data.etc.tasks.components.CustomJdbcOutputFormatParameterWrapper;
import com.bcxin.tenant.data.etc.tasks.components.DataSourceUtil;
import com.bcxin.tenant.data.etc.tasks.components.builder.DataBuilderAbstract;
import com.bcxin.tenant.data.etc.tasks.components.impls.CustomJdbcAcceptPreparedStatementParameterImpl;
import com.bcxin.tenant.data.etc.tasks.properties.DataEtcConfigProperty;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.flink.api.common.restartstrategy.RestartStrategies;
import org.apache.flink.api.common.serialization.AbstractDeserializationSchema;
import org.apache.flink.api.common.serialization.DeserializationSchema;
import org.apache.flink.api.java.utils.ParameterTool;
import org.apache.flink.configuration.*;
import org.apache.flink.connector.jdbc.JdbcConnectionOptions;
import org.apache.flink.connector.jdbc.JdbcExecutionOptions;
import org.apache.flink.connector.jdbc.internal.GenericJdbcSinkFunction;
import org.apache.flink.connector.jdbc.internal.JdbcOutputFormat;
import org.apache.flink.connector.jdbc.internal.executor.JdbcBatchStatementExecutor;
import org.apache.flink.connector.kafka.source.reader.deserializer.KafkaRecordDeserializationSchema;
import org.apache.flink.contrib.streaming.state.EmbeddedRocksDBStateBackend;
import org.apache.flink.runtime.taskmanager.TaskManager;
import org.apache.flink.streaming.api.CheckpointingMode;
import org.apache.flink.streaming.api.datastream.KeyedStream;
import org.apache.flink.streaming.api.datastream.SingleOutputStreamOperator;
import org.apache.flink.streaming.api.environment.CheckpointConfig;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.sink.SinkFunction;
import org.apache.flink.streaming.connectors.kafka.internals.KafkaDeserializationSchemaWrapper;
import org.apache.flink.util.Collector;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.common.header.Header;
import org.apache.kafka.common.header.Headers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.PreparedStatementCreatorFactory;
import org.springframework.util.StopWatch;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.sql.*;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.*;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public abstract class DataJobAbstract  extends FlinkJobAbstract {
    private static final Logger logger = LoggerFactory.getLogger(DataEtcJob.class);

    private final Set<String> _calculatedActionIdHash = Collections.synchronizedSet(new HashSet<>());
    private final Collection<DataEtcConfigProperty> configProperties;
    private final String configFile;
    private final boolean isDebug;

    protected DataJobAbstract(Collection<DataEtcConfigProperty> configProperties, String configFile, boolean isDebug) {
        this.configProperties = configProperties;
        this.configFile = configFile;
        this.isDebug = isDebug;
    }

    @Override
    protected void coreExecute() throws Exception {
        if (CollectionUtils.isEmpty(configProperties)) {
            throw new BadEventException("无效配置数据; 无法启动归集功能");
        }
    }

    /**
     * 分组合并Topic信息
     *
     * @return
     */
    protected Collection<DataEtcConfigProperty.TmpMergedKafkaConnectionTopicInfo> getMergedKafkaConnections(DataEtcConfigProperty config) {
        if (config == null || CollectionUtils.isEmpty(config.getTopicSubscribers())) {
            throw new BadEventException("无效主题配置数据; 无法启动归集功能");
        }

        /**
         * 应该通过主题跟算子直接串起来
         */
        return config.getTopicSubscribers().stream().map(subscriberConfigProperty -> {
            Optional<DataEtcConfigProperty.KafkaConnectionConfigProperty>
                    kafkaConnectionConfigOptional = config.getKafkaConnections().stream()
                    .filter(ii -> ii.getName().equalsIgnoreCase(subscriberConfigProperty.getRefKafkaName()))
                    .findFirst();
            if (!kafkaConnectionConfigOptional.isPresent()) {
                throw new BadEventException(String.format("无效refKafkaName配置(%s), 无法找到对应的配置", subscriberConfigProperty.getRefKafkaName()));
            }

            DataEtcConfigProperty.TmpMergedKafkaConnectionTopicInfo mergedKafkaConnectionTopicInfo =
                    DataEtcConfigProperty.TmpMergedKafkaConnectionTopicInfo.create(kafkaConnectionConfigOptional.get());
            mergedKafkaConnectionTopicInfo.addTopic(subscriberConfigProperty.getTopic());

            return mergedKafkaConnectionTopicInfo;
        }).collect(Collectors.toList());
    }

    protected void buildJdbcSubscriber(
            String bootstrapServer,
            SingleOutputStreamOperator<BinlogRawValue> binlogRawValueKeyedStream,
            JdbcExecutionOptions jdbcExecutionOptions,
            DataEtcConfigProperty config,
            Collection<DataEtcConfigProperty.JdbcSubscriberContentConfigProperty> jdbcSubscriberContentConfigProperties,
            String sourceUid) {
        if (CollectionUtils.isEmpty(jdbcSubscriberContentConfigProperties)) {
            throw new BadEventException("无效的目标订阅配置");
        }

        jdbcSubscriberContentConfigProperties
                .stream().map(ix -> ix.getRefJdbcName())
                .distinct().forEach(jdbcName -> {
                    Optional<DataEtcConfigProperty.JdbcConnectionConfigProperty>
                            jdbcConnectionConfigPropertyOptional =
                            config.getJdbcConnections().stream()
                                    .filter(jdbcC -> jdbcC.getName().equalsIgnoreCase(jdbcName))
                                    .findFirst();

                    if (!jdbcConnectionConfigPropertyOptional.isPresent()) {
                        throw new BadEventException("找不到数据源配置");
                    }

                    DataEtcConfigProperty.JdbcConnectionConfigProperty jdbcConnectionConfigProperty
                            = jdbcConnectionConfigPropertyOptional.get();


                    JdbcConnectionOptions jdbcConnectionOptions = getJdbcConnectionOption(jdbcConnectionConfigProperty);
                    if (!CollectionUtils.isEmpty(jdbcSubscriberContentConfigProperties)) {
                        Collection<String> batchSql =
                                jdbcSubscriberContentConfigProperties.stream()
                                        .map(ix -> ix.getContent())
                                        .collect(Collectors.toList());

                        /**
                         * todo: 针对归集添加一个日志表
                         */
                        batchSql.add(
                                String.format(
                                        "INSERT INTO `companyinfocollect`.`collect_logs`(`business_id`, `business_table_name`, `db_partition`, `last_sync_version`, `created_time`, `step`)" +
                                                " VALUES (:%s, :%s,:%s,:%s, CURRENT_TIMESTAMP,'[[@step1@]]')",
                                        BinlogRawValue.FIELD_ID,BinlogRawValue.FIELD_FULL_TABLE,
                                        BinlogRawValue.FIELD_PARTITION,BinlogRawValue.FIELD_LAST_SYNC_VERSION
                                )
                        );
                        Collection<String> batchOrConditionExpress =
                                jdbcSubscriberContentConfigProperties.stream()
                                        .filter(ix -> StringUtils.hasLength(ix.getConditionExpress()))
                                        .map(ix -> ix.getConditionExpress())
                                        .collect(Collectors.toList());

                        DataEtcConfigProperty.JdbcSubscriberContentConfigProperty scp
                                = jdbcSubscriberContentConfigProperties.stream().findFirst().get();
                        {
                            SinkFunction<BinlogRawValue> sinkFunction = null;
                            switch (scp.getType()) {
                                case JDBC:
                                    sinkFunction = sync2JdbcSink(bootstrapServer, batchSql, batchOrConditionExpress, scp, jdbcConnectionOptions, jdbcExecutionOptions);
                                    break;
                                default:
                                    throw new NoSupportEventException("不支持该归集类型");
                            }

                            String uidHash = String.format("%s_%s", sourceUid, scp.getUid());
                            if (!_calculatedActionIdHash.add(uidHash)) {
                                throw new BadEventException("该算子节点已经存储, 请确保同一个文件中的该标题信息是唯一的");
                            }

                            String operatorName = String.format("flink-算子-%s", scp.getTitle());
                            if (operatorName.length() > 50) {
                                operatorName = operatorName.substring(0, 50);
                            }
                            operatorName = operatorName.concat(String.format("总共执行%s个存储过程", batchSql.size()));

                            KeyedStream<BinlogRawValue, String> keyedStream
                                    = binlogRawValueKeyedStream
                                    .keyBy(ii -> {
                                        JsonProvider jsonProvider = new JsonProviderImpl();
                                        String jsonValue = new String(ii.getValue());
                                        JSONObject valueJsonObject = jsonProvider.toObject(JSONObject.class, jsonValue);

                                        JSONObject dataNode = valueJsonObject.getJSONObject("before");
                                        if (dataNode == null) {
                                            dataNode = valueJsonObject.getJSONObject("after");
                                        }

                                        String parallelismKeyValue = FlinkConstants.getExtractedParallelismOriginalKey(dataNode);

                                        String pKey = FlinkConstants.getCalculatedParallelismKey(parallelismKeyValue);

                                        return pKey;
                                    });

                            keyedStream
                                    .addSink(sinkFunction)
                                    .setParallelism(1)
                                    /**
                                     * 设置为kafka的分区数量
                                     */
                                    .uid(String.format("sk:%s", uidHash))
                                    .name(operatorName);
                        }
                    }
                });
    }

    private static Map<String, JdbcConnectionOptions> jdbcConnectionOptionsMap = new ConcurrentHashMap<>();

    protected JdbcConnectionOptions getJdbcConnectionOption(DataEtcConfigProperty.JdbcConnectionConfigProperty jdbcConnectionConfigProperty) {
        if (jdbcConnectionOptionsMap.containsKey(jdbcConnectionConfigProperty.getUrl())) {
            return jdbcConnectionOptionsMap.get(jdbcConnectionConfigProperty.getUrl());
        }

        JdbcConnectionOptions jdbcConnectionOptions
                = new JdbcConnectionOptions.JdbcConnectionOptionsBuilder()
                .withUrl(jdbcConnectionConfigProperty.getUrl())
                .withDriverName(jdbcConnectionConfigProperty.getDriverClassName())
                .withUsername(jdbcConnectionConfigProperty.getUserName())
                .withPassword(jdbcConnectionConfigProperty.getPassword())
                .withConnectionCheckTimeoutSeconds(10)
                .build();
        jdbcConnectionOptionsMap.put(jdbcConnectionConfigProperty.getUrl(), jdbcConnectionOptions);

        return jdbcConnectionOptionsMap.get(jdbcConnectionConfigProperty.getUrl());
    }

    protected SinkFunction<BinlogRawValue> sync2JdbcSink(
            String bootstrapServer,
            Collection<String> batchSql,
            Collection<String> batchOrConditionExpress,
            DataEtcConfigProperty.JdbcSubscriberContentConfigProperty scp,
            JdbcConnectionOptions jdbcConnectionOptions,
            JdbcExecutionOptions jdbcExecutionOptions
    ) {

        String batchSqlContent = batchSql.stream()
                .filter(ix -> StringUtils.hasLength(ix))
                .collect(Collectors.joining(";"));

        JdbcNameParameterSqlParser nameParameterSqlParser = new JdbcNameParameterSqlParserImpl();
        ParseSqlParameter parse2 = nameParameterSqlParser.parse(batchSqlContent);
        List<int[]> parameterIndexes = parse2.getParameterIndexes();
        List<String> parameterNames = parse2.getParameterNames();
        PreparedStatementCreatorFactory pscf = parse2.getPscf();

        String sql = pscf.getSql();
        RedisConfig redisConfig = RedisConfig.getDefaultFromMainThread();

        CustomJdbcOutputFormatParameterWrapper jdbcOutputFormatParameterWrapper =
                CustomJdbcOutputFormatParameterWrapper.create(
                        parameterNames,
                        jdbcConnectionOptions,
                        parameterIndexes,
                        sql
                );
        JdbcOutputFormat.StatementExecutorFactory statementExecutorFactory =
                (context) -> JdbcBatchStatementExecutor.simple(
                        jdbcOutputFormatParameterWrapper.getSql(),
                        (preparedStatement, ik) -> {

                        }, Function.identity());

        SinkFunction<BinlogRawValue> sinkFunction
                = new GenericJdbcSinkFunction<BinlogRawValue>(
                new CustomJdbcOutputFormat(
                        batchOrConditionExpress,
                        jdbcConnectionOptions,
                        jdbcExecutionOptions,
                        statementExecutorFactory,
                        sql, bootstrapServer, redisConfig,
                        jdbcOutputFormatParameterWrapper,
                        new CustomJdbcAcceptPreparedStatementParameterImpl(),
                        JdbcOutputFormat.RecordExtractor.identity())
        );

        return sinkFunction;
    }

    protected StreamExecutionEnvironment getStreamExecutionEnvironment(CheckpointConfigProperty configProperty) {
        StreamExecutionEnvironment env = null;
        /**
         * 不可以使用该false方法内部的逻辑来创建savepoint; 否则会出现阶段错误, 而导致无法执行
         */
        if (false) {

            Configuration configuration = new Configuration();
            if (!org.apache.commons.lang3.StringUtils.isEmpty(configProperty.getSavepointPath())) {
                logger.info("savepoint: 系统将从如下位置加载上次的执行点:{}", configProperty.getSavepointPath());
                configuration.setString("execution.savepoint.path", configProperty.getSavepointPath());
            } else {
                logger.info("savepoint: 系统没找到对于的savepoint; 因此可能会重新加载");
            }

            logger.info("getStreamExecutionEnvironment:获取得到的当前checkpoint的路径={};savepointPath={};原始值={}",
                    configProperty.getCheckpointPath(),
                    configProperty.getSavepointPath(),
                    System.getProperty(StreamingConfigConstants.DISK_CHECKPOINT_LOCATION)
            );

            if (isDebug) {
                configuration.set(TaskManagerOptions.NETWORK_MEMORY_MIN, MemorySize.parse("300m"));
                env = StreamExecutionEnvironment.getExecutionEnvironment(configuration);
                env.setParallelism(5);
            } else {
                env = StreamExecutionEnvironment.getExecutionEnvironment(configuration);
            }
            //com.sun.jmx.mbeanserver.Repository
            CheckpointConfig checkpointConfig = env.getCheckpointConfig();


            env.setStateBackend(new EmbeddedRocksDBStateBackend());
            checkpointConfig.setCheckpointStorage(configProperty.getCheckpointPath());
            //checkpointConfig.setExternalizedCheckpointCleanup(CheckpointConfig.ExternalizedCheckpointCleanup.DELETE_ON_CANCELLATION);
            checkpointConfig.setTolerableCheckpointFailureNumber(8);
            /**
             * 配置这个的话; 那么在取消job的时候; 系统会自动保存savepoint
             */
            configuration.setString("state.savepoints.dir", configProperty.getCheckpointPath());
            checkpointConfig.setCheckpointingMode(CheckpointingMode.EXACTLY_ONCE);

            env.setRestartStrategy(RestartStrategies.fallBackRestart());
            /**
             * 10秒一个checkpoint
             * 听说启用checkpoint之后; kafka中的offset设置是无用的, 只有没启用的时候才有用
             */
            env.enableCheckpointing(10_000, CheckpointingMode.EXACTLY_ONCE);
        }

        Configuration configuration = new Configuration();
        configuration.set(BlobServerOptions.CLEANUP_INTERVAL, 120l);
        ParameterTool parameterTool = configProperty.getArgParameter();
        if (parameterTool != null) {
            List<ConfigOption> configOptions = Stream.of(
                    JobManagerOptions.PORT,
                    JobManagerOptions.TOTAL_PROCESS_MEMORY,
                    TaskManagerOptions.TOTAL_PROCESS_MEMORY,
                    TaskManagerOptions.NUM_TASK_SLOTS,
                    TaskManagerOptions.TASK_OFF_HEAP_MEMORY,
                    TaskManagerOptions.TASK_HEAP_MEMORY,
                    TaskManagerOptions.JVM_METASPACE,
                    TaskManagerOptions.MANAGED_MEMORY_SIZE,
                    TaskManagerOptions.MANAGED_MEMORY_SIZE,
                    RestOptions.PORT
            ).collect(Collectors.toList());

            for (ConfigOption selectedConfigOption : configOptions) {
                String optionValue = parameterTool.get(selectedConfigOption.key());
                if (StringUtils.hasLength(optionValue)) {
                    configuration.set(selectedConfigOption, Integer.parseInt(optionValue));
                }
            }
        }

        env = StreamExecutionEnvironment.getExecutionEnvironment();
        env.setStateBackend(new EmbeddedRocksDBStateBackend());
        env.getCheckpointConfig().setCheckpointStorage(configProperty.getCheckpointPath());
        logger.error("etc.checkpoint的PointStorage位置={};", configProperty.getCheckpointPath());

        env.enableCheckpointing(10 * 60 * 1000);
        env.getCheckpointConfig().setCheckpointingMode(CheckpointingMode.EXACTLY_ONCE);
        env.getCheckpointConfig().setMinPauseBetweenCheckpoints(2000);
        env.getCheckpointConfig().setCheckpointTimeout(160_0000);
        env.getCheckpointConfig().setMaxConcurrentCheckpoints(1);
        env.getCheckpointConfig().enableExternalizedCheckpoints(
                CheckpointConfig.ExternalizedCheckpointCleanup.DELETE_ON_CANCELLATION);
        /**
         * 60秒
         */
        env.getConfig().setAutoWatermarkInterval(60 * 1000);

        return env;
    }

    protected KafkaRecordDeserializationSchema<BinlogRawValue> getDeserializationSchema() {
        KafkaDeserializationSchemaWrapper<BinlogRawValue> binlogRawValueKafkaDeserializationSchemaWrapper
                = new KafkaDeserializationSchemaWrapper(new AbstractDeserializationSchema<BinlogRawValue>() {
            @Override
            public BinlogRawValue deserialize(byte[] message) throws IOException {
                return BinlogRawValue.create(message, new JsonProviderImpl());
            }
        }) {
            private Map<String,Long> topicOffsetMap;
            @Override
            public void open(DeserializationSchema.InitializationContext context) throws Exception {
                super.open(context);
                topicOffsetMap = new HashMap<>();
            }

            /*
            @Override
            public void deserialize(ConsumerRecord message, Collector out) throws Exception {
                Headers headers = message.headers();
                Header regionHeader = headers.lastHeader("region_code");
                if(regionHeader == null || regionHeader.value() == null || !(new String(regionHeader.value()).startsWith(getRegionPrefix()))) {
                    //logger.error("当前区域({}; 头部区域信息:{})原本要忽略的执行任务:{}", getRegionPrefix(), regionHeader == null ? "EMPTY" : regionHeader.value(), message.value());
                    //return;
                }

                super.deserialize(message, out);
            }

             */
        };


        KafkaRecordDeserializationSchema<BinlogRawValue> kafkaRecordDeserializationSchema =
                KafkaRecordDeserializationSchema.of(binlogRawValueKafkaDeserializationSchemaWrapper);

        return kafkaRecordDeserializationSchema;
    }
}