package com.bcxin.tenant.data.etc.tasks.components;

import com.bcxin.event.core.exceptions.BadEventException;
import com.zaxxer.hikari.HikariDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.sql.SQLException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class DataSourceUtil {
    private static Logger logger = LoggerFactory.getLogger(DataSourceUtil.class);
    private static volatile Map<String, DataSource> dataSourceMapContainer = new ConcurrentHashMap<>();

    /**
     * https://blog.csdn.net/qq_44962429/article/details/106406954
     * @param driveClassName
     * @param url
     * @param username
     * @param password
     * @return
     */
    public static DataSource getDataSource(String driveClassName,
                                           String url,String username,String password) {
        String dbKey = String.format("%s-%s-%s", driveClassName, url, username);
        if (dataSourceMapContainer.containsKey(dbKey)) {
            return dataSourceMapContainer.get(dbKey);
        }

        synchronized (DataSourceUtil.class) {
            if (dataSourceMapContainer.containsKey(dbKey)) {
                return dataSourceMapContainer.get(dbKey);
            }

            try {
                HikariDataSource dataSource = new HikariDataSource();
                dataSource.setDriverClassName(driveClassName);
                if (url.contains("rewriteBatchedStatements")) {
                    dataSource.setJdbcUrl(url);
                } else {
                    /**
                     * https://blog.csdn.net/lgh1117/article/details/80212924
                     */
                    dataSource.setJdbcUrl(String.format("%s&rewriteBatchedStatements=true", url));
                }

                url=String.format("%s&sessionVariables=sql_mode='STRICT_ALL_TABLES'&testWhileIdle=true", url);
                dataSource.setUsername(username);
                dataSource.setPassword(password);
                dataSource.setAutoCommit(false);
                dataSource.setConnectionTimeout(260_000);
                dataSource.setIdleTimeout(20_000);
                dataSource.addDataSourceProperty("autoReconnect", true);
                /**
                 * event-hikari - Failed to validate connection com.mysql.cj.jdbc
                 * 当idleTimeout或maxLifetime超过interactive_timeout或者wait_timeout的时候，连接池里的连接没有过期
                 * 但是数据库已经过期了，就会出现如上的错误
                 * interactive_timeout或者wait_timeout 可以通过show variables like '%timeout%'来查询
                 */
                dataSource.setMaxLifetime(1800 * 1000);
                dataSource.setMaximumPoolSize(2000);
                dataSource.setMinimumIdle(10);

                dataSource.setPoolName("event-hikari");

                dataSourceMapContainer.put(dbKey, dataSource);
                logger.warn("V3-创建DataSource:dbKey={},ds.hash={}", dbKey, dataSource.hashCode());
                return dataSource;
            } catch (Exception ex) {
                logger.error("初始化DruidDataSource({})数据源发生异常:{}", url, ex);
                throw new BadEventException(ex);
            }
        }
    }
}
