package com.bcxin.tenant.data.etc.tasks.components;
import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.exceptions.RetryEventException;
import com.bcxin.event.core.exceptions.SkipRetryEventException;
import com.bcxin.event.core.utils.RetryUtil;
import com.bcxin.event.job.core.domain.CacheProvider;
import com.bcxin.event.job.core.domain.dtos.RedisConfig;
import com.bcxin.event.job.core.domain.impls.CacheProviderImpl;
import com.bcxin.event.core.exceptions.AffectedEventException;
import com.bcxin.event.core.KafkaConstants;
import com.bcxin.tenant.data.etc.tasks.dtos.DtqRecordDto;
import com.mysql.cj.jdbc.ClientPreparedStatement;
import com.zaxxer.hikari.pool.HikariProxyPreparedStatement;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.connector.jdbc.JdbcConnectionOptions;
import org.apache.flink.connector.jdbc.JdbcExecutionOptions;
import org.apache.flink.connector.jdbc.internal.JdbcOutputFormat;
import org.apache.flink.connector.jdbc.internal.connection.SimpleJdbcConnectionProvider;
import org.apache.flink.util.concurrent.ExecutorThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import scala.collection.mutable.StringBuilder;

import javax.annotation.Nonnull;
import javax.sql.DataSource;
import java.io.IOException;
import java.io.Serializable;
import java.sql.*;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

public class CustomJdbcOutputFormat extends JdbcOutputFormat implements Serializable,Runnable {
    private static final Logger logger = LoggerFactory.getLogger(CustomJdbcOutputFormat.class);

    private final JdbcConnectionOptions jdbcConnectionOptions;
    /**
     * 通过Open的时候来初始化DataSource
     */
    private volatile DataSource dataSource;
    private final List<Object> batch;
    private final CustomJdbcAcceptPreparedStatementParameter acceptPreparedStatementParameter;
    private final String sql;
    private final JdbcExecutionOptions executionOptions;
    private transient ScheduledExecutorService scheduler;
    private final String bootstrapServer;
    private final RedisConfig redisConfig;

    private final Collection<String> batchOrConditionExpress;

    private transient ScheduledFuture<?> scheduledFuture;
    private transient KafkaOutputFormat kafkaOutputFormat;
    private final CustomJdbcOutputFormatParameterWrapper parameterWrapper;


    public CustomJdbcOutputFormat(
            Collection<String> batchOrConditionExpress,
            @Nonnull JdbcConnectionOptions jdbcConnectionOptions,
                                  @Nonnull JdbcExecutionOptions executionOptions,
                                  @Nonnull StatementExecutorFactory statementExecutorFactory,
                                  @Nonnull String sql,
                                  @Nonnull String bootstrapServer,
                                  RedisConfig redisConfig,
                                  CustomJdbcOutputFormatParameterWrapper parameterWrapper,
                                  CustomJdbcAcceptPreparedStatementParameter acceptPreparedStatementParameter,
                                  @Nonnull RecordExtractor recordExtractor) {
        super(new SimpleJdbcConnectionProvider(jdbcConnectionOptions), executionOptions, statementExecutorFactory, recordExtractor);
        this.batchOrConditionExpress = batchOrConditionExpress;
        this.jdbcConnectionOptions = jdbcConnectionOptions;
        this.batch = new ArrayList<>();
        this.sql = sql;
        this.acceptPreparedStatementParameter = acceptPreparedStatementParameter;
        this.parameterWrapper = parameterWrapper;
        this.executionOptions = executionOptions;
        this.bootstrapServer = bootstrapServer;
        this.redisConfig = redisConfig;

        Runtime.getRuntime().addShutdownHook(new Thread(this));
    }

    @Override
    public void configure(Configuration parameters) {
        super.configure(parameters);
    }


    private transient CacheProvider cacheProvider;
    @Override
    public void open(int taskNumber, int numTasks) throws IOException {
        //super.open(taskNumber, numTasks);
        this.dataSource = DataSourceUtil.getDataSource(jdbcConnectionOptions.getDriverName(),
                jdbcConnectionOptions.getDbURL(),
                jdbcConnectionOptions.getUsername().get(),
                jdbcConnectionOptions.getPassword().get()
        );

        this.kafkaOutputFormat = new KafkaOutputFormat(
                this.bootstrapServer,
                KafkaConstants.getDtqTopic(KafkaConstants.DTQ_JDBC_CONSUMER_TOPIC),
                100, 5_000);

        this.kafkaOutputFormat.open(taskNumber, numTasks);
        if (executionOptions.getBatchIntervalMs() != 0 && executionOptions.getBatchSize() != 1) {
            this.scheduler =
                    Executors.newScheduledThreadPool(
                            1, new ExecutorThreadFactory("jdbc-upsert-output-format"));
            this.scheduledFuture =
                    this.scheduler.scheduleWithFixedDelay(
                            () -> {
                                synchronized (CustomJdbcOutputFormat.class) {
                                    try {
                                        flush();
                                    } catch (Exception e) {
                                        logger.error("scheduleWithFixedDelay.flush发生异常", e);
                                    }
                                }
                            },
                            executionOptions.getBatchIntervalMs(),
                            executionOptions.getBatchIntervalMs(),
                            TimeUnit.MILLISECONDS);
        }

        cacheProvider = new CacheProviderImpl(redisConfig);
    }

    @Override
    public synchronized void flush() throws IOException {
        super.flush();
    }

    @Override
    public synchronized void close() {
        if (this.scheduledFuture != null) {
            this.scheduledFuture.cancel(true);
        }

        if (this.cacheProvider != null) {
            this.cacheProvider.close();
        }
    }

    @Override
    public Connection getConnection() {
        //return super.getConnection();
        return null;
    }

    @Override
    public void updateExecutor(boolean reconnect) throws SQLException, ClassNotFoundException {
        /*
        super.updateExecutor(reconnect);
         */
    }


    /**
     * 该方法是线程安全的
     * @param original
     * @param extracted
     * @throws SQLException
     */
    @Override
    protected void addToBatch(Object original, Object extracted) throws SQLException {
        synchronized (this.batch) {
            this.batch.add(extracted);
        }
    }

    @Override
    protected void attemptFlush() throws SQLException {
        if (!batch.isEmpty()) {
            StringBuilder cost = new StringBuilder();
            int expectedSize = batch.size();
            int actualSize = expectedSize;
            cost.append(String.format("v3-当前batch的数据量=%s;", expectedSize));
            StopWatch stopWatch = new StopWatch();
            Collection<BinlogRawValue> matchRawValues = new ArrayList<>();
            JsonProvider jsonProvider = new JsonProviderImpl();
            String lasExpression = null;
            Collection<BinlogRawValue> notMatchRawValues = new ArrayList<>();
            /**
             * 锁住, 避免多线程后续清除的问题
             */
            synchronized (this.batch) {
                actualSize = batch.size();
                for (Object r : batch) {
                    BinlogRawValue rawValue = (BinlogRawValue) r;
                    boolean isMatchExecuteExpress = rawValue.isMatchCondition(jsonProvider, batchOrConditionExpress);
                    if (isMatchExecuteExpress) {
                        matchRawValues.add(rawValue);
                    } else {
                        lasExpression = rawValue.getConditionExecuteResult();
                        /**
                         * todo: 也加入到队列中
                         */
                        matchRawValues.add(rawValue);

                        /**
                         * 跟踪信息
                         */
                        notMatchRawValues.add(rawValue);
                    }
                }
                batch.clear();
            }

            String tranId = UUID.randomUUID().toString();
            stopWatch.start();
            try {
                logger.error(
                        "{}-开始JDBC操作[(matchRawValues.size={}, batchSize={}), ({}) actualSize={}, expectedSize={}], [nomatch={},ids=[{}]]",
                        tranId,
                        matchRawValues.size(),
                        batch.size(),
                        (actualSize != expectedSize || batch.size() > 0) ? "预警数据节点" : "正常",
                        actualSize,
                        expectedSize,
                        notMatchRawValues.stream().map(ii -> ii.getFullTable()).distinct().collect(Collectors.joining(",")),
                        notMatchRawValues.stream().map(ii -> ii.getId()).distinct().collect(Collectors.joining(","))
                );

                if (!matchRawValues.isEmpty()) {
                    try {
                        RetryUtil.execute(() -> {
                            try (Connection connection = this.dataSource.getConnection()) {
                                connection.setAutoCommit(false);
                                try {
                                    //connection.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
                                    /**
                                     * setCommit的话,会造成.
                                     * Deadlock found when trying to get lock; try restarting transaction
                                     * 正在错误信息
                                     */
                                    String trace = executeSql(tranId, connection, matchRawValues);
                                    cost.append(trace);

                                    connection.commit();
                                } catch (BatchUpdateException ex) {
                                    connection.rollback();
                                    throw new BadEventException(String.format("执行脚本发生异常:%s", ex.getUpdateCounts()), ex);
                                } catch (Exception ex) {
                                    connection.rollback();
                                    throw ex;
                                }
                            }

                            return true;
                        }, 5);
                    }
                    catch (AffectedEventException ex) {

                        RetryUtil.execute(() -> {
                            String retrySql = ex.getParseSql().replace("'[[@step1@]]'","'[[@step2@]]'");
                            try (Connection connection = this.dataSource.getConnection()) {
                                int affectedCount = 0;

                                connection.setAutoCommit(false);
                                try {
                                    try (PreparedStatement statement = connection.prepareStatement(retrySql)) {
                                        affectedCount = statement.executeUpdate();
                                    }
                                    connection.commit();
                                } catch (Exception ee) {
                                    connection.rollback();

                                    throw ee;
                                } finally {
                                    logger.error("{}-[{}]v2.重试之后的逻辑:affectedCount={};dbTable={},ids={},sql={}",
                                            tranId,
                                            affectedCount > 0 ? "Success" : "dbCheck",
                                            affectedCount, ex.getDbTable(), ex.getIds(), retrySql);
                                }

                                /*
                                if (affectedCount <= 0) {
                                    throw ex;
                                }
                                 */
                            }

                            return true;
                        }, 2);
                    }
                } else {
                    cost.append(String.format("总共有%s/%s条数据不符合condition条件[lasExpression=%s];", batch.size() - matchRawValues.size(), batch.size(), lasExpression));
                }
            } catch (Exception ex) {
                logger.error("{}-v2.事务执行发生异常", tranId, ex);

                try {
                    String businessIds = matchRawValues.stream().map(ii -> String.format("'%s'", ii.getId()))
                            .distinct()
                            .collect(Collectors.joining(","));
                    String dbTableName = matchRawValues.stream()
                            .map(ii -> String.format("%s#%s", ii.getFullTable(), ii.getPartition())).distinct()
                            .collect(Collectors.joining(","));

                    logError2Db(String.format("%s-事务执行发生异常(%s)", tranId, this.sql), matchRawValues.size(), ex, businessIds, dbTableName);
                } catch (Exception ee) {
                    logger.error("执行logError2Db发生异常", ee);
                }
            } finally {
                try {
                    stopWatch.stop();
                    cost.append(String.format("总共耗时=%s ms", stopWatch.getTotalTimeMillis()));
                } finally {
                    logger.info(cost.toString());
                }
            }
        }
    }

    private String executeSql(
            String tranId,
            Connection connection,
            Collection<BinlogRawValue> valueWrappers
    ) throws SQLException {
        boolean isExecutable = false;
        StopWatch topWatch = new StopWatch();
        topWatch.start();
        StringBuilder sb = new StringBuilder();
        String affectedCountDesc = null;

        StringBuilder batchPlainSql = new StringBuilder();
        try (PreparedStatement statement = connection.prepareStatement(this.sql)) {
            StopWatch topStopWatch = new StopWatch();
            topStopWatch.start("准备参数信息");
            sb.append(String.format("执行的表达式:%s;",
                    (CollectionUtils.isEmpty(batchOrConditionExpress) ? "NULL" :
                            batchOrConditionExpress.stream().collect(Collectors.joining(";"))))
            );

            StringBuilder paramSb = new StringBuilder();
            for (BinlogRawValue o : valueWrappers) {
                try {
                    StopWatch stopWatch = new StopWatch();
                    stopWatch.start("截取对象的值信息");
                    acceptPreparedStatementParameter.accept(statement, o, cacheProvider,this.parameterWrapper);
                    stopWatch.stop();

                    batchPlainSql.append(String.format("%s;",extractPlainSql(statement)));

                    statement.addBatch();
                    isExecutable = true;
                    paramSb.append(String.format("%s,", o.getReadyPkId()));
                } catch (Exception ex) {
                    try {
                        String dbName = (String) o.getReadyParameter("source.db");
                        String tableName = (String) o.getReadyParameter("source.table");
                        String id = String.valueOf(o.getReadyPkId());
                        logger.error("跟踪：数据id={}，表名={},添加失败", id, tableName);
                        this.kafkaOutputFormat.writeRecord(DtqRecordDto.create(dbName, tableName, id, new String(o.getValue())));
                        sb.append(String.format("可忽略: 跳过异常并加入死信队列: 无效的jdbc参数信息; 导致该数据无法正常更新到db:%s", ex.toString()));
                    } catch (Exception ee) {
                        sb.append(String.format("消息(%s)推送到死信队列发生失败:%s", new String(o.getValue()), ee.toString()));
                    }
                }
            }
            topStopWatch.stop();
            sb.append(String.format("参数(%s)耗时:%s ms", paramSb, topStopWatch.getTotalTimeMillis()));

            if (isExecutable) {
                boolean isSuccess = true;
                try {
                    try {
                        StopWatch execStopWatch = new StopWatch();
                        execStopWatch.start("开始执行存储过程");
                        long[] result = statement.executeLargeBatch();
                        isSuccess = true;

                        execStopWatch.stop();

                        sb.append(String.format("(执行完毕[数量=%s]:)", valueWrappers.size()));
                        sb.append(String.format("存储过程总共耗时=%s ms", execStopWatch.getTotalTimeMillis()));

                        if (result != null) {
                            String affectedCount = Arrays.stream(result)
                                    .mapToObj(String::valueOf).collect(Collectors.joining(","));
                            sb.append(String.format("其中总共有%s条受影响;", affectedCount));
                            affectedCountDesc = String.format("affected=%s;", affectedCount);

                            boolean notAffected = Arrays.stream(result).allMatch(ii -> ii < 1);
                            /**
                             * 我添加了一个INSERT INTO `companyinfocollect`.`collect_logs`
                             * 因此; 至少有一条的影响行数是大于0
                             */
                            if (notAffected) {
                                /**
                                 * 格式为：
                                 * HikariProxyPreparedStatement@1003372882 wrapping com.mysql.cj.jdbc.ClientPreparedStatement: call companyinfocollect.proc_sync_rd_employee_info_collect_by_user_v2('__Zx68MVR5JudbR7VTnfu','旁真一','17700000644a',0,NULL,NULL,'1990-12-10 00:00:00',NULL,NULL,1,0,'2021-08-30 17:54:51','xx',1,'2023-05-27 11:06:56',NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,'__Zx68MVR5JudbR7VTnfu','旁真一','17700000644a',0,NULL,NULL,'1990-12-10 00:00:00',NULL,NULL,1,0,'2021-08-30 17:54:51','xx',1,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL);call companyinfocollect.proc_sync_attendance_PRINCIPAL_update_by_tenant_users('__Zx68MVR5JudbR7VTnfu','旁真一','17700000644a',1);call companyinfocollect.proc_sync_attendance_REALITY_update_by_tenant_users('__Zx68MVR5JudbR7VTnfu','旁真一','旁真一','17700000644a','17700000644a');call companyinfocollect.proc_sync_tlk_attendance_site_person_info_by_users('__Zx68MVR5JudbR7VTnfu',1,0,'17700000644a',1,0,'17700000644a');INSERT INTO `companyinfocollect`.`collect_logs`(`business_id`, `business_table_name`, `db_partition`, `last_sync_version`, `created_time`) VALUES ('__Zx68MVR5JudbR7VTnfu', 'obpm2.tenant_users',0,'2023-05-26 08:00:00', CURRENT_TIMESTAMP)
                                 */
                                String processSql = batchPlainSql.toString();
                                Boolean hasProcessed = false;
                                if(true) {
                                    throw new AffectedEventException(processSql,
                                            valueWrappers.stream().map(ii -> ii.getFullTable()).distinct()
                                                    .collect(Collectors.joining(",")),
                                            valueWrappers.stream().map(ii -> ii.getId())
                                                    .collect(Collectors.joining(",")
                                                    ));
                                }

                                if (!hasProcessed) {
                                    throw new RetryEventException(
                                            String.format("【致命问题】%s-影响结果(%s); [statement=%s];(sql=%s), fullTable=%s; ids=%s;",
                                                    tranId,
                                                    affectedCountDesc,
                                                    statement.toString(),
                                                    this.sql,
                                                    valueWrappers.stream().map(ii -> ii.getFullTable()).distinct()
                                                            .collect(Collectors.joining(",")),
                                                    valueWrappers.stream().map(ii -> ii.getId())
                                                            .collect(Collectors.joining(","))
                                            ));
                                }
                            }
                        } else {
                            affectedCountDesc = "affected=NULL;";

                            throw new BadEventException(String.format("不符合预期(%s)", affectedCountDesc));
                        }
                    } catch (RetryEventException  ex) {
                        isSuccess = false;
                        throw ex;
                    }  catch (AffectedEventException  ex) {
                        isSuccess = false;
                        throw ex;
                    }catch (Exception ex) {
                        isSuccess = false;
                        if (ExceptionUtils.getStackTrace(ex).contains("Incorrect")) {
                            throw new SkipRetryEventException(String.format("%s-【不进行重试-v2】执行(%s)数据库发生异常(参数异常)", tranId, this.sql), ex);
                        }
                        throw new BadEventException(
                                String.format("%s-影响结果(%s); fullTable=%s;ids=%s;",
                                        tranId,
                                        affectedCountDesc,
                                        valueWrappers.stream().map(ii -> ii.getFullTable()).distinct()
                                                .collect(Collectors.joining(",")),
                                        valueWrappers.stream().map(ii -> ii.getId())
                                                .collect(Collectors.joining(","))
                                ), ex);
                    }
                } finally {
                    topWatch.stop();
                    if (isSuccess) {
                        logger.error(
                                "{}[{}秒]-[{}] -{} - {} id=[{}]",
                                (isSuccess ? "Success" : "Failed"),
                                topWatch.getTotalTimeSeconds(),
                                tranId,
                                affectedCountDesc,
                                valueWrappers.stream().map(ii -> ii.getFullTable()).distinct()
                                        .collect(Collectors.joining(",")),
                                valueWrappers.stream().map(ii -> ii.getId())
                                        .collect(Collectors.joining(","))
                        );
                    }
                }
            }
        }

        return sb.toString();
    }

    @Override
    public void run() {
        boolean flag = false;
        StringBuilder sb = new StringBuilder();
        try {
            if (batch == null) {
                sb.append("暂无数据");
            } else {
                sb.append(String.format("批量数据=%s",
                        batch.stream().filter(ii -> ii != null).map(ii -> String.valueOf(ii))
                                .collect(Collectors.joining(";"))));
            }

            this.flush();

            flag = true;
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            logger.error("系统出现非预期的行为导致异常: 可能丢失的数据为:{}, 数据大小={}", flag, sb);
        }
    }

    private void logError2Db(String message,
                             int count,
                             Exception exception,
                             String businessIds,
                             String dbTables) {
        try (Connection connection = this.dataSource.getConnection()) {
            connection.setAutoCommit(false);
            try {
                connection.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
                /**
                 * setCommit的话,会造成Deadlock found when trying to get lock; try restarting transaction
                 * 正在错误信息
                 */
                try (PreparedStatement statement = connection.prepareStatement(
                        "INSERT INTO `companyinfocollect`.`collect_log_errors`(`business_id`,`count_of_error`, `status`,`business_table_name`, `message`, `exception`, `created_time`)" +
                                " VALUES (?,?,0, ?, ?, ?, CURRENT_TIMESTAMP)"
                )) {
                    statement.setObject(1, businessIds);
                    statement.setObject(2, count);
                    statement.setObject(3, dbTables);
                    statement.setObject(4, message);

                    String error = String.format("%s-%s", exception.getMessage(), ExceptionUtils.getStackTrace(exception));
                    statement.setObject(5,
                            error
                    );

                    statement.execute();
                }

                connection.commit();
            } catch (Exception ex) {
                connection.rollback();
            }
        } catch (Exception ex) {
            logger.error("提交异常日志到数据库发生异常:{}", message, ex);
        }
    }


    private String extractPlainSql(Statement statement) {
        String processSql = statement.toString();
        String sqlKeyword = "com.mysql.cj.jdbc.ClientPreparedStatement:";
        if (processSql.contains(sqlKeyword)) {
            int index = processSql.indexOf(sqlKeyword);
            processSql = processSql.substring(index + sqlKeyword.length()).trim();
        }

        return processSql;
    }
}
