package com.bcxin.flink.cdc.kafka.source.task.compnents;

import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.job.core.domain.BeanFactory;
import com.bcxin.event.job.core.domain.CacheProvider;
import com.bcxin.event.job.core.domain.JedisPoolFactory;
import com.bcxin.event.job.core.domain.dtos.RedisConfig;
import com.bcxin.flink.cdc.kafka.source.task.StartupOptionUtil;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.BinlogCdcValue;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.BinlogOffsetValue;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.flink.api.common.state.MapState;
import org.apache.flink.api.common.state.MapStateDescriptor;
import org.apache.flink.api.common.state.ValueState;
import org.apache.flink.api.common.state.ValueStateDescriptor;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.runtime.state.FunctionInitializationContext;
import org.apache.flink.runtime.state.FunctionSnapshotContext;
import org.apache.flink.streaming.api.checkpoint.CheckpointedFunction;
import org.apache.flink.streaming.api.functions.KeyedProcessFunction;
import org.apache.flink.util.Collector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;
import redis.clients.jedis.Jedis;

import java.text.SimpleDateFormat;
import java.util.*;

public class BinlogCheckpointProcessFunction extends KeyedProcessFunction<String, BinlogCdcValue,BinlogCdcValue> implements CheckpointedFunction {
    private static final Logger logger = LoggerFactory.getLogger(BinlogCheckpointProcessFunction.class);

    private transient ValueState<BinlogOffsetValue> binlogOffsetValueValueState;

    private transient MapState<String, Date> lastMapKeySyncDateState;
    private transient boolean hasValueChangedAfterSnapshot = false;
    private transient Jedis jedis;

    private final RedisConfig redisConfig;
    private final BinlogOffsetValue binlogOffsetValue;


    public BinlogCheckpointProcessFunction(RedisConfig redisConfig, BinlogOffsetValue binlogOffsetValue) {
        this.redisConfig = redisConfig;
        this.binlogOffsetValue = binlogOffsetValue;
    }


    @Override
    public void open(Configuration parameters) throws Exception {
        super.open(parameters);
        try {
            ValueStateDescriptor<BinlogOffsetValue> binlogOffsetValueValueStateDescriptor = new ValueStateDescriptor<BinlogOffsetValue>(
                    "mysql-binlog-offset-instance",
                    BinlogOffsetValue.class);
            binlogOffsetValueValueState = getRuntimeContext().getState(binlogOffsetValueValueStateDescriptor);
        } catch (Exception ex) {
            ex.printStackTrace();
            throw new BadEventException("程序处理出现异常", ex);
        }


        try {
            MapStateDescriptor<String, Date> lastFullTableSyncDateStateDescriptor = new MapStateDescriptor<String, Date>(
                    "mysql-binlog-mp-full-table-sync-date-state",
                    String.class,
                    Date.class
            );

            this.lastMapKeySyncDateState = getRuntimeContext().getMapState(lastFullTableSyncDateStateDescriptor);
        } catch (Exception ex) {
            ex.printStackTrace();
            throw new BadEventException("FullTableMap状态初始化异常", ex);
        }
    }

    @Override
    public void close() throws Exception {
        super.close();

        if (this.jedis != null) {
            this.jedis.close();
        }
    }

    @Override
    public void snapshotState(FunctionSnapshotContext context) throws Exception {
        //todo
        //将数据序列化到redis
        if (this.binlogOffsetValueValueState != null) {
            if (hasValueChangedAfterSnapshot) {
                BinlogOffsetValue offsetValue = this.binlogOffsetValueValueState.value();
                if (offsetValue != null && StringUtils.hasLength(offsetValue.getFile())) {
                    /**
                     * 显示重试10次
                     */
                    for (int index = 0; index < 10; index++) {
                        try {
                            if (this.jedis == null) {
                                this.jedis = JedisPoolFactory.getNewJedisResource(redisConfig);
                            }

                            Map<String, String> binlogMp = new HashMap<>();
                            binlogMp.put(StartupOptionUtil.REDIS_MS_BINLOG_FILE_NAME_OFFSET_BINLOG_FILE_NAME, offsetValue.getFile());
                            binlogMp.put(StartupOptionUtil.REDIS_MS_BINLOG_FILE_NAME_OFFSET_BINLOG_FILE_NAME_LAST_TABLE, String.format("%s:%s", offsetValue.getNote(), new Date()));

                            binlogMp.put(StartupOptionUtil.REDIS_MS_BINLOG_FILE_NAME_OFFSET_BINLOG_OFFSET_VALUE, String.valueOf(offsetValue.getOffset()));
                            if (StringUtils.hasLength(offsetValue.getGtIds())) {
                                binlogMp.put(StartupOptionUtil.REDIS_MS_BINLOG_GT_IDS_VALUE, offsetValue.getGtIds());
                            }

                            this.jedis.hset(StartupOptionUtil.REDIS_MS_BINLOG_FILE_NAME_OFFSET, binlogMp);
                            this.hasValueChangedAfterSnapshot = false;

                            return;
                        } catch (Exception ex) {
                            if (index >= 9) {
                                throw ex;
                            }

                            if (ExceptionUtils.getStackTrace(ex).contains("JedisConnectionException")) {

                                /**
                                 * 针对该情况, 系统自动重现获取连接并执行重试
                                 */
                                if (this.jedis != null) {
                                    try {
                                        this.jedis.close();
                                    } catch (Exception e) {
                                    }

                                    this.jedis = JedisPoolFactory.getNewJedisResource(redisConfig);
                                }

                                logger.error("系统重新链接并尝试重试:{}-{}", index, ex);
                            } else {
                                logger.error("其他其他异常: 系统将复用同一个连接进行重试:{}-{}", index, ex);
                            }
                        }
                    }
                }
            }
        }
    }

    @Override
    public void initializeState(FunctionInitializationContext context) throws Exception {
        //todo
        /*
        if (binlogFileValueState != null) {
            if (this.binlogOffsetValue != null) {
                this.offsetValueState.update(binlogOffsetValue.getOffset());
                this.binlogFileValueState.update(binlogOffsetValue.getFile());
            }
        }
         */

        if (this.lastMapKeySyncDateState != null && this.lastMapKeySyncDateState.keys() != null) {
            Iterator<String> keys = this.lastMapKeySyncDateState.keys().iterator();
            String getKey = null;
            while (keys.hasNext()) {
                if (StringUtils.hasLength(getKey)) {
                    throw new BadEventException(String.format("该分区状态中存在多个key(fullTable)的情况(current=%s;new=%s); 请检查程序进行修复",
                            getKey, keys.next())
                    );
                }
                getKey = keys.next();
            }
        }
    }

    @Override
    public void processElement(BinlogCdcValue value, KeyedProcessFunction<String, BinlogCdcValue, BinlogCdcValue>.Context ctx, Collector<BinlogCdcValue> out) throws Exception {
        StringBuilder sb = new StringBuilder();
        try {
            sb.append("开始接收binlog的数据信息");
            out.collect(value);
            sb.append("开始存储binlogOffsetValueState的信息");
            if (value.getBinlogOffsetValue() != null) {
                this.binlogOffsetValueValueState.update(value.getBinlogOffsetValue());
            }
            sb.append("完成存储binlogOffsetValueState的信息");
            this.hasValueChangedAfterSnapshot = true;
        } catch (Exception ex) {
            ex.printStackTrace();
            logger.error(
                    "BinlogCheckpointProcessFunction.processElement执行失败:{}; 数据={} 异常:{}",
                    sb,
                    value == null ? "NULL" : value.getValue(),
                    ex);
        }
    }
}
