package com.bcxin.event.job.domain.entities;

import com.baomidou.mybatisplus.annotation.*;
import com.bcxin.event.core.*;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.job.domain.enums.JobType;
import com.bcxin.event.job.domain.repositories.JdbcDefinitionRepository;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.type.EnumOrdinalTypeHandler;
import org.apache.ibatis.type.JdbcType;

import java.sql.Timestamp;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.logging.SimpleFormatter;
import java.util.stream.Collectors;

/**
 * 对于mp注入的方法, 如果没有配置autoResultMap=true的话, 那么系统在获取的时候默认还是无法使用如下jobType中指定的typeHandler=EnumOrdinalTypeHandler.class
 */
@Getter
@Setter(AccessLevel.PROTECTED)
@TableName(value = "flink_jobs",autoResultMap = true)
public class JobEntity extends TableEntityAbstract {
    @TableId(value = "id", type = IdType.AUTO)
    private Integer id;

    @TableField(insertStrategy = FieldStrategy.NOT_EMPTY, updateStrategy = FieldStrategy.NOT_EMPTY)
    private String name;

    @TableField(insertStrategy = FieldStrategy.NOT_EMPTY, updateStrategy = FieldStrategy.NOT_EMPTY)
    private String title;

    @TableField(value = "job_type",jdbcType = JdbcType.TINYINT,javaType = true, typeHandler = EnumOrdinalTypeHandler.class)
    private JobType jobType;

    @TableField(value = "parameter_json", jdbcType = JdbcType.CLOB)
    private String parameterJson;
    private String description;

    @TableField(value = "is_online",jdbcType = JdbcType.BOOLEAN,insertStrategy = FieldStrategy.NOT_NULL,updateStrategy = FieldStrategy.NOT_NULL)
    private boolean online;

    @TableField(value = "last_updated_time", jdbcType = JdbcType.TIMESTAMP,updateStrategy = FieldStrategy.NOT_NULL)
    private Timestamp lastUpdatedTime;

    @TableField(value = "created_time", jdbcType = JdbcType.TIMESTAMP,updateStrategy = FieldStrategy.NOT_NULL)
    private Timestamp createdTime;

    @TableField(value = "flink_job_id")
    private String jobId;

    @TableField(value = "flink_job_started_time", jdbcType = JdbcType.TIMESTAMP)
    private Timestamp startJobTime;

    @TableField(value = "flink_job_started_response", jdbcType = JdbcType.CLOB)
    private String startJobResponse;

    @TableField(value = "is_deleted",
            jdbcType = JdbcType.TIMESTAMP,
            insertStrategy = FieldStrategy.NOT_NULL,
            updateStrategy = FieldStrategy.NOT_NULL)
    private boolean deleted;

    public JobParameterDTO getParameter(JsonProvider jsonProvider) {
        if (StringUtils.isEmpty(this.getParameterJson())) {
            return null;
        }

        return jsonProvider.toObject(JobParameterDTO.class, this.getParameterJson());
    }

    protected JobEntity() {
        this.setCreatedTime(Timestamp.from(Instant.now()));
        this.setLastUpdatedTime(Timestamp.from(Instant.now()));
        this.setOnline(true);
        this.setDeleted(false);
    }

    public void change(String name,String title, JobType jobType,
                       String parameterJson, String description,
                       JsonProvider jsonProvider,
                       JdbcDefinitionRepository jdbcDefinitionRepository) {
        this.setName(name);
        this.setTitle(title);
        this.setJobType(jobType);
        validateParameterJson(parameterJson, jsonProvider, jdbcDefinitionRepository);

        this.setParameterJson(parameterJson);
        this.setDescription(description);
        this.setLastUpdatedTime(Timestamp.from(Instant.now()));
    }

    public void markOnline(boolean isOnline) {
        this.setOnline(isOnline);
        this.setLastUpdatedTime(Timestamp.from(Instant.now()));
    }

    public void markDeleted() {
        this.setDeleted(true);
        this.setLastUpdatedTime(Timestamp.from(Instant.now()));
        this.setName(String.format("%s-%s-%s", this.getName(), this.getId(), Instant.now()));
    }

    public static JobEntity create(String name,String title, JobType jobType,
                                   String parameterJson,
                                   String description,
                                   JsonProvider jsonProvider,
                                   JdbcDefinitionRepository jdbcDefinitionRepository) {
        JobEntity job = new JobEntity();

        job.change(name,title,jobType,parameterJson,description,jsonProvider,jdbcDefinitionRepository);
        return job;
    }

    private static void validateParameterJson(
            String parameterJson,
            JsonProvider jsonProvider,JdbcDefinitionRepository jdbcDefinitionRepository) {
        JobExecutorParam jobExecutorParam =
                jsonProvider.toObject(JobExecutorParam.class, parameterJson);
        if (jobExecutorParam == null) {
            throw new BadEventException(String.format("无效参数:%s", parameterJson));
        }


        Kafka2JdbcDefinitionWrapperDTO Kafka2JdbcDefinitionWrapper = jsonProvider.toObject(Kafka2JdbcDefinitionWrapperDTO.class,
                jobExecutorParam.getParam2());
        if (Kafka2JdbcDefinitionWrapper == null) {
            throw new BadEventException(String.format("无效Param2参数:%s", jobExecutorParam.getParam2()));
        }

        Collection<Kafka2JdbcDefinitionWrapperDTO.Kafka2JdbcDefinitionDTO> kafka2JdbcDefinitions =
                Kafka2JdbcDefinitionWrapper.getKafka2JdbcDefinitions();
        Collection<Long> ids = kafka2JdbcDefinitions.stream()
                .filter(ii -> ii.getJdbcDefinitions() != null)
                .flatMap(ix -> ix.getJdbcDefinitions().stream().map(it -> it.getJdbcId()))
                .distinct()
                .collect(Collectors.toList());//jdbcSqlTemplateDefinitions.stream().map(ix -> ix.getJdbcId()).collect(Collectors.toList());

        Collection<JdbcDefinitionEntity> jdbcDefinitions =
                jdbcDefinitionRepository.getAllByIds(ids);

        Collection<Long> notMatchedIds =
                ids.stream().filter(ix -> !jdbcDefinitions.stream().anyMatch(it -> it.getId().equals(ix)))
                        .collect(Collectors.toList());
        if (!notMatchedIds.isEmpty()) {
            throw new BadEventException(String.format("找不到对应的jdbcIds=%s",
                    notMatchedIds.stream().map(ix -> String.valueOf(ix)).collect(Collectors.joining(",")))
            );
        }
    }
}
