import { Corners, Coord } from '@turf/helpers';
import { GeoJSON, GeometryCollection } from 'geojson';

/**
 * Scale GeoJSON objects from a given point by a scaling factor e.g. factor=2
 * would make each object 200% larger.
 * If a FeatureCollection is provided, the origin point will be calculated
 * based on each individual feature _unless_ an exact
 *
 * @function
 * @param {GeoJSON|GeometryCollection} geojson objects to be scaled
 * @param {number} factor of scaling, positive values greater than 0. Numbers between 0 and 1 will shrink the geojson, numbers greater than 1 will expand it, a factor of 1 will not change the geojson.
 * @param {Object} [options={}] Optional parameters
 * @param {Corners|Coord} [options.origin='centroid'] Point from which the scaling will occur (string options: sw/se/nw/ne/center/centroid)
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance improvement if true)
 * @returns {GeoJSON|GeometryCollection} scaled GeoJSON
 * @example
 * const poly = turf.polygon([[[0,29],[3.5,29],[2.5,32],[0,29]]]);
 * const scaledPoly = turf.transformScale(poly, 3);
 *
 * //addToMap
 * const addToMap = [poly, scaledPoly];
 * scaledPoly.properties = {stroke: '#F00', 'stroke-width': 4};
 */
declare function transformScale<T extends GeoJSON | GeometryCollection>(geojson: T, factor: number, options?: {
    origin?: Corners | Coord;
    mutate?: boolean;
}): T;

export { transformScale as default, transformScale };
