/**
 * The MIT License (MIT)
 * <p>
 * Copyright (c) 2016 Caratacus
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
package com.bcxin.risk.hibernateplus.dao.impl;

import com.bcxin.risk.hibernateplus.condition.SelectWrapper;
import com.bcxin.risk.hibernateplus.condition.wrapper.Wrapper;
import com.bcxin.risk.hibernateplus.dao.IDao;
import com.bcxin.risk.hibernateplus.entity.Convert;
import com.bcxin.risk.hibernateplus.entity.page.Page;
import com.bcxin.risk.hibernateplus.utils.*;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.transform.Transformers;
import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.math.BigInteger;
import java.util.*;


/**
 * <p>
 * IDao接口实现类
 * </p>
 *
 * @author Caratacus
 * @date 2016-11-23
 */
public class DaoImpl<P extends Convert> implements IDao<P> {

    private static final Logger logger = Logger.getLogger(DaoImpl.class);

    @Autowired
    protected SessionFactory sessionFactory;

    protected Session getSession(){
        return HibernateUtils.getSession(this.masterSession(), this.isCurrent());
    }


    protected SessionFactory masterSession() {
        return EntityInfoUtils.getEntityInfo(this.poClass()).getMaster();
    }

    protected SessionFactory slaveSession() {
        Set slaves = EntityInfoUtils.getEntityInfo(this.poClass()).getSlaves();
        return CollectionUtils.isEmpty(slaves)?this.masterSession():(SessionFactory) RandomUtils.getRandomElement(slaves);
    }

    protected Class<P> poClass() {
        return ReflectionKit.getSuperClassGenricType(this.getClass(), 0);
    }

    protected Boolean isCurrent() {
        return Boolean.valueOf(true);
    }



    /**
     * 清除缓冲区对象
     */
    @Override
    public void evictObj(Object obj){
        Session session = getSession();
        session.evict(obj);
    }

    /**
     * 合并缓冲区对象
     */
    @Override
    public void mergeObj(Object obj){
        Session session = getSession();
        reflectSetCreateOn(obj);
        reflectSetUpdateOn(obj);
        session.merge(obj);
    }

    @Override
    public boolean insertBatch(List<P> list, int size) {

        try {
            Session e = HibernateUtils.getSession(this.masterSession(), this.isCurrent());

            for(int i = 0; i < list.size(); ++i) {
                e.save(list.get(i));
                if(i % size == 0) {
                    e.flush();
                    e.clear();
                }
            }

            return true;
        } catch (Exception var5) {
            logger.warn("Warn: Unpected exception.  Cause:" + var5);
            return false;
        }
    }

    @Override
    public Criteria createCriteria(Class clazz){
        return getSession().createCriteria(clazz);
    }

    @Override
    public <T>T selectById(Long oid){
        return (T)getSession().get(this.poClass(),oid);
    }


    /**
     * 查询唯一的对象，如oid，用户名等
     */
    @Override
    public <T>T selectOne(Criterion criterion){
        Criteria criteria = createCriteria(this.poClass());
        if(criterion != null){
            criteria.add(criterion);
        }
        return (T)criteria.uniqueResult();
    }
    /**
     * 查询唯一的对象，如oid，用户名等
     */
    @Override
    public <T>T selectOne(List<Criterion> criterionList){
        Criteria criteria = createCriteria(this.poClass());
        if(criterionList != null) {
            for(int i = 0; i < criterionList.size(); i++) {
                criteria.add(criterionList.get(i));
            }
        }
        return (T)criteria.uniqueResult();
    }



    @Override
    public void save(Object obj){
        reflectSetCreateOn(obj);
        getSession().save(obj);
    }

    @Override
    public void saveOrUpdate(Object obj) {
        try{
            Field[] fields = obj.getClass().getSuperclass().getDeclaredFields();
            boolean hasField = checkExistsOid(obj, fields);
            if(!hasField) {
                fields = obj.getClass().getSuperclass().getSuperclass().getDeclaredFields();
            }
            for( Field field : fields){
                if (field.getName().equals("oid")) {
                    field.setAccessible(true);
                    if(field.get(obj) != null){
                        reflectSetUpdateOn(obj);
                    }else{
                        reflectSetCreateOn(obj);
                    }
                    break;
                }
            }
        } catch(Exception e){
            e.printStackTrace();
        }
        getSession().saveOrUpdate(obj);
    }


    private boolean checkExistsOid(Object object, Field[] fields) throws IllegalAccessException {
        boolean hasField = false;
        for( Field field : fields){
            if (field.getName().equals("oid")) {
                field.setAccessible(true);
                hasField = true;
                break;
            }
        }
        return hasField;
    }



    @Override
    public void update(Object obj){
        reflectSetUpdateOn(obj);
        getSession().update(obj);
    }


    public int update(Wrapper wrapper) {
        String sqlUpdate = SqlUtils.sqlUpdate(this.poClass(), wrapper);
        return this.executeSqlUpdate(sqlUpdate);
    }

    @Override
    public void deleteObj(Object obj){
        Session session = getSession();
        if(obj != null){
            session.delete(obj);
        }
    }

    @Override
    public int delete(Serializable id) {
        String sqlDelete = SqlUtils.sqlDelete(this.poClass(), id);
        return this.executeSqlUpdate(sqlDelete);
    }

    @Override
    public int delete(Wrapper wrapper) {
        String sqlDelete = SqlUtils.sqlDelete(this.poClass(), wrapper);
        return this.executeSqlUpdate(sqlDelete);
    }

    @Override
    public boolean updateBatch(List<P> list, int size) {

        try {
            Session e = HibernateUtils.getSession(this.masterSession(), this.isCurrent());

            for(int i = 0; i < list.size(); ++i) {
                e.merge(list.get(i));
                if(i % size == 0) {
                    e.flush();
                    e.clear();
                }
            }

            return true;
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
            return false;
        }
    }

    /**
     * 查询所有结果集
     * @param <T>
     * @return
     */
    @Override
    public <T>List selectList(){
        Criteria criteria = createCriteria(this.poClass());
        return criteria.list();
    }


    /**
     * 查询结果集，单个条件
     * @param criterion
     * @param <T>
     * @return
     */
    @Override
    public <T>List selectList(Criterion criterion){
        Criteria criteria = createCriteria(this.poClass());
        if(criterion != null){
            criteria.add(criterion);
        }
        return criteria.list();
    }

    /**
     *
     * <b>根据条件查询集合,主动申明排序方式 </b>
     *
     * @return
     * @author LuoPeng
     * @date 2016年12月15日 下午1:38:10
     * @注意事项 </b>
     * <b>
     */
    @Override
    public <T>List selectList(Criterion criterion,Order order) {
        Criteria criteria = createCriteria(this.poClass());
        if(criterion != null){
            criteria.add(criterion);
        }
        if(order != null) {
            criteria.addOrder(order);
        }
        return criteria.list();
    }

    /**
     *
     * <b>根据条件查询集合,主动申明排序方式 </b>
     *
     * @return
     * @author LuoPeng
     * @date 2016年12月15日 下午1:38:10
     * @注意事项 </b>
     * <b>
     */
    @Override
    public <T>List selectList(Order order) {
        Criteria criteria = createCriteria(this.poClass());
        if(order != null) {
            criteria.addOrder(order);
        }
        return criteria.list();
    }


    public <T>List selectList(Order[] orders) {
        Criteria criteria = createCriteria(this.poClass());
        if(orders != null) {
            for (int i = 0; i < orders.length; i++) {
                criteria.addOrder(orders[i]);
            }
        }
        return criteria.list();
    }


    /**
     * 查询结果集，多个条件
     * @param criterionList
     * @param <T>
     * @return
     */
    @Override
    public <T>List selectList(List<Criterion> criterionList){
        Criteria criteria = createCriteria(this.poClass());
        if(criterionList != null) {
            for(int i = 0; i < criterionList.size(); i++) {
                criteria.add(criterionList.get(i));
            }
        }
        return criteria.list();
    }

    /**
     * 查询结果集，多个条件
     * @param criterionList
     * @param <T>
     * @return
     */
    @Override
    public <T>List selectList(List<Criterion> criterionList,Order order){
        Criteria criteria = createCriteria(this.poClass());
        if(criterionList != null) {
            for(int i = 0; i < criterionList.size(); i++) {
                criteria.add(criterionList.get(i));
            }
        }
        if(order != null) {
            criteria.addOrder(order);
        }
        return criteria.list();
    }

    /**
     * 查询结果集，多个条件,多个排序
     * @param criterionList
     * @param <T>
     * @return
     */

    public <T>List selectList(List<Criterion> criterionList,Order[] orders){
        Criteria criteria = createCriteria(this.poClass());
        if(criterionList != null) {
            for(int i = 0; i < criterionList.size(); i++) {
                criteria.add(criterionList.get(i));
            }
        }
        if(orders != null) {
            for (int i = 0; i < orders.length; i++) {
                criteria.addOrder(orders[i]);
            }
        }
        return criteria.list();
    }

    /**
     * 查询单表条数
     * @return
     */
    @Override
    public int selectCount() {
        int count = 0;

        try {
            String e = SqlUtils.sqlCount(this.poClass(), null);
            Query query = HibernateUtils.getSqlQuery(e, this.slaveSession(), this.isCurrent());
            BigInteger bigInteger = (BigInteger)query.uniqueResult();
            count = bigInteger.intValue();
        } catch (Exception var6) {
        }

        return count;
    }

    /**
     *
     * <b>根据条件查询集合并分页 </b>
     *
     * @return
     * @author LuoPeng
     * @date 2016年12月15日 下午1:38:10
     * <b>
     */
    @Override
    public <T>List selectPage(Criterion criterion, Page page) {
        Criteria criteria = createCriteria(this.poClass());
        if(criterion!=null) {
            criteria.add(criterion);
        }
        if( page != null) {
            int totalCount = ((Long) criteria.setProjection(Projections.rowCount()).uniqueResult()).intValue();
            page.setTotal(totalCount);

            criteria.setProjection(null);
            criteria.setFirstResult((page.getCurrent() - 1) * page.getSize()).setMaxResults(page.getSize());
        }

        return criteria.list();
    }

    /**
     *
     * <b>根据条件查询集合并分页 </b>
     *
     * @return
     * @author LuoPeng
     * @date 2016年12月15日 下午1:38:10
     * <b>
     */
    @Override
    public <T>List selectPage(List<Criterion> criterionList, Page page) {
        Criteria criteria = createCriteria(this.poClass());
        if(criterionList != null) {
            for(int i = 0; i < criterionList.size(); i++) {
                criteria.add(criterionList.get(i));
            }
        }
        if( page != null) {
            int totalCount = ((Long) criteria.setProjection(Projections.rowCount()).uniqueResult()).intValue();
            page.setTotal(totalCount);

            criteria.setProjection(null);
            criteria.setFirstResult((page.getCurrent() - 1) * page.getSize()).setMaxResults(page.getSize());
        }

        return criteria.list();
    }

    /**
     *
     * <b>根据条件查询集合并分页，主动申明排序方式 </b>
     *
     * @return
     * @author LuoPeng
     * @date 2016年12月15日 下午1:38:10
     * @注意事项 </b>
     * <b>
     */
    @Override
    public <T>List selectPage(List<Criterion> criterionList, Page page, Order order) {
        Criteria criteria = createCriteria(this.poClass());
        if(criterionList != null) {
            for(int i = 0; i < criterionList.size(); i++) {
                criteria.add(criterionList.get(i));
            }
        }
        if(order != null) {
            criteria.addOrder(order);
        }
        if( page != null) {
            int totalCount = ((Long) criteria.setProjection(Projections.rowCount()).uniqueResult()).intValue();
            page.setTotal(totalCount);

            criteria.setProjection(null);
            criteria.setFirstResult((page.getCurrent() - 1) * page.getSize()).setMaxResults(page.getSize());
        }

        return criteria.list();
    }

    /******************************sql*************************************/

    protected int executeSqlUpdate(String sql) {
        return this.executeSqlUpdate(sql, Collections.EMPTY_MAP);
    }

    protected int executeSqlUpdate(String sql, Object[] args) {
        Assert.hasLength(sql);
        int resultCount = 0;

        try {
            Query e = HibernateUtils.getSqlQuery(sql, this.masterSession(), this.isCurrent());
            if(null != args) {
                for(int i = 0; i < args.length; ++i) {
                    HibernateUtils.setParams(e, StringUtils.toString(Integer.valueOf(i)), args[i]);
                }
            }

            resultCount = e.executeUpdate();
        } catch (Exception var6) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var6);
        }

        return resultCount;
    }

    protected int executeSqlUpdate(String sql, Map<String, Object> params) {
        Assert.hasLength(sql);
        int resultCount = 0;
        if(StringUtils.isNotBlank(sql)) {
            try {
                Query e = HibernateUtils.getSqlQuery(sql, this.masterSession(), this.isCurrent());
                this.setParamMap(params, e);
                resultCount = e.executeUpdate();
            } catch (Exception var5) {
                logger.warn("Warn: Unexpected exception.  Cause:" + var5);
            }
        }

        return resultCount;
    }

    protected int queryAggregate(String sql) {
        return this.queryAggregate(sql, SelectWrapper.DEFAULT);
    }

    protected int queryAggregate(String sql, Wrapper wrapper) {
        sql = sql + wrapper.getSqlSegment();
        byte count = 0;

        try {
            Query e = HibernateUtils.getSqlQuery(sql, this.slaveSession(), this.isCurrent());
            Object obj = e.uniqueResult();
            if ( obj != null) {
                return Integer.parseInt(obj.toString());
            }
        } catch (Exception var6) {
            var6.printStackTrace();
        }

        return count;
    }

    @Override
    public Map<String, Object> queryMap(String sql, Wrapper wrapper) {
        Assert.hasLength(sql);
        Map map = Collections.emptyMap();

        try {
            sql = sql + wrapper.getSqlSegment();
            Query e = HibernateUtils.getSqlQuery(sql, this.slaveSession(), this.isCurrent()).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
            map = (Map)e.uniqueResult();
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }

        return map;
    }

    @Override
    public List<Map<String, Object>> queryMaps(String sql, Wrapper wrapper) {
        List list = Collections.emptyList();

        try {
            sql = sql + wrapper.toString();
            Query e = HibernateUtils.getSqlQuery(sql, this.slaveSession(), this.isCurrent()).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
            list = e.list();
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }

        return list;
    }


 // 查询sql记录条数
    @Override
    public int querySqlCounts(String sql, Wrapper wrapper) {
        List list = Collections.emptyList();
        try {
            sql = sql + wrapper.toString();
            Query e = HibernateUtils.getSqlQuery(sql, this.slaveSession(), this.isCurrent()).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
            list = e.list();
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }
        return list.size();
    }

    // 查询sql记录条数
    public int querySqlCounts(String sql) {
        List list = Collections.emptyList();
        try {
            Query e = HibernateUtils.getSqlQuery(sql, this.slaveSession(), this.isCurrent()).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
            list = e.list();
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }
        return list.size();
    }


    protected Page<Map<String, Object>> queryMapsPage(String sql, Page page) {
        return this.queryMapsPage(sql, SelectWrapper.DEFAULT, page);
    }

    protected Page<Map<String, Object>> queryMapsPage(String sql, Wrapper wrapper, Page page) {
        Assert.hasLength(sql);

        try {
            sql = SqlUtils.sqlList(sql, wrapper, page);
            Query e = HibernateUtils.getSqlQuery(sql, this.slaveSession(), this.isCurrent()).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
            HibernateUtils.setPage(page.getCurrent(), page.getSize(), e);
            page.setRecords(e.list());
            this.setPageTotal(sql, page);
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }

        return page;
    }

    protected List<Map<String, Object>> queryMaps(String sql) {
        return this.queryMaps(sql, Collections.EMPTY_MAP);
    }

    protected List<Map<String, Object>> queryMaps(String sql, Map<String, Object> params) {
        Assert.hasLength(sql);
        List list = Collections.emptyList();

        try {
            Query e = HibernateUtils.getSqlQuery(sql, this.slaveSession(), this.isCurrent()).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
            this.setParamMap(params, e);
            list = e.list();
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }

        return list;
    }


    private void setPageTotal(String sql, Page<?> page) {
        if(page.isSearchCount()) {
            String sqlCount = SqlUtils.sqlCountOptimize(sql);
            Query countQuery = HibernateUtils.getSqlQuery(sqlCount, this.slaveSession(), this.isCurrent());
            BigInteger bigInteger = (BigInteger)countQuery.uniqueResult();
            page.setTotal(bigInteger.intValue());
        }

    }

    private void setParamMap(Map<String, Object> params, Query query) {
        if(MapUtils.isNotEmpty(params)) {
            Iterator i$ = params.keySet().iterator();

            while(i$.hasNext()) {
                String key = (String)i$.next();
                Object obj = params.get(key);
                HibernateUtils.setParams(query, key, obj);
            }
        }

    }


    /***need delete*/
    @Override
    public <P> List<P> selectList(Wrapper wrapper) {
        List list = Collections.emptyList();

        try {
            String e = SqlUtils.sqlEntityList(this.poClass(), wrapper, (Page)null);
            Query query = HibernateUtils.getEntitySqlQuery(this.poClass(), e, this.slaveSession(), this.isCurrent());
            list = query.list();
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }

        return list;
    }

    @Override
    public List<Map<String, Object>> selectMaps(Wrapper wrapper) {
        List list = Collections.emptyList();

        try {
            String e = SqlUtils.sqlList(this.poClass(), wrapper, (Page)null);
            Query query = HibernateUtils.getSqlQuery(e, this.slaveSession(), this.isCurrent()).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
            list = query.list();
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }

        return list;
    }

    @Override
    public P selectOne(Wrapper wrapper) {
        List list = this.selectList(wrapper);
        if(CollectionUtils.isNotEmpty(list)) {
            int size = list.size();
            if(size > 1) {
                logger.warn(String.format("Warn: selectOne Method There are  %s results.", new Object[]{Integer.valueOf(size)}));
            }

            return (P) list.get(0);
        } else {
            return null;
        }
    }

    @Override
    public Page selectPage(Wrapper wrapper, Page page) {
        try {
            String e = SqlUtils.sqlEntityList(this.poClass(), wrapper, page);
            Query query = HibernateUtils.getEntitySqlQuery(this.poClass(), e, this.slaveSession(), this.isCurrent());
            HibernateUtils.setPage(page.getCurrent(), page.getSize(), query);
            page.setRecords(query.list());
            this.setPageTotal(e, page);
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }

        return page;
    }

    private void reflectSetCreateOn(Object object){
        try{
            Field[] fields = object.getClass().getSuperclass().getDeclaredFields();
            boolean hasField = setCreateOn(object, fields);
            if (!hasField) {
                fields = object.getClass().getSuperclass().getSuperclass().getDeclaredFields();
                setCreateOn(object, fields);
            }

        } catch(Exception e){
            e.printStackTrace();
        }
    }

    private boolean setCreateOn(Object object, Field[] fields) throws IllegalAccessException {
        boolean hasField = false;
        for( Field field : fields){
            if (field.getName().equals("createOn")) {
                field.setAccessible(true);
                if( field.getType() == Date.class) {
                    field.set(object, new Date());
                }
                hasField = true;
                break;
            }
        }
        return hasField;
    }

    private void reflectSetUpdateOn(Object object){
        try{
            Field[] fields = object.getClass().getSuperclass().getDeclaredFields();
            boolean hasField = setUpdateOn(object, fields);
            if (!hasField) {
                fields = object.getClass().getSuperclass().getSuperclass().getDeclaredFields();
                setUpdateOn(object, fields);
            }

        } catch(Exception e){
            e.printStackTrace();
        }
    }

    private boolean setUpdateOn(Object object, Field[] fields) throws IllegalAccessException {
        boolean hasField = false;
        for( Field field : fields){
            if (field.getName().equals("updateOn")) {
                hasField = true;
                field.setAccessible(true);
                if( field.getType() == Date.class) {
                    field.set(object, new Date());
                }
                break;
            }
        }
        return hasField;
    }

    public <T>List queryLists(String sql, Wrapper wrapper) {
        List list = Collections.emptyList();

        try {
            sql = sql + wrapper.toString();
            Query e = HibernateUtils.getSqlQuery(sql, this.slaveSession(), this.isCurrent()).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
            list = e.list();
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }

        return list;
    }

    public <P> List<P> selectListBySql(String sql) {
        List list = Collections.emptyList();

        try {
            String e = sql ;
            Query query = HibernateUtils.getEntitySqlQuery(this.poClass(), e, this.slaveSession(), this.isCurrent());
            list = query.list();
        } catch (Exception var5) {
            logger.warn("Warn: Unexpected exception.  Cause:" + var5);
        }

        return list;
    }



}
