package com.bcxin.sync.service;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.sync.common.emus.OpenType;
import com.bcxin.sync.common.exception.GetResponseException;
import com.bcxin.sync.common.utils.AssertUtils;
import com.bcxin.sync.common.utils.CommonUtils;
import com.bcxin.sync.common.utils.HttpUtil;
import com.bcxin.sync.common.utils.TuoluojiangUtil;
import com.bcxin.sync.configs.SyncConfig;
import com.bcxin.sync.dtos.request.*;
import com.bcxin.sync.dtos.response.BcxDepartmentResponse;
import com.bcxin.sync.dtos.response.BcxEmployeeResponse;
import com.bcxin.sync.entity.tenant.*;
import com.bcxin.sync.entity.tuoluojiang.EbAdminEntity;
import com.bcxin.sync.entity.tuoluojiang.EbAttendanceGroupMemberEntity;
import com.bcxin.sync.entity.tuoluojiang.EbEnterpriseEntity;
import com.bcxin.sync.entity.tuoluojiang.EbUserEntity;
import com.bcxin.sync.service.datasync.OrganizationOpenService;
import com.bcxin.sync.service.tenant.*;
import com.bcxin.sync.service.tuoluojiang.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.RequestBody;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * description：数据同步陀螺匠服务层
 * author：linchunpeng
 * date：2024/12/31
 */
@Slf4j
@Service
public class DataSyncToTljService {

    @Autowired
    private TenantOrganizationsService tenantOrganizationsService;
    @Autowired
    private TenantEmployeesService tenantEmployeesService;
    @Autowired
    private TenantDepartmentsService tenantDepartmentsService;
    @Autowired
    private TenantUsersService tenantUsersService;
    @Autowired
    private TenantUserCredentialsService tenantUserCredentialsService;
    @Autowired
    private TenantDepartmentEmployeeRelationsService tenantDepartmentEmployeeRelationsService;
    @Autowired
    private TlkAttendanceSiteBaseInfoService tlkAttendanceSiteBaseInfoService;
    @Autowired
    private TlkAttendanceSitePersonInfoService tlkAttendanceSitePersonInfoService;
    @Autowired
    private TlkEventProjectService tlkEventProjectService;
    @Autowired
    private TlkEventTeamService tlkEventTeamService;

    @Autowired
    private EbUserService ebUserService;
    @Autowired
    private EbAdminService ebAdminService;
    @Autowired
    private EbEnterpriseService ebEnterpriseService;
    @Autowired
    private EbFrameService ebFrameService;
    @Autowired
    private EbAttendanceGroupService ebAttendanceGroupService;
    @Autowired
    private EbAttendanceGroupMemberService ebAttendanceGroupMemberService;
    @Autowired
    private EbEnterpriseRoleService ebEnterpriseRoleService;
    @Autowired
    private EbApproveService ebApproveService;
    @Autowired
    private EbAttendanceShiftService ebAttendanceShiftService;

    @Autowired
    private OrganizationOpenService organizationOpenService;

    @Autowired
    private SyncConfig syncConfig;
    @Value("${spring.profiles.active}")
    private String activeFile;

    @Autowired
    private RedisCache redisCache;

    private static final String WEB_LOGIN_URL = "/admin/landing";
    private static final String H5_LOGIN_URL = "/work/pages/landing";


    /**
     * description：获取登录链接的加密参数
     * author：linchunpeng
     * date：2025/1/2
     */
    public String getLoginEncrypt(NotPasswordLoginRequest request, boolean isH5) {
        if (!organizationOpenService.isOpen(request.getOrgId(), OpenType.TLJ.getCode())) {
            throw new GetResponseException(403, "当前企业权限未开通.");
        }
        if (!organizationOpenService.isOpenAndEffective(request.getOrgId(), OpenType.TLJ.getCode())) {
            throw new GetResponseException(403, "当前企业权限已过期.");
        }

        String key = "data-sync:tenant-user:card-no:" + request.getUserId();
        String cardNo = redisCache.getCacheObject(key);
        if (StringUtils.isBlank(cardNo)) {
            List<TenantUserCredentialsEntity> credentialsEntityList = tenantUserCredentialsService.getByTenantUserId(request.getUserId());
            if (CollectionUtil.isEmpty(credentialsEntityList)) {
                throw new GetResponseException(401, "当前用户的证件号码为空，禁止访问.");
            }

            cardNo = credentialsEntityList.get(0).getNumber();
            for (TenantUserCredentialsEntity credentials : credentialsEntityList) {
                if (credentials.getCredentialType() == 0) {
                    //优先身份证
                    cardNo = credentials.getNumber();
                    break;
                }
            }
            redisCache.setCacheObject(key, cardNo, 30, TimeUnit.MINUTES);
        }

        long timeSecond = System.currentTimeMillis()/1000;
        String encryptString = request.getUserId().concat(request.getOrgId().toUpperCase(Locale.ROOT)) + timeSecond + "57e13b9b5c841d1a";
        String encryptResult = TuoluojiangUtil.md5(encryptString);
        if (StringUtils.isNotBlank(syncConfig.getApiHost().getTuoluojiang())
                && StringUtils.isNotBlank(request.getUserId())
                && StringUtils.isNotBlank(cardNo)
                && StringUtils.isNotBlank(request.getOrgId())) {
            String result = syncConfig.getApiHost().getTuoluojiang().concat(isH5 ? H5_LOGIN_URL : WEB_LOGIN_URL).concat("?sign=").concat(encryptResult)
                    .concat("&cid=").concat(cardNo)
                    .concat("&uid=").concat(request.getUserId())
                    .concat("&eid=").concat(request.getOrgId())
                    .concat("&redirect=").concat(request.getRedirect());
            return result;
        }
        return encryptResult;
    }


    /**
     * description：开通陀螺匠
     * author：linchunpeng
     * date：2025/1/17
     */
    public String sync(String organizationId) {
        AssertUtils.isTrue(organizationOpenService.isOpenAndEffective(organizationId, OpenType.TLJ.getCode()), "该组织未开通，或者开通的权限已到期");
        TenantOrganizationsEntity organizationsEntity = tenantOrganizationsService.getById(organizationId);
        AssertUtils.isNotNullObject(organizationsEntity, "企业不存在");

        String adminUserId = "";
        String adminEmployeeId = "";
        List<BcxEmployeeResponse> employeeResponseList = tenantEmployeesService.getEmployeeResponseListByOrganizationId(organizationsEntity.getId(), 0);
        Optional<BcxEmployeeResponse> admin = employeeResponseList.stream().filter(emp -> emp.getIsDomainAdmin() == 1).findFirst();
        if (admin.isPresent()) {
            adminUserId = admin.get().getUserId();
            adminEmployeeId = admin.get().getId();
        } else {
            TenantEmployeesEntity adminEmployeeEntity = tenantEmployeesService.getAdminByOrgId(organizationsEntity.getId());
            TenantUsersEntity adminUserEntity = tenantUsersService.getById(adminEmployeeEntity.getTenantUserId());
            adminUserId = adminUserEntity.getId();
            adminEmployeeId = adminEmployeeEntity.getId();
        }

        StringBuffer syncResult = new StringBuffer(1024);
        log.info("同步用户");
        syncResult.append(ebUserService.userCreateBatch(employeeResponseList));
        log.info("同步企业");
        syncResult.append(ebEnterpriseService.companyCreate(organizationsEntity.getId(), organizationsEntity.getName(), adminUserId));
        log.info("同步部门，等陀螺匠创建企业和部门，等10秒");
        try {
            Thread.sleep(10000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        EbEnterpriseEntity enterpriseEntity = ebEnterpriseService.getEnterpriseByOrgId(organizationsEntity.getId());
        log.info("开始同步部门");
        syncResult.append("\n开始同步部门");
        List<BcxDepartmentResponse> departmentTree = tenantDepartmentsService.getDepartmentTree(organizationsEntity.getId(), null);
        Map<String, Long> frameIdMap = ebFrameService.frameCreateAll(enterpriseEntity.getId(), departmentTree, 0L, syncResult);
        syncResult.append("\n同步部门完成，部门数量：").append(frameIdMap.size());
        log.info("新增考勤和oa角色");
        Long oaAndAttendanceRoleId = ebEnterpriseRoleService.createGetOaAndAttendanceRole(enterpriseEntity.getId());
        log.info("查询部门-员工关联关系列表");
        List<TenantDepartmentEmployeeRelationsEntity> departmentEmployeeRelationsEntityList = tenantDepartmentEmployeeRelationsService.getListByDepartmentIdList(frameIdMap.keySet());
        Map<String, List<TenantDepartmentEmployeeRelationsEntity>> employeeDepartmentMap = departmentEmployeeRelationsEntityList.stream().collect(Collectors.groupingBy(TenantDepartmentEmployeeRelationsEntity::getEmployeeId));
        for (BcxEmployeeResponse response : employeeResponseList) {
            //赛演的分组成员，分组就是部门
            if (StringUtils.isNotBlank(response.getGroupIdsJson())) {
                List<TenantDepartmentEmployeeRelationsEntity> groupMemberRelationList = response.getGroupMemberRelationList();
                if (CollectionUtil.isNotEmpty(groupMemberRelationList)) {
                    employeeDepartmentMap.put(response.getId(), groupMemberRelationList);
                }
            }
        }
        log.info("用户加入企业");
        syncResult.append(ebAdminService.userJoinBatch(organizationsEntity.getId(), enterpriseEntity.getId(), employeeResponseList, oaAndAttendanceRoleId,
                employeeDepartmentMap));
        log.info("同步驻勤点");
        List<TlkAttendanceSiteBaseInfoEntity> attendanceSiteBaseInfoEntityList = tlkAttendanceSiteBaseInfoService.getEffectiveListByOrganizationId(organizationsEntity.getId());
        if (CollectionUtil.isNotEmpty(attendanceSiteBaseInfoEntityList)) {
            syncResult.append(this.syncAttendanceSite(enterpriseEntity.getId(), attendanceSiteBaseInfoEntityList));
        }
        log.info("初始化审批模版");
        syncResult.append("\n初始化审批模版");
        ebApproveService.initApprove(adminEmployeeId, enterpriseEntity.getId());

        syncResult.append("\n同步消息通知模版");
        this.syncMessageNoticeTemplate(adminEmployeeId, organizationsEntity.getId());

        syncResult.append("\n核对默认班次");
        ebAttendanceShiftService.checkDefaultShift(enterpriseEntity.getId(), adminEmployeeId);
        return syncResult.toString();
    }

    /**
     * description：同步用户
     * author：linchunpeng
     * date：2025/1/17
     */
    public void syncUser(@RequestBody @Validated UserSyncRequest request) {
        for (String userId : request.getTenantUserIdList()) {
            TenantUsersEntity tenantUsersEntity = tenantUsersService.getById(userId);
            AssertUtils.isNotNullObject(tenantUsersEntity, "用户不存在");
            List<TenantUserCredentialsEntity> credentialsEntityList = tenantUserCredentialsService.getByTenantUserId(tenantUsersEntity.getId());
            AssertUtils.isNotEmptyList(credentialsEntityList, "用户的证件为空");
            String cardNo = credentialsEntityList.get(0).getNumber();
            String headPhoto = credentialsEntityList.get(0).getHeadPhoto();
            String cardPhoto = credentialsEntityList.get(0).getHeadPhoto();
            for (TenantUserCredentialsEntity credentials : credentialsEntityList) {
                if (credentials.getCredentialType() == 0) {
                    //优先身份证
                    cardNo = credentials.getNumber();
                    headPhoto = credentials.getHeadPhoto();
                    cardPhoto = credentials.getHeadPhoto();
                    break;
                }
            }
            if (StringUtils.isNotBlank(tenantUsersEntity.getOneInchColorWhitePhoto())) {
                //一寸照不为空，以一寸照为准
                headPhoto = tenantUsersEntity.getOneInchColorWhitePhoto();
            }
            if (StringUtils.isNotBlank(tenantUsersEntity.getOneInchColorWhitePhoto())) {
                //活体采集照片不为空，以活体采集照片为准
                headPhoto = tenantUsersEntity.getCertificateImage();
            }
            ebUserService.userCreate(tenantUsersEntity.getId(), cardNo, cardPhoto, headPhoto, tenantUsersEntity.getName(), tenantUsersEntity.getTelephone());
        }
    }


    /**
     * description：同步部门
     * author：linchunpeng
     * date：2025/1/17
     */
    public void syncDepartment(@RequestBody @Validated DepartmentSyncRequest request) {
        AssertUtils.isTrue(organizationOpenService.isOpenAndEffective(request.getOrganizationId(), OpenType.TLJ.getCode()), "该组织未开通，或者开通的权限已到期");
        EbEnterpriseEntity enterpriseEntity = ebEnterpriseService.getEnterpriseByOrgId(request.getOrganizationId());
        List<BcxDepartmentResponse> departmentTree = tenantDepartmentsService.getDepartmentTree(request.getOrganizationId(), request.getDepartmentIdList());
        ebFrameService.frameCreateAll(enterpriseEntity.getId(), departmentTree, 0L, new StringBuffer());
    }

    /**
     * description：同步员工
     * author：linchunpeng
     * date：2025/1/17
     */
    public void syncEmployee(@RequestBody @Validated EmployeeSyncRequest request) {
        AssertUtils.isTrue(organizationOpenService.isOpenAndEffective(request.getOrganizationId(), OpenType.TLJ.getCode()), "该组织未开通，或者开通的权限已到期");
        //先同步用户
        List<BcxEmployeeResponse> employeeResponseList = tenantEmployeesService.getEmployeeResponseListByEmployeeIdList(request.getEmployeeIdList(), null);
        for (BcxEmployeeResponse employeeResponse : employeeResponseList) {
            ebUserService.userCreate(employeeResponse.getUserId(), employeeResponse.getCredentialNumber(),
                    employeeResponse.getCardPhoto(), employeeResponse.getHeadPhoto(),
                    employeeResponse.getEmployeeName(), employeeResponse.getTelephone());
        }
        //查询部门-员工关联关系列表
        List<TenantDepartmentEmployeeRelationsEntity> departmentEmployeeRelationsEntityList = tenantDepartmentEmployeeRelationsService.getListByEmployeeIdList(request.getEmployeeIdList());
        Map<String, List<TenantDepartmentEmployeeRelationsEntity>> employeeDepartmentMap = departmentEmployeeRelationsEntityList.stream().collect(Collectors.groupingBy(TenantDepartmentEmployeeRelationsEntity::getEmployeeId));

        Long entid = ebEnterpriseService.getIdByOrgId(request.getOrganizationId());
        Long oaAndAttendanceRoleId = ebEnterpriseRoleService.createGetOaAndAttendanceRole(entid);
        //用户加入企业
        for (BcxEmployeeResponse employeeResponse : employeeResponseList) {
            ebAdminService.userJoin(request.getOrganizationId(), entid, employeeResponse.getUserId(), employeeResponse.getId(),
                    employeeDepartmentMap.get(employeeResponse.getId()), null, null);
            ebAdminService.setAdminOtherInfo(employeeResponse.getId(), oaAndAttendanceRoleId, employeeResponse.getCardPhoto(),
                    null, employeeResponse.getIsDomainAdmin(), 0);
        }
    }

   /**
    * description：同步驻勤点和人员
    * author：linchunpeng
    * date：2025/1/17
    */
    public void syncAttendanceSite(@RequestBody AttendanceSiteSyncRequest request) {
        AssertUtils.isTrue(organizationOpenService.isOpenAndEffective(request.getOrganizationId(), OpenType.TLJ.getCode()), "该组织未开通，或者开通的权限已到期");
        EbEnterpriseEntity enterpriseEntity = ebEnterpriseService.getEnterpriseByOrgId(request.getOrganizationId());
        List<TlkAttendanceSiteBaseInfoEntity> attendanceSiteBaseInfoEntityList = new ArrayList<>();
        if (StringUtils.isNotBlank(request.getAttendanceSiteId())) {
            TlkAttendanceSiteBaseInfoEntity attendanceSiteBaseInfoEntity = tlkAttendanceSiteBaseInfoService.getById(request.getAttendanceSiteId());
            AssertUtils.isTrue(enterpriseEntity.getUniqued().equals(attendanceSiteBaseInfoEntity.getDomainid()), "组织id与驻勤点id不匹配");
            AssertUtils.isTrue(DateUtil.offsetDay(attendanceSiteBaseInfoEntity.getItemAttendanceEndDate(), 1).after(new Date()), "不是有效的驻勤点");
            attendanceSiteBaseInfoEntityList.add(attendanceSiteBaseInfoEntity);
        } else {
            attendanceSiteBaseInfoEntityList = tlkAttendanceSiteBaseInfoService.getEffectiveListByOrganizationId(request.getOrganizationId());
        }
        this.syncAttendanceSite(enterpriseEntity.getId(), attendanceSiteBaseInfoEntityList);
    }

    /**
     * description：同步驻勤点和人员
     * author：linchunpeng
     * date：2025/1/10
     */
    private String syncAttendanceSite(Long entid, List<TlkAttendanceSiteBaseInfoEntity> attendanceSiteBaseInfoEntityList) {
        StringBuffer syncResult = new StringBuffer(64);
        log.info("同步驻勤点，驻勤点数量：{}", attendanceSiteBaseInfoEntityList.size());
        syncResult.append("\n同步驻勤点，驻勤点数量：").append(attendanceSiteBaseInfoEntityList.size());
        //驻勤人员
        List<TlkAttendanceSitePersonInfoEntity> personInfoEntityList = tlkAttendanceSitePersonInfoService.getByAttendanceSiteIdList(attendanceSiteBaseInfoEntityList.stream()
                .map(TlkAttendanceSiteBaseInfoEntity::getId).collect(Collectors.toList()));
        //驻勤人员-考勤组数量 map
        Map<String, List<TlkAttendanceSitePersonInfoEntity>> personInfoCountMap = personInfoEntityList.stream().collect(Collectors.groupingBy(TlkAttendanceSitePersonInfoEntity::getItemSecurityId));
        //驻勤点-驻勤人员 map
        Map<String, List<TlkAttendanceSitePersonInfoEntity>> personInfoListMap = personInfoEntityList.stream().collect(Collectors.groupingBy(TlkAttendanceSitePersonInfoEntity::getItemAttendanceSiteId));
        List<EbAttendanceGroupMemberEntity> memberEntityList = new ArrayList<>();
        //同步驻勤点
        for (TlkAttendanceSiteBaseInfoEntity attendanceSiteBaseInfoEntity : attendanceSiteBaseInfoEntityList) {
            Long groupId = ebAttendanceGroupService.attendanceGroupCreate(entid, attendanceSiteBaseInfoEntity.getId(),
                    attendanceSiteBaseInfoEntity.getItemAttendanceSiteName(), attendanceSiteBaseInfoEntity.getItemAttendanceSiteFullAddress(),
                    attendanceSiteBaseInfoEntity.getItemAttendanceSiteLongitude(), attendanceSiteBaseInfoEntity.getItemAttendanceSiteLatitude(),
                    attendanceSiteBaseInfoEntity.getItemPerformRange(), attendanceSiteBaseInfoEntity.getItemPrincipal(),
                    attendanceSiteBaseInfoEntity.getItemAttType());

            List<TlkAttendanceSitePersonInfoEntity> personList = personInfoListMap.get(attendanceSiteBaseInfoEntity.getId());
            //同步驻勤人员
            if (CollectionUtil.isNotEmpty(personList)) {
                //取出智能人事职员id列表
                List<String> employeeIdList = personList.stream().map(TlkAttendanceSitePersonInfoEntity::getItemSecurityId).collect(Collectors.toList());
                List<EbAdminEntity> adminEntityList = ebAdminService.getListByEmployeeIdList(employeeIdList, entid);
                Map<String, String> employeeIdAndOutEmployeeIdMap = new HashMap<>();
                if (CommonUtils.isLinBao(attendanceSiteBaseInfoEntity.getItemAttType())) {
                    //临保
                    List<String> outEmployeeIdList = new ArrayList<>();
                    for (String employeeId : employeeIdList) {
                        if (adminEntityList.stream().noneMatch(a -> a.getUniqued().equals(employeeId))) {
                            //不在本企业职员里面，说明是外部企业人员
                            outEmployeeIdList.add(employeeId);
                        }
                    }
                    if (CollectionUtil.isNotEmpty(outEmployeeIdList)) {
                        for (String employeeId : outEmployeeIdList) {
                            log.info("临保项目，同步驻勤人员信息，人员是外部企业人员");
                            List<BcxEmployeeResponse> employeeResponseList = tenantEmployeesService.getEmployeeResponseListByEmployeeIdList(Collections.singletonList(employeeId), 0);
                            if (CollectionUtil.isNotEmpty(employeeResponseList)) {
                                BcxEmployeeResponse outEmployee = employeeResponseList.get(0);
                                EbUserEntity ebUserEntity = ebUserService.getUserByUserId(outEmployee.getUserId());
                                String outEmployeeId = outEmployee.getUserId() + "_" + attendanceSiteBaseInfoEntityList.get(0).getDomainid();
                                employeeIdAndOutEmployeeIdMap.put(employeeId, outEmployeeId);
                                if (ebUserEntity == null) {
                                    log.info("同步驻勤人员信息，人员是外部企业人员,还未同步过陀螺匠，所以需要同步用户信息");
                                    ebUserService.userCreate(outEmployee.getUserId(), outEmployee.getCredentialNumber(),
                                            outEmployee.getCardPhoto(), outEmployee.getHeadPhoto(),
                                            outEmployee.getEmployeeName(), outEmployee.getTelephone());
                                }
                                Long oaAndAttendanceRoleId = ebEnterpriseRoleService.createGetOaAndAttendanceRole(entid);
                                ebAdminService.userJoin(attendanceSiteBaseInfoEntityList.get(0).getDomainid(), entid, outEmployee.getUserId(), outEmployeeId,
                                        null, null, null);
                                //同步员工其他信息
                                adminEntityList.add(ebAdminService.setAdminOtherInfo(outEmployeeId, oaAndAttendanceRoleId, outEmployee.getCardPhoto(), null, 0, 1));
                            }
                        }
                    }
                }

                //智能人事职员id 与 陀螺匠职员id集合
                Map<String, Long> employeeIdAndAdminIdMap = adminEntityList.stream().collect(Collectors.toMap(EbAdminEntity::getUniqued, EbAdminEntity::getId));

                for (TlkAttendanceSitePersonInfoEntity personInfoEntity : personList) {
                    boolean isJoinGroup = true;
                    //如果当前驻勤点是临保类型，需要判断人员是否报到，报到了才需要加入考勤组
                    if (CommonUtils.isLinBao(attendanceSiteBaseInfoEntity.getItemAttType())
                            && personInfoEntity.getItemReportingTime() == null
                            && (personInfoEntity.getItemPushAttendance() == null || personInfoEntity.getItemPushAttendance() == 0)) {
                        isJoinGroup = false;
                    }
                    if (isJoinGroup) {
                        Long adminId = employeeIdAndAdminIdMap.get(personInfoEntity.getItemSecurityId());
                        if (adminId == null) {
                            adminId = employeeIdAndAdminIdMap.get(employeeIdAndOutEmployeeIdMap.get(personInfoEntity.getItemSecurityId()));
                        }
                        if (groupId != null && adminId != null) {
                            EbAttendanceGroupMemberEntity memberEntity = ebAttendanceGroupMemberService.addMember(entid, groupId, adminId, 0, false);
                            if (memberEntity != null) {
                                memberEntityList.add(memberEntity);
                            }
                            List<TlkAttendanceSitePersonInfoEntity> personInfoCountList = personInfoCountMap.get(personInfoEntity.getItemSecurityId());
                            if (personInfoCountList.size() > 1
                                    && CommonUtils.isChanngBao(attendanceSiteBaseInfoEntity.getItemAttType())) {
                                //有归属多个驻勤点，如果本驻勤点类型是0或者空(常驻的，不是临保，优先加入临保)，且临保已报到，那就需要增加排除考勤成员
                                boolean isNeedExclude = false;
                                for (TlkAttendanceSitePersonInfoEntity info : personInfoCountList) {
                                    if (!info.getItemAttendanceSiteId().equals(attendanceSiteBaseInfoEntity.getId())
                                            && (info.getItemReportingTime() != null
                                            || (info.getItemPushAttendance() != null && info.getItemPushAttendance() == 1))) {
                                        //不是当前驻勤点 && (报到时间不为空 || 推送到考勤)
                                        isNeedExclude = true;
                                        break;
                                    }
                                }
                                if (isNeedExclude) {
                                    EbAttendanceGroupMemberEntity excludeMemberEntity = ebAttendanceGroupMemberService.addMember(entid, groupId, adminId, 1, false);
                                    if (excludeMemberEntity != null) {
                                        memberEntityList.add(excludeMemberEntity);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        if (CollectionUtil.isNotEmpty(memberEntityList)) {
            log.info("同步驻勤人员，驻勤人员数量：{}", memberEntityList.size());
            syncResult.append("\n同步驻勤人员，驻勤人员数量：").append(memberEntityList.size());
            ebAttendanceGroupMemberService.saveOrUpdateBatch(memberEntityList);
        }
        return syncResult.toString();
    }

    /**
     * description：初始化oa审批
     * author：linchunpeng
     * date：2025/3/6
     */
    public void initOaTemp(OrganizationSyncRequest request) {
        AssertUtils.isTrue(organizationOpenService.isOpenAndEffective(request.getOrganizationId(), OpenType.TLJ.getCode()), "该组织未开通，或者开通的权限已到期");
        TenantOrganizationsEntity organizationsEntity = tenantOrganizationsService.getById(request.getOrganizationId());
        AssertUtils.isNotNullObject(organizationsEntity, "企业不存在");

        TenantEmployeesEntity adminEmployeeEntity = tenantEmployeesService.getAdminByOrgId(request.getOrganizationId());
        EbEnterpriseEntity enterpriseEntity = ebEnterpriseService.getEnterpriseByOrgId(request.getOrganizationId());
        ebApproveService.initApprove(adminEmployeeEntity.getId(), enterpriseEntity.getId());
    }


    /**
     * description：同步消息通知模版
     * author：linchunpeng
     * date：2025/3/7
     */
    public void syncMessageNoticeTemplate(String employeeId, String orgId) {
        EbAdminEntity ebAdmin = ebAdminService.getByEmployeeId(employeeId);
        EbUserEntity ebUser = ebUserService.getUserByUserId(ebAdmin.getUid());
        long timeSecond = System.currentTimeMillis()/1000;
        Map<String, String> headerMap = TuoluojiangUtil.getHeaderMap("user/login".concat("POST") + timeSecond);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("uid", ebAdmin.getUid());
        paramMap.put("ent_id", orgId);
        paramMap.put("card_id", ebUser.getCardId());
        String result = HttpUtil.post(syncConfig.getApiHost().getTuoluojiang().concat("/api/ent/internal/user/login"), JSONObject.toJSONString(paramMap), headerMap, "test".equals(activeFile));
        //{"message":"ok","status":200,"data":{"token":"eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpc3MiOiJodHRwOi8vc2Fhcy50dW9sdW9qaWFuZy5jb20iLCJpYXQiOjE3NDEzMzE0NjEsImV4cCI6MTc0MjE5NTQ2MSwibmJmIjoxNzQxMzMxNDYxLCJqdGkiOiJkOGE0cHlJTFprNzdBT3lHIiwic3ViIjoiMSIsInBydiI6ImI5NDYyMjZmNWU0YmJjN2FiN2FkYjQzNjg4MmY5NWE0MGM2ODA1YjEiLCJlbnRJZCI6MSwidGltZXN0YW1wcyI6MTc0MTMzMTQ2MSwiZW50aWQiOjF9.3zybp-bCz32F14BYt2qoEk5SM4G5Na8HzkyKrj16WlE","token_type":"bearer","expires_in":864000},"tips":1}
        JSONObject resultObject = JSONObject.parseObject(result);
        if ("ok".equals(resultObject.getString("message"))) {
            JSONObject data = resultObject.getJSONObject("data");
            String token = data.getString("token");

            Map<String, String> headerMap2 = new HashMap<>();
            headerMap2.put("Authorization", "Bearer " + token);
            Map<String, Object> paramMap2 = new HashMap<>();
            HttpUtil.put(syncConfig.getApiHost().getTuoluojiang().concat("/api/ent/system/message/sync"), JSONObject.toJSONString(paramMap2), headerMap2, "test".equals(activeFile));
        }
    }

    /**
     * 同步活动承办方（通过项目）
     */
    @Transactional
    public void tljSyncDomainByProject(String orgId) {
        //开通陀螺匠
        TenantOrganizationsEntity organizationsEntity = tenantOrganizationsService.getById(orgId);
        OrganizationOpenRequest request = new OrganizationOpenRequest();
        request.setOrganizationId(orgId);
        request.setOrganizationName(organizationsEntity.getName());
        request.setOpenType(1);
        Date now = new Date();
        Date endTime = tlkEventProjectService.getEndTimeByDomainId(orgId, now);
        request.setEndTime(endTime);
        request.setOpenDay((int) DateUtil.between(now, endTime, DateUnit.DAY));
        organizationOpenService.openThird(request);
    }

    /**
     * 同步活动承办方（通过项目）
     */
    @Transactional
    public void tljSyncDomainByTeam(String orgId) {
        //开通陀螺匠
        TenantOrganizationsEntity organizationsEntity = tenantOrganizationsService.getById(orgId);
        OrganizationOpenRequest request = new OrganizationOpenRequest();
        request.setOrganizationId(orgId);
        request.setOrganizationName(organizationsEntity.getName());
        request.setOpenType(1);
        Date now = new Date();
        Date endTime = tlkEventTeamService.getEndTimeByTeamId(orgId, now);
        request.setEndTime(endTime);
        request.setOpenDay((int) DateUtil.between(now, endTime, DateUnit.DAY));
        organizationOpenService.openThird(request);
    }
}
