package com.bcxin.sync.consumer;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.sync.common.CommonConstant;
import com.bcxin.sync.common.emus.DataOperationType;
import com.bcxin.sync.common.emus.OpenType;
import com.bcxin.sync.common.utils.KafkaMessageUtil;
import com.bcxin.sync.dtos.kafka.message.DepartmentSyncMessage;
import com.bcxin.sync.entity.jvs.SysTenantEntity;
import com.bcxin.sync.service.datasync.OrganizationOpenService;
import com.bcxin.sync.service.jvs.SysDeptService;
import com.bcxin.sync.service.jvs.SysTenantService;
import com.bcxin.sync.service.tuoluojiang.EbEnterpriseService;
import com.bcxin.sync.service.tuoluojiang.EbFrameService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;

/**
 * 部门kafka消息监听器
 */
@Slf4j
@Component
public class DepartmentKafkaConsumer {

    @Autowired
    private EbFrameService ebFrameService;
    @Autowired
    private EbEnterpriseService ebEnterpriseService;
    @Autowired
    private SysDeptService sysDeptService;
    @Autowired
    private SysTenantService sysTenantService;
    @Autowired
    private OrganizationOpenService organizationOpenService;

    @KafkaListener(id = "bcx_data_sync_listener_tenant_department_to_tlj",
            topics = CommonConstant.DEPARTMENT_KAFKA_TOPIC,
            groupId = "bcx_data_sync_dept_to_tlj")
    public void bcxDataSyncDeptToTlj(ConsumerRecord<String, String> record, Acknowledgment ack) {
        DepartmentSyncMessage department = KafkaMessageUtil.transferKafkaDeptRecord2Message(record);
        log.info("同步陀螺匠，kafka监听tenant-department表，数据转化为：{}", JSONObject.toJSONString(department));
        try {
            boolean openAndEffective = organizationOpenService.isOpenAndEffective(department.getOrganizationId(), OpenType.TLJ.getCode());
            if (openAndEffective) {
                Long entid = ebEnterpriseService.getIdByOrgId(department.getOrganizationId());
                if (entid != null) {
                    //有开通陀螺匠，才需要同步部门信息
                    log.info("同步陀螺匠，组织id：{}，有开通陀螺匠，所以需要同步部门信息", department.getOrganizationId());
                    if (department.getType().getCode() == DataOperationType.DELETE.getCode()) {
                        //删除
                        log.info("同步陀螺匠，组织id：{}，删除部门", department.getOrganizationId());
                        ebFrameService.frameDelete(department.getDepartmentId(), department.getAfterDepartmentName());
                    } else {
                        log.info("同步陀螺匠，组织id：{}，新增或修改部门", department.getOrganizationId());
                        Long pid = null;
                        if (StringUtils.isNotBlank(department.getAfterParentId())) {
                            pid = ebFrameService.getIdByDepartmentId(department.getAfterParentId());
                        }
                        ebFrameService.frameCreate(entid, department.getDepartmentId(), department.getAfterDepartmentName(), pid);
                    }
                }
            } else {
                log.info("同步陀螺匠，组织id：{}，没开通陀螺匠，所以不需要同步部门信息", department.getOrganizationId());
            }
        } catch (Exception e) {
            log.error("同步陀螺匠，kafka监听tenant-department表，异常：{}", e.getMessage(), e);
        }
        ack.acknowledge();
    }

    @KafkaListener(id = "bcx_data_sync_listener_tenant_department_to_jvs",
            topics = CommonConstant.DEPARTMENT_KAFKA_TOPIC,
            groupId = "bcx_data_sync_dept_to_jvs")
    public void bcxDataSyncDeptToJvs(ConsumerRecord<String, String> record, Acknowledgment ack) {
        DepartmentSyncMessage department = KafkaMessageUtil.transferKafkaDeptRecord2Message(record);
        log.info("同步JVS，kafka监听tenant-department表，数据转化为：{}", JSONObject.toJSONString(department));
        try {
            boolean openAndEffective = organizationOpenService.isOpenAndEffective(department.getOrganizationId(), OpenType.JVS.getCode());
            if (openAndEffective) {
                SysTenantEntity sysTenant = sysTenantService.getById(department.getOrganizationId());
                if (sysTenant != null) {
                    //有开通JVS，才需要同步部门信息
                    log.info("同步JVS，组织id：{}，有开通JVS，所以需要同步部门信息", department.getOrganizationId());
                    if (department.getType().getCode() == DataOperationType.DELETE.getCode()) {
                        //删除
                        log.info("同步JVS，组织id：{}，删除部门", department.getOrganizationId());
                        sysDeptService.delete(department.getDepartmentId(), department.getAfterDepartmentName());
                    } else {
                        log.info("同步JVS，组织id：{}，新增或修改部门", department.getOrganizationId());
                        sysDeptService.create(sysTenant.getId(), department.getDepartmentId(), department.getAfterDepartmentName(), department.getAfterParentId());
                    }
                }
            } else {
                log.info("同步JVS，组织id：{}，没开通JVS，所以不需要同步部门信息", department.getOrganizationId());
            }
        } catch (Exception e) {
            log.error("同步JVS，kafka监听tenant-department表，异常：{}", e.getMessage(), e);
        }
        ack.acknowledge();
    }
}
