package com.bcxin.platform.service.meeting;

import cn.hutool.core.text.StrSpliter;
import com.alibaba.fastjson.JSON;
import com.bcxin.platform.common.constant.DictConst;
import com.bcxin.platform.common.exception.PlatFormBusinessException;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.common.utils.StringUtils;
import com.bcxin.platform.domain.company.PerBaseInfo;
import com.bcxin.platform.domain.meeting.ComHuaweiMeetMatch;
import com.bcxin.platform.domain.meeting.PerHuaweiMeetMatch;
import com.bcxin.platform.dto.meeting.PerHuaweiMeetMatchDto;
import com.bcxin.platform.mapper.company.PerBaseInfoMapper;
import com.bcxin.platform.mapper.meeting.ComHuaweiMeetMatchMapper;
import com.bcxin.platform.mapper.meeting.PerHuaweiMeetMatchMapper;
import com.bcxin.platform.service.cache.TaskCacheService;
import com.bcxin.platform.service.common.CommonService;
import com.bcxin.platform.util.DateUtils;
import com.bcxin.platform.util.constants.CommonConst;
import com.bcxin.platform.util.constants.HuaweiMeetConst;
import com.bcxin.platform.util.constants.SMSConst;
import com.bcxin.platform.util.huawei.HttpBuildUtil;
import com.bcxin.platform.util.huawei.RestRequest;
import com.bcxin.platform.util.huawei.RestResponse;
import com.github.pagehelper.util.StringUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;

/**
 * 人员华为云会议映射
 *
 * @author llc
 * @date 2019-12-03
 */
@Service
@Transactional("transactionManager")
public class PerHuaweiMeetMatchServiceImpl implements PerHuaweiMeetMatchService {

    /**
     * 日志对象
     */
    protected Logger logger = LoggerFactory.getLogger(getClass());

    @Resource
    private PerHuaweiMeetMatchMapper perHuaweiMeetMatchMapper;


    @Resource
    private ComHuaweiMeetMatchMapper comHuaweiMeetMatchMapper;

    @Resource
    private PerBaseInfoMapper perBaseInfoMapper;

    @Resource
    private TaskCacheService taskCacheService;

    @Resource
    private CommonService commonService;

    /**
     * 华为云鉴权
     *
     * @param perHuaweiMeetMatch
     * @return
     * @auth llc
     * @date 2019-12-10
     */
    @Override
    public Result getHuaweiAccessToken(PerHuaweiMeetMatch perHuaweiMeetMatch) throws PlatFormBusinessException {

        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(perHuaweiMeetMatch.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }
        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), perHuaweiMeetMatch.getPerId());

        Map<String, String> resultMap = new HashMap<>();
        resultMap.put("huaweiUserName", meetMatch.getHuaweiUserName());
        resultMap.put("huaweiPassWord", meetMatch.getHuaweiPassWord());
        resultMap.put("huaweiUserId", meetMatch.getHuaweiUserId());
        resultMap.put("huaweiSipName", meetMatch.getHuaweiSipName());
        resultMap.put("accessToken", accessToken);
        return Result.success(Result.SUCCESS_QUERY_MSG, resultMap);
    }

    /**
     * 新增用户
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2019-12-10
     */
    @Override
    public Result addUser(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {

        if (dto.getPerId() == null) {
            return Result.fail("人员ID不能为空");
        }
        PerHuaweiMeetMatch perHuaweiMeetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(dto.getPerId());
        if (perHuaweiMeetMatch != null) {
            return Result.fail("此人已有会议权限，无需重复添加");
        }
        /*** 由于华为云新增删除用户需要管理权限***/
        ComHuaweiMeetMatch comHuaweiMeetMatch = comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(dto.getComId());
        if (comHuaweiMeetMatch == null) {
            return Result.fail("对不起，您所在的企业不存在会议使用权限");
        }

        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getComManageHuaweiMeetAccessToken(comHuaweiMeetMatch.getHuaweiManageUserName(), comHuaweiMeetMatch.getHuaweiManagePassWord(), dto.getComId());

        /*** 调用华为云会议接口 ***/
        dto.setHuaweiUserName("user" + dto.getPerId());
        dto.setHuaweiPassWord("pwd" + dto.getPerId().toString().substring(dto.getPerId().toString().length() - 6));

        PerBaseInfo perBaseInfo = new PerBaseInfo();
        perBaseInfo.setPerId(dto.getPerId());
        perBaseInfo = perBaseInfoMapper.selectPerBaseInfoById(perBaseInfo.getPerId());
        Map<String, Object> map = new HashMap<>();
        map.put("name", perBaseInfo.getName());
        map.put("account", dto.getHuaweiUserName());
        map.put("pwd", dto.getHuaweiPassWord());
        map.put("phone", perBaseInfo.getMobilePhone());
        map.put("sendNotify", CommonConst.N); // 不发送邮件和短信通知。
        map.put("country", "chinaPR");  // 当填写手机号时 “country”参数必填。
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.ADD_USER;
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, JSON.toJSONString(map), null);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        Map<String, Object> resultMap = HttpBuildUtil.getMapData(response);
        dto.setHuaweiUserId(resultMap.get("id").toString());
        dto.setHuaweiSipName(resultMap.get("sipNum").toString());

        /*** 插入人员华为云会议映射表 ***/
        perHuaweiMeetMatchMapper.insertPerHuaweiMeetMatch(dto);

        return Result.success("添加成功");
    }


    /**
     * 设置监管机构负责人
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2020-12-03
     */
    @Override
    public Result setRegulatorMeetCharge(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {
        if (dto.getPerId() == null) {
            return Result.fail("人员ID不能为空");
        }
        PerHuaweiMeetMatch perHuaweiMeetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(dto.getPerId());
        if (perHuaweiMeetMatch == null) {
            return Result.fail("未查找到人员消息");
        }
        perHuaweiMeetMatch.setRegulatorMeetChargeType(dto.getRegulatorMeetChargeType());
        perHuaweiMeetMatchMapper.updatePerHuaweiMeetMatch(perHuaweiMeetMatch);
        return Result.success("设置成功");
    }

    /**
     * 取消监管机构负责人
     * @param dto
     * @auth llc
     * @date 2020-12-03
     * @return
     */
    @Override
    public Result cancelRegulatorMeetCharge(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {
        if (dto.getPerId() == null) {
            return Result.fail("人员ID不能为空");
        }
        PerHuaweiMeetMatch perHuaweiMeetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(dto.getPerId());
        if (perHuaweiMeetMatch == null) {
            return Result.fail("未查找到人员消息");
        }
        perHuaweiMeetMatch.setRegulatorMeetChargeType(CommonConst.N);
        perHuaweiMeetMatchMapper.updatePerHuaweiMeetMatch(perHuaweiMeetMatch);
        return Result.success("取消成功");
    }

    /**
     * 批量删除用户
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2019-12-12
     */
    @Override
    public Result batchDeleteUser(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {

        if (StringUtils.isEmpty(dto.getPerIds())) {
            return Result.fail("人员ID不能为空");
        }

        List<String> perIdList = StrSpliter.split(dto.getPerIds(), CommonConst.COMMA, 0, true, true);

        List<PerHuaweiMeetMatch> list = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerIdList(perIdList);

        if (list != null && list.size() > 0) {
            List<String> huaweiUserNameList = new ArrayList<>();
            List<Long> existsPerIdList = new ArrayList<>();
            for (PerHuaweiMeetMatch perHuaweiMeetMatch : list) {
                huaweiUserNameList.add(perHuaweiMeetMatch.getHuaweiUserName());
                existsPerIdList.add(perHuaweiMeetMatch.getPerId());
            }
            /*** 由于华为云新增删除用户需要管理权限***/
            ComHuaweiMeetMatch comHuaweiMeetMatch = comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(dto.getComId());
            if (comHuaweiMeetMatch == null) {
                return Result.fail("对不起，您所在的企业不存在会议使用权限");
            }
            /*** 获取华为云会议的accessToken ***/
            String accessToken = taskCacheService.getComManageHuaweiMeetAccessToken(comHuaweiMeetMatch.getHuaweiManageUserName(), comHuaweiMeetMatch.getHuaweiManagePassWord(), dto.getComId());

            /*** 调用华为云会议接口 ***/
            Map<String, String> headMap = new HashMap<>();
            headMap.put("X-Access-Token", accessToken);
            String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.BATCH_DELETE_USER;
            RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, JSON.toJSONString(huaweiUserNameList), null);
            RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
            //如果返回报文处理resultMsg有值，说明有失败信息
            String resultMsg = HttpBuildUtil.failMsg(response);
            if (StringUtils.isNotEmpty(resultMsg)) {
                return Result.fail(resultMsg);
            }
            /*** 插入人员华为云会议映射表 ***/
            perHuaweiMeetMatchMapper.batchDeletePerHuaweiMeetMatch(existsPerIdList);
            return Result.success("删除成功");
        } else {
            return Result.success("删除成功");
        }
    }

    /**
     * 离职自动批量删除用户
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2019-12-16
     */
    @Override
    public Result leaveJonBatchDeleteUser(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {

        if (StringUtils.isEmpty(dto.getPerIds())) {
            return Result.fail("人员ID不能为空");
        }

        List<String> perIdList = StrSpliter.split(dto.getPerIds(), CommonConst.COMMA, 0, true, true);

        /*** 企业开通了华为云会议才需要自动清除人员 ***/
        ComHuaweiMeetMatch comHuaweiMeetMatch = comHuaweiMeetMatchMapper.getComHuaweiMeetMatchByComId(dto.getComId());

        if (comHuaweiMeetMatch != null) {

            List<PerHuaweiMeetMatch> list = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerIdList(perIdList);
            if (list != null && list.size() > 0) {
                List<String> huaweiUserNameList = new ArrayList<>();
                List<Long> existsPerIdList = new ArrayList<>();
                for (PerHuaweiMeetMatch perHuaweiMeetMatch : list) {
                    huaweiUserNameList.add(perHuaweiMeetMatch.getHuaweiUserName());
                    existsPerIdList.add(perHuaweiMeetMatch.getPerId());
                }
                /*** 获取华为云会议的accessToken ***/
                String accessToken = taskCacheService.getComManageHuaweiMeetAccessToken(comHuaweiMeetMatch.getHuaweiManageUserName(), comHuaweiMeetMatch.getHuaweiManagePassWord(), dto.getComId());

//            String authorization = Base64.getEncoder().encodeToString((accessToken).getBytes());

                /*** 调用华为云会议接口 ***/
                Map<String, String> headMap = new HashMap<>();
                headMap.put("X-Access-Token", accessToken);
                String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.BATCH_DELETE_USER;
                RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, JSON.toJSONString(huaweiUserNameList), null);
                RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
                //如果返回报文处理resultMsg有值，说明有失败信息
                String resultMsg = HttpBuildUtil.failMsg(response);
                if (StringUtils.isNotEmpty(resultMsg)) {
                    return Result.fail(resultMsg);
                }
                /*** 插入人员华为云会议映射表 ***/
                perHuaweiMeetMatchMapper.batchDeletePerHuaweiMeetMatch(existsPerIdList);
            }
        }
        return Result.success(Result.SUCCESS_QUERY_MSG);
    }


    /**
     * 查询会议列表
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2019-12-16
     */
    @Override
    public Result getHuaweiMeetList(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {

        if (dto.getPerId() == null) {
            return Result.fail("人员ID不能为空");
        }
        if (dto.getPageNumber() == null) {
            return Result.fail("第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            return Result.fail("每页多少条不能为空");
        }
        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(dto.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }

        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), meetMatch.getPerId());

        /*** 调用华为云会议接口 ***/

        Map<String, String> map = new HashMap<>();
        map.put("userUUID", meetMatch.getHuaweiUserId());
        Integer offset = (dto.getPageNumber() - 1) * dto.getPageSize();
        map.put("offset", offset.toString());
        map.put("limit", dto.getPageSize().toString());
        map.put("sortType", "DSC_StartTIME");
        if (StringUtil.isNotEmpty(dto.getKeyWord())) { // 关键字
            map.put("searchKey", dto.getKeyWord());
        }
        if (StringUtil.isNotEmpty(dto.getQueryConfMode())) { // 查询维度
            map.put("queryConfMode", dto.getQueryConfMode());
        }
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_MEET_LIST;

        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("GET", headMap, null, map);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, HttpBuildUtil.getMapData(response));
    }

    /**
     * 发送短信通知
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2019-12-12
     */
    @Override
    public Result sendHuaweiMeetSms(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {

        if (StringUtils.isEmpty(dto.getPhones())) {
            return Result.fail("手机号码不能为空");
        }
        List<String> phoneList = StrSpliter.split(dto.getPhones(), CommonConst.COMMA, 0, true, true);
        List<Map<String, String>> mList = Lists.newArrayList();
        String content = SMSConst.HUAWEI_MEET_NOTICE.replace("{meetTheme}", dto.getMeetTheme()).replace("{meetStartTime}", dto.getMeetStartTime())
                .replace("{meetId}", dto.getMeetId()).replace("{guestPwd}", dto.getGuestPwd()).replace("{chairPwd}", dto.getChairPwd());
        for (String mobile : phoneList) {
            Map<String, String> map = Maps.newHashMap();
            map.put("mobile", mobile);
            map.put("content", content);
            map.put("params", JSON.toJSONString(new String[]{dto.getMeetTheme(), dto.getMeetStartTime(), dto.getMeetId(), dto.getGuestPwd(), dto.getChairPwd(), dto.getMeetId()}));
            mList.add(map);
        }
        if (mList.size() > 0) {
            commonService.sendBatchSMS(mList, DictConst.SMSCODE_B11);
        }
        return Result.success(Result.SUCCESS_QUERY_MSG);
    }


    /**
     * 获取录播文件列表
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2020-02-03
     */
    @Override
    public Result getTranscribeFileList(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {

        if (dto.getPageNumber() == null) {
            return Result.fail("第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            return Result.fail("每页多少条不能为空");
        }
        if (StringUtils.isEmpty(dto.getStartTime())) {
            return Result.fail("查询开始时间不能为空");
        }
        if (StringUtils.isEmpty(dto.getEndTime())) {
            return Result.fail("查询截止时间不能为空");
        }
        Date startTime = null;
        Date endTime = null;
        try {
            startTime = DateUtils.parseDate(dto.getStartTime() + " 00:00:00", "yyyy-MM-dd HH:mm:ss");
            endTime = DateUtils.parseDate(dto.getEndTime() + " 23:59:59", "yyyy-MM-dd HH:mm:ss");
        } catch (Exception e) {
        }

        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(dto.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }

        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), meetMatch.getPerId());

        /*** 调用华为云会议接口 ***/

        Map<String, String> map = new HashMap<>();
        map.put("userUUID", meetMatch.getHuaweiUserId());
        map.put("offset", String.valueOf(dto.getPageNumber() - 1));
        map.put("limit", dto.getPageSize().toString());
        if (StringUtils.isNotEmpty(dto.getMeetTheme())) {
            map.put("searchKey", dto.getMeetTheme());
        }
        map.put("startDate", String.valueOf(startTime.getTime()));
        map.put("endDate", String.valueOf(endTime.getTime()));
        map.put("sortType", "DSC_StartTIME");
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_TRANSCRIBE_FILE_LIST;
        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("GET", headMap, null, map);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, HttpBuildUtil.getMapData(response));
    }

    /**
     * 批量删除录播文件
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2020-02-03
     */
    @Override
    public Result batchDeleteTranscribeFile(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {
        if (StringUtils.isEmpty(dto.getConfUUIDs())) {
            return Result.fail("会议UUID集合不能为空");
        }

        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(dto.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }

        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), meetMatch.getPerId());

        /*** 调用华为云会议接口 ***/

        Map<String, String> map = new HashMap<>();
        map.put("confUUIDs", dto.getConfUUIDs());
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.DELETE_TRANSCRIBE_FILE;

        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("DELETE", headMap, null, map);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success("删除成功");
    }

    /**
     * 查询会议详情
     *
     * @param dto
     * @return
     * @auth llc
     * @date 2020-10-13
     */
    @Override
    public Result getHuaweiMeetDetail(PerHuaweiMeetMatchDto dto) throws PlatFormBusinessException {

        if (StringUtils.isEmpty(dto.getConferenceId())) {
            return Result.fail("会议ID不能为空");
        }
        PerHuaweiMeetMatch meetMatch = perHuaweiMeetMatchMapper.getPerHuaweiMeetMatchByPerId(dto.getPerId());
        if (meetMatch == null) {
            return Result.fail("企业管理员还未把您添加到会议使用权限，请联系您所在企业管理员添加");
        }

        /*** 获取华为云会议的accessToken ***/
        String accessToken = taskCacheService.getHuaweiMeetAccessToken(meetMatch.getHuaweiUserName(), meetMatch.getHuaweiPassWord(), meetMatch.getPerId());

        /*** 调用华为云会议接口 ***/

        Map<String, String> map = new HashMap<>();
        map.put("conferenceID", dto.getConferenceId());
        map.put("userUUID", meetMatch.getHuaweiUserId());
        if (dto.getPageNumber() != null && dto.getPageNumber() != 0) {
            Integer offset = (dto.getPageNumber() - 1) * dto.getPageSize();
            map.put("offset", offset.toString());
        }
        if (dto.getPageSize() != null && dto.getPageSize() != 0) {
            map.put("limit", dto.getPageSize().toString());
        }
        if (StringUtil.isNotEmpty(dto.getKeyWord())) { // 关键字
            map.put("searchKey", dto.getKeyWord());
        }
        String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_MEET_DETAIL;

        Map<String, String> headMap = new HashMap<>();
        headMap.put("X-Access-Token", accessToken);
        RestRequest restRequest = HttpBuildUtil.buildRestRequest("GET", headMap, null, map);
        RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
        //如果返回报文处理resultMsg有值，说明有失败信息
        String resultMsg = HttpBuildUtil.failMsg(response);
        if (StringUtils.isNotEmpty(resultMsg)) {
            return Result.fail(resultMsg);
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, HttpBuildUtil.getMapData(response));
    }

}