package com.bcxin.platform.service.company.impl;

import cn.hutool.core.util.StrUtil;
import com.bcxin.platform.common.constant.Constants;
import com.bcxin.platform.common.core.text.Convert;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.DateUtils;
import com.bcxin.platform.common.utils.IdWorker;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.common.utils.StringUtils;
import com.bcxin.platform.common.utils.bean.BeanUtils;
import com.bcxin.platform.domain.company.ComBaseInfo;
import com.bcxin.platform.domain.company.ComDepart;
import com.bcxin.platform.dto.company.ComDepartDto;
import com.bcxin.platform.dto.company.ComDepartTreeDto;
import com.bcxin.platform.dto.company.OrgTreeDto;
import com.bcxin.platform.mapper.company.ComBaseInfoMapper;
import com.bcxin.platform.mapper.company.ComDepartMapper;
import com.bcxin.platform.mapper.company.PerOrgRelationMapper;
import com.bcxin.platform.service.company.ComDepartService;
import com.bcxin.platform.util.constants.CommonConst;
import com.github.pagehelper.util.StringUtil;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 部门Service业务层处理
 * 
 * @author task
 * @date 2021-04-09
 */
@Service
public class ComDepartServiceImpl implements ComDepartService {

    @Resource
    private ComDepartMapper comDepartMapper;

    @Resource
    private ComBaseInfoMapper comBaseInfoMapper;

    @Resource
    private PerOrgRelationMapper perOrgRelationMapper;

    @Resource
    private IdWorker idWorker;

    /**
     * 查询部门
     * 
     * @param departId 部门ID
     * @return 部门
     */
    @Override
    public ComDepart selectComDepartById(Long departId){
        return comDepartMapper.selectComDepartById(departId);
    }

    /**
     * 查询部门列表
     * 
     * @param comDepart 部门
     * @return 部门
     */
    @Override
    public List<ComDepart> selectComDepartList(ComDepart comDepart){
        return comDepartMapper.selectComDepartList(comDepart);
    }

    /**
     * 新增部门
     * 
     * @param comDepart 部门
     * @return 结果
     */
    @Override
    public int insertComDepart(ComDepart comDepart){
        comDepart.setDepartId(idWorker.nextId());
        comDepart.setCreateTime(DateUtils.getNowDate());
        return comDepartMapper.save(comDepart);
    }

    /**
     * 修改部门
     * 
     * @param comDepart 部门
     * @return 结果
     */
    @Override
    public int updateComDepart(ComDepart comDepart){
        ComDepart dbComDepart = comDepartMapper.selectComDepartByTlkId(comDepart.getTlkDepartId());
        if(dbComDepart == null){
            comDepart.setDepartId(idWorker.nextId());
            comDepart.setCreateTime(DateUtils.getNowDate());
            return comDepartMapper.save(comDepart);
        }else {
            comDepart.setUpdateTime(DateUtils.getNowDate());
            return comDepartMapper.updateSelective(comDepart);
        }
    }

    /**
     * 修改部门
     *
     * @param list 部门
     * @return 结果
     */
    @Override
    public int editBatch(String tklComId,List<ComDepart> list){
        ComBaseInfo comBaseInfo = comBaseInfoMapper.selectComBaseInfoByTlkId(tklComId);
        if(comBaseInfo == null){
            throw new V5BusinessException(Result.ERROR, "公司不存在");
        }

        List<ComDepart> dbList = comDepartMapper.findByBatchTlkId(list);
        List<ComDepart> dbParentList = comDepartMapper.findByBatchTlkParentId(list);
        Map<String, ComDepart> comDepartMap = dbList.stream().collect(Collectors.toMap(ComDepart::getTlkDepartId, Function.identity()));
        Map<String, ComDepart> comParentDepartMap = dbParentList.stream().collect(Collectors.toMap(ComDepart::getTlkDepartId, Function.identity()));

        Map<String,List<ComDepart>> tlkParentMap = list.stream().filter((departData)-> StringUtils.isNotEmpty(departData.getTlkParentDepartId())).collect(Collectors.groupingBy(ComDepart::getTlkParentDepartId));

        ComDepart depart = null;
        List<ComDepart> saveList = new ArrayList<>();
        List<ComDepart> departList = null;

        for (ComDepart comDepart : list) {
            if(StringUtil.isEmpty(comDepart.getTlkParentDepartId())){
                depart = comDepartMap.get(comDepart.getTlkDepartId());
                if (depart != null){
                    if( Constants.Y.equals(depart.getIsDelete())){
                        depart.setIsDelete(Constants.N);
                    }
                    BeanUtils.copyPropertiesIgnore(depart,comDepart,true);
                }else{
                    comDepart.setDepartId(idWorker.nextId());
                    comDepart.setCreateTime(DateUtils.getNowDate());
                    comDepart.setComId(comBaseInfo.getComId());
                }
                comDepart.setDepartTreeCode(comDepart.getDepartId()+"-");
                comDepart.setParentDepartId(-1L);
                comDepart.setUpdateTime(DateUtils.getNowDate());
                saveList.add(comDepart);

                departList = tlkParentMap.get(comDepart.getTlkDepartId());

                if(departList != null){
                    setSubDepart(comDepart,departList,saveList,comDepartMap,tlkParentMap);
                }

            }
        }
        if(saveList.size() > 0){
            comDepartMapper.saveBatch(saveList);
        }
        return saveList.size();
    }

    @Override
    public void saveBatch(List<ComDepart> list){
        comDepartMapper.saveBatch(list);
    }

    private void setSubDepart(ComDepart parentDepart,List<ComDepart> list, List<ComDepart> saveList, Map<String, ComDepart> comDepartMap, Map<String,List<ComDepart>> tlkParentMap){
        ComDepart depart = null;

        List<ComDepart> departList = null;
        for (ComDepart comDepart : list) {
            depart = comDepartMap.get(comDepart.getTlkDepartId());
            if (depart != null){
                if( Constants.Y.equals(depart.getIsDelete())){
                    depart.setIsDelete(Constants.N);
                }
                BeanUtils.copyPropertiesIgnore(depart,comDepart,true);
            }else{
                comDepart.setDepartId(idWorker.nextId());
                comDepart.setCreateTime(DateUtils.getNowDate());
                comDepart.setComId(parentDepart.getComId());
            }
            comDepart.setDepartTreeCode(parentDepart.getDepartTreeCode() + comDepart.getDepartId()+"-");
            comDepart.setParentDepartId(parentDepart.getDepartId());
            comDepart.setUpdateTime(DateUtils.getNowDate());
            saveList.add(comDepart);
            departList = tlkParentMap.get(comDepart.getTlkDepartId());

            if(departList != null){
                setSubDepart(comDepart,departList,saveList,comDepartMap,tlkParentMap);
            }
        }
    }

    /**
     * 删除部门对象
     * 
     * @param ids 需要删除的数据ID
     * @return 结果
     */
    @Override
    public int deleteComDepartByIds(String ids){
        return comDepartMapper.deleteComDepartByIds(Convert.toStrArray(ids));
    }

    /**
     * 删除部门信息
     * 
     * @param departId 部门ID
     * @return 结果
     */
    @Override
    public int deleteComDepartById(Long departId){
        return comDepartMapper.deleteComDepartById(departId);
    }


    /**
     * 组织架构部门树（新）
     *
     * @param comDepartDto
     * @return
     * @auth llc
     * @date 2019-07-16
     */
    @Override
    public Result getDepartTree(ComDepartDto comDepartDto) {
        String parentDepartId = "-1";
        if(comDepartDto.getComId() == null){
            return Result.success(Result.SUCCESS_QUERY_MSG);
        }
        Long comId = comDepartDto.getComId();
        List<OrgTreeDto> list = new ArrayList<OrgTreeDto>();
        // 获取公司信息
        Map<String, Object> m = perOrgRelationMapper.getComPerCount(comId);
        OrgTreeDto root = new OrgTreeDto();
        root.setDepartName(m.get("comName") != null ? m.get("comName").toString() : "");
        root.setPersonCount(Integer.parseInt(m.get("personCount").toString()));
        root.setDepartId("-1");
        root.setIsExistsDataAuth(CommonConst.Y); // 默认没权限

        // 根据公司ID、登陆用户ID获取部门数据权限
        List<OrgTreeDto> departList = comDepartMapper.getDepartList(comId);
        /*** 递归生成部门树 ***/
        List<OrgTreeDto> childlist = getDepartChild(parentDepartId, departList);
        if (childlist != null && childlist.size() > 0) {
            root.setChildDepart(childlist);
        }
        list.add(root);
        return Result.success(Result.SUCCESS_QUERY_MSG, list);
    }


    @Override
    public Result getTreeDepartNotDataAuth(ComDepartDto comDepartDto) {
        Long parentDepartId = -1L;
        Long comId = comDepartDto.getComId();
        // 增加顶级公司信息
        List<ComDepartTreeDto> list = new ArrayList<ComDepartTreeDto>();
        ComBaseInfo comBaseInfo = comBaseInfoMapper.selectComBaseInfoById(comId);
        ComDepartTreeDto root = new ComDepartTreeDto();
        root.setTitle(comBaseInfo.getComName());
        root.setKey("-1");
        root.setLabel(comBaseInfo.getComName());
        root.setValue("-1");
        Map<String, Object> param = new HashMap<String, Object>();
        param.put("parentDepartId", parentDepartId);
        param.put("comId", comId);
        List<ComDepartTreeDto> childlist = new ArrayList<ComDepartTreeDto>();
        List<Map> allTreeDepart = comDepartMapper.getTreeDepartNotDataAuth(comId);// 获取部门列表
        childlist = getAllDepartChild(param, allTreeDepart);
        if (childlist != null && childlist.size() > 0) {
            root.setChildren(childlist);
        }
        list.add(root);
        return Result.success("", list);
    }

    /**
     * 选人公共部门树接口
     *
     * @param comDepartDto
     * @return
     * @update llc
     * @date 2019-08-28
     */
    @Override
    public Result getTreeDepart(ComDepartDto comDepartDto) {
        Long parentDepartId = -1L;
        Long comId = comDepartDto.getComId();
        // 增加顶级公司信息
        List<ComDepartTreeDto> list = new ArrayList<ComDepartTreeDto>();
        ComBaseInfo comBaseInfo = comBaseInfoMapper.selectComBaseInfoById(comId);
        ComDepartTreeDto root = new ComDepartTreeDto();
        root.setTitle(comBaseInfo.getComName());
        root.setKey("-1");
        root.setLabel(comBaseInfo.getComName());
        root.setValue("-1");
        Map<String, Object> param = new HashMap<String, Object>();
        param.put("parentDepartId", parentDepartId);
        param.put("comId", comId);
        List<ComDepartTreeDto> childlist = new ArrayList<ComDepartTreeDto>();
        List<Map> allTreeDepart = comDepartMapper.getTreeDepart(comId);// 获取部门列表
        childlist = getAllDepartChild(param, allTreeDepart);
        if (childlist != null && childlist.size() > 0) {
            root.setChildren(childlist);
        }
        list.add(root);
        return Result.success("", list);
    }


    /**
     * 选人公共部门树接口(考勤打卡)
     *
     * @param comDepartDto
     * @return
     * @update llc
     * @date 2019-08-28
     */
    @Override
    public List<ComDepartTreeDto> getTreeDepartForAttend(ComDepartDto comDepartDto) {
        //加上数据权限
        if(!comDepartDto.isDomainAdmin() && (comDepartDto.getAdminDepartIds()== null || comDepartDto.getAdminDepartIds().size() == 0)){
            return new ArrayList<>();
        }
        // 增加顶级公司信息
        List<ComDepartTreeDto> list = new ArrayList<ComDepartTreeDto>();
        List<ComDepartTreeDto> allTreeDepart = comDepartMapper.getTreeDepartForAttend(comDepartDto);// 获取部门列表
        Map<String,List<ComDepartTreeDto>> departMap = allTreeDepart.stream().filter(d -> StrUtil.isNotBlank(d.getParentId())).collect(Collectors.groupingBy(ComDepartTreeDto::getParentId));
        Map<String, ComDepartTreeDto> comDepartMap = allTreeDepart.stream().collect(Collectors.toMap(ComDepartTreeDto::getKey, Function.identity()));

        for (ComDepartTreeDto treeDto : allTreeDepart) {
            if(departMap.get(treeDto.getKey()) != null){
                treeDto.setChildren(departMap.get(treeDto.getKey()));
            }else{
                treeDto.setChildren(new ArrayList<>());
            }
            if(StringUtil.isEmpty(treeDto.getParentId()) || !comDepartMap.containsKey(treeDto.getParentId())){
                list.add(treeDto);
            }
        }
        return list;
    }

    @Override
    public ComDepart selectComDepartByTlkId(String tlkDepartId) {
        return comDepartMapper.selectComDepartByTlkId(tlkDepartId);
    }

    @Override
    public List<ComDepart> selectListByComId(Long comId) {
        return comDepartMapper.selectListByComId(comId);
    }

    /**
     * 递归生成部门树
     *
     * @param
     * @return
     * @auth llc
     * @date 2019-07-16
     */
    @Override
    public List<OrgTreeDto> getDepartChild(String parentDepartId, List<OrgTreeDto> departList) {
        List<OrgTreeDto> deptVosList = new ArrayList<OrgTreeDto>();
        if (departList != null && departList.size() > 0) {
            for (OrgTreeDto depart : departList) {
                if (parentDepartId.equals(depart.getParentDepartId())) {
                    OrgTreeDto orgTreeDto = new OrgTreeDto();
                    orgTreeDto.setDepartId(depart.getDepartId());
                    orgTreeDto.setPersonCount(depart.getPersonCount());
                    orgTreeDto.setDepartName(depart.getDepartName());
                    orgTreeDto.setIsExistsDataAuth(depart.getIsExistsDataAuth());
                    orgTreeDto.setParentDepartId(depart.getParentDepartId());
                    orgTreeDto.setChildDepart(getDepartChild(depart.getDepartId(), departList));
                    deptVosList.add(orgTreeDto);
                }
            }
        }
        return deptVosList;
    }


    /**
     * @param param
     * @return
     * @descript:递归部门
     */
    @SuppressWarnings("rawtypes")
    private List<ComDepartTreeDto> getAllDepartChild(Map<String, Object> param, List<Map> allTreeDepart) {
        List<ComDepartTreeDto> deptVosList = new ArrayList<ComDepartTreeDto>();
        Long parentDepartId = Long.valueOf(param.get("parentDepartId").toString());
        Long comId = Long.valueOf(param.get("comId").toString());
        if (allTreeDepart != null && allTreeDepart.size() > 0) {
            for (Map map : allTreeDepart) {
                if(map.get("parentDepartId") == null){
                    map.put("parentDepartId","-1");
                }
                if (parentDepartId.equals(Long.valueOf(map.get("parentDepartId").toString()))) {
                    ComDepartTreeDto comDepartTreeDto = new ComDepartTreeDto();
                    comDepartTreeDto.setKey(String.valueOf(map.get("departId")));
                    comDepartTreeDto.setValue(String.valueOf(map.get("departId")));
                    comDepartTreeDto.setTitle(String.valueOf(map.get("departName")));
                    comDepartTreeDto.setLabel(String.valueOf(map.get("departName")));
                    Map<String, Object> paramMap = new HashMap<String, Object>();
                    paramMap.put("parentDepartId", String.valueOf(map.get("departId")));
                    paramMap.put("comId", comId);
                    comDepartTreeDto.setChildren(getAllDepartChild(paramMap, allTreeDepart));
                    deptVosList.add(comDepartTreeDto);
                }
            }
        }
        return deptVosList;
    }
}
