package com.teemlink.pm.task.service;

import java.io.File;
import java.io.FileInputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.util.CellRangeAddress;

import com.teemlink.pm.activity.model.Activity;
import com.teemlink.pm.activity.service.ActivityProcess;
import com.teemlink.pm.activity.service.ActivityProcessBean;
import com.teemlink.pm.base.dao.BaseDAO;
import com.teemlink.pm.base.dao.DaoManager;
import com.teemlink.pm.base.service.AbstractBaseProcessBean;
import com.teemlink.pm.notification.TaskNotificationService;
import com.teemlink.pm.notification.TaskNotificationServiceImpl;
import com.teemlink.pm.project.model.Member;
import com.teemlink.pm.project.model.Project;
import com.teemlink.pm.project.service.ProjectProcess;
import com.teemlink.pm.project.service.ProjectProcessBean;
import com.teemlink.pm.tag.model.Tag;
import com.teemlink.pm.tag.service.TagProcess;
import com.teemlink.pm.tag.service.TagProcessBean;
import com.teemlink.pm.task.dao.TaskDAO;
import com.teemlink.pm.task.model.Follower;
import com.teemlink.pm.task.model.Log;
import com.teemlink.pm.task.model.Remark;
import com.teemlink.pm.task.model.SubTask;
import com.teemlink.pm.task.model.Task;
import com.teemlink.pm.task.model.TaskUserCount;
import com.teemlink.pm.util.json.JsonUtil;

import cn.myapps.base.web.WebUser;
import cn.myapps.common.data.DataPackage;
import cn.myapps.common.model.ValueObject;
import cn.myapps.common.util.StringUtil;
import cn.myapps.util.DateUtil;
import cn.myapps.util.sequence.Sequence;

public class TaskProcessBean extends AbstractBaseProcessBean<Task>
		implements TaskProcess {

	private TaskNotificationService taskNotificationService = new TaskNotificationServiceImpl();

	@Override
	public BaseDAO getDAO() throws Exception {
		return DaoManager.getTaskDAO(getConnection());
	}

	public ValueObject doCreate(ValueObject vo,WebUser user) throws Exception {
		Task task = (Task)vo;
		try {
			beginTransaction();
			if(StringUtil.isBlank(task.getId())){
				task.setId(Sequence.getSequence());
			}
			if(StringUtil.isBlank(task.getCreator())){
				task.setCreator(user.getName());
				task.setCreatorId(user.getId());
			}
			if(StringUtil.isBlank(task.getExecutorId())){
				task.setExecutor(user.getName());
				task.setExecutorId(user.getId());
			}
			
			if(task.getCreateDate() == null){
				task.setCreateDate(new Date());
			}
			if(task.getStartDate() == null){
				task.setStartDate(new Date());
			}
			if(StringUtil.isBlank(task.getProjectId())){
				task.setType(Task.TYPE_CREATE_BY_MYSELF);
			}else{
				task.setType(Task.TYPE_ASSING_BY_OBJECT);
			}
			task.setDomainid(user.getDomainid());
			
			if(!task.getFollowers().isEmpty()){
				((TaskDAO)getDAO()).addFollowers(task.getId(), task.getFollowers());
			}
			
			Collection<Log> logs = new ArrayList<Log>();
			logs.add(new Log(task, Log.TYPE_CREATE_TASK, user));
			task.setLogs(JsonUtil.list2JSON(logs));
			
			task = (Task) getDAO().create(task);
			
			//更新所属项目的任务数字段
			if(!StringUtil.isBlank(task.getProjectId())){
				ProjectProcess projectProcess = new ProjectProcessBean();
				projectProcess.updateTasksTotal(task.getProjectId());
			}
			
			//创建Activity
			ActivityProcess activityProcess = new ActivityProcessBean();
			Activity activity = new Activity(task.getId(), Activity.TYPE_NEW_TASK, user);
			activityProcess.doCreate(activity);
			
			taskNotificationService.create(task, user);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		
		return task;
	}
	
	/**
	 * 更新任务的单个字段
	 * @param taskId
	 * 		任务主键
	 * @param updateField
	 * 		更新的字段名
	 * @param updateValue
	 * 		更新的字段值
	 * @param user
	 * 		用户
	 * 
	 * @return
	 * @throws Exception
	 */
	public Task doSimpleUpdate(String taskId,String updateField,String updateValue,WebUser user) throws Exception{
		
		Task task = (Task) doView(taskId);
		try {
			beginTransaction();
			//插入日志
			Collection<Log> logs = task.getLogList();
			logs.add(new Log(task, updateField, updateValue, user));
			task.setLogs(JsonUtil.list2JSON(logs));
			task = (Task) getDAO().update(task);
			if("projectId".equals(updateField)){
				ProjectProcess projectProcess = new ProjectProcessBean();
				projectProcess.updateTasksTotal(task.getProjectId());
			}
			
			if("endDate".equalsIgnoreCase(updateField)){
				taskNotificationService.updateTime(task, user);
			}
			
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		
		return task;
	}

	/* (non-Javadoc)
	 * @see cn.myapps.pm.task.model.TaskProcess#addFollowers(java.lang.String, java.util.Collection, cn.myapps.core.user.action.WebUser)
	 */
	public void addFollowers(String taskId, Collection<Follower> followers,WebUser user)
			throws Exception {
		try {
			beginTransaction();
			//插入日志
			Task task = (Task) doView(taskId);
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_ADD_FOLLOWER);
			StringBuilder userNames = new StringBuilder();
			for(Follower f :followers){
				userNames.append(f.getUserName()).append(" ");
			}
			log.setSummary(userNames.toString());
			logs.add(log);
			
			((TaskDAO)getDAO()).addFollowers(taskId, followers);
			updateLogs(taskId,JsonUtil.list2JSON(logs));
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}

	/* (non-Javadoc)
	 * @see cn.myapps.pm.task.model.TaskProcess#deleteFollower(java.lang.String, java.lang.String, java.lang.String, cn.myapps.core.user.action.WebUser)
	 */
	public void deleteFollower(String userId,String userName, String taskId,WebUser user) throws Exception {
		try {
			beginTransaction();
			//插入日志
			Task task = (Task) doView(taskId);
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_DELETE_FOLLOWER);
			log.setSummary(userName);
			logs.add(log);
			
			((TaskDAO)getDAO()).deleteFollower(userId, taskId);
			updateLogs(taskId,JsonUtil.list2JSON(logs));
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		
	}
	
	/* (non-Javadoc)
	 * @see cn.myapps.pm.task.model.TaskProcess#doFollow(java.lang.String, cn.myapps.core.user.action.WebUser)
	 */
	public void doFollow(String taskId,WebUser user) throws Exception {
		
		try {
			beginTransaction();
			//插入日志
			Task task = (Task) doView(taskId);
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_FOLLOW_TASK);
			log.setSummary(user.getName());
			logs.add(log);
			
			Collection<Follower> followers = new ArrayList<Follower>();
			Follower follower = new Follower();
			follower.setUserId(user.getId());
			follower.setUserName(user.getName());
			follower.setDomainId(user.getDomainid());
			followers.add(follower);
			
			((TaskDAO)getDAO()).addFollowers(taskId,followers);
			
			updateLogs(taskId,JsonUtil.list2JSON(logs));
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	/* (non-Javadoc)
	 * @see cn.myapps.pm.task.model.TaskProcess#doUnFollow(java.lang.String, cn.myapps.core.user.action.WebUser)
	 */
	public void doUnFollow(String taskId,WebUser user) throws Exception {
		
		try {
			beginTransaction();
			//插入日志
			Task task = (Task) doView(taskId);
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_UNFOLLOW_TASK);
			log.setSummary(user.getName());
			logs.add(log);
			
			((TaskDAO)getDAO()).deleteFollower(user.getId(), taskId);
			
			updateLogs(taskId,JsonUtil.list2JSON(logs));
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	public void doComplete(String taskId,WebUser user) throws Exception {
		try {
			beginTransaction();
			//插入日志
			Task task = (Task) doView(taskId);
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_COMPLETE_TASK);
			log.setSummary(task.getName());
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("STATUS", Task.STATUS_COMPLETE);
			((TaskDAO)getDAO()).update(items, taskId);
			
			
			//更新所属项目的完成任务数
			if(!StringUtil.isBlank(task.getProjectId())){
				ProjectProcess projectProcess = new ProjectProcessBean();
				projectProcess.updateTasksTotal(task.getProjectId());
			}
			
			updateLogs(taskId,JsonUtil.list2JSON(logs));
			//级联创建Activity
			ActivityProcess activityProcess = new ActivityProcessBean();
			Activity activity = new Activity(task.getId(), Activity.TYPE_COMPLETE_TASK, user);
			activityProcess.doCreate(activity);
			commitTransaction();
			
			//发送通知
			taskNotificationService.complate(task, user);
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	public void redoTask(String taskId,WebUser user) throws Exception {
		try {
			beginTransaction();
			//插入日志
			Task task = (Task) doView(taskId);
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_REDO_TASK);
			log.setSummary(task.getName());
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("STATUS", Task.STATUS_IN_PROGRESS);
			((TaskDAO)getDAO()).update(items, taskId);
			
			
			//更新所属项目的完成任务数
			if(!StringUtil.isBlank(task.getProjectId())){
				ProjectProcess projectProcess = new ProjectProcessBean();
				projectProcess.updateTasksTotal(task.getProjectId());
			}
			
			updateLogs(taskId,JsonUtil.list2JSON(logs));
			//级联创建Activity
			ActivityProcess activityProcess = new ActivityProcessBean();
			Activity activity = new Activity(task.getId(), Activity.TYPE_REDO_TASK, user);
			activityProcess.doCreate(activity);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	
	
	
	@Override
	public void doRemove(String pk) throws Exception {
		try {
			beginTransaction();
			Task task = (Task) doView(pk);
			
			//更新所属项目的任务总数和完成任务数
			if(!StringUtil.isBlank(task.getProjectId())){
				ProjectProcess projectProcess = new ProjectProcessBean();
				projectProcess.updateTasksTotal(task.getProjectId());
				if(task.getStatus() == Task.STATUS_COMPLETE){
					projectProcess.updateTasksTotal(task.getProjectId());
				}
			}
			
			getDAO().remove(pk);
			//记录删除Activity
			ActivityProcess activityProcess = new ActivityProcessBean();
			activityProcess.removeByTask(pk);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	public void createSubTask(String taskId,SubTask subTask,WebUser user) throws Exception{
		try {
			beginTransaction();
			//插入日志
			Task task = (Task) doView(taskId);
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_CREATE_SUB_TASK);
			log.setSummary(subTask.getName());
			logs.add(log);
			
			Collection<SubTask> subTasks = task.getSubTaskList();
			subTasks.add(subTask);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("SUB_TASKS",JsonUtil.list2JSON(subTasks));
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	public void removeSubTask(String taskId,String subTaskId,WebUser user) throws Exception{
		try {
			Task task = (Task) doView(taskId);
			beginTransaction();
			Collection<SubTask> subTasks = task.getSubTaskList();
			SubTask po = null;
			for(SubTask subTask : subTasks){
				if(subTaskId.equals(subTask.getId())){
					po = subTask;
					break;
				}
			}
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_DELETE_SUB_TASK);
			log.setSummary(po.getName());
			logs.add(log);
			
			subTasks.remove(po);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("SUB_TASKS",JsonUtil.list2JSON(subTasks));
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	public SubTask updateSubTask(String taskId,String subTaskId,String subTaskName,WebUser user) throws Exception{
		SubTask subTask = null;
		try {
			Task task = (Task) doView(taskId);
			beginTransaction();
			
			List<SubTask> subTasks = (List<SubTask>) task.getSubTaskList();
			int index = 0;
			
			for (Iterator<SubTask> iterator = subTasks.iterator(); iterator.hasNext();) {
				SubTask st = iterator.next();
				if(subTaskId.equals(st.getId())){
					subTask = st;
					break;
				}
				index++;
				
			}
			if(subTask ==null) throw new RuntimeException("找不到对象");
			subTask.setName(subTaskName);
			subTasks.remove(index);
			subTasks.add(index,subTask);
			
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_UPDATE_SUB_TASK);
			log.setSummary(subTaskName);
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("SUB_TASKS",JsonUtil.list2JSON(subTasks));
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		return subTask;
	}
	
	public SubTask completeSubTask(String taskId,String subTaskId,WebUser user) throws Exception{
		SubTask subTask = null;
		try {
			Task task = (Task) doView(taskId);
			beginTransaction();
			
			List<SubTask> subTasks = (List<SubTask>) task.getSubTaskList();
			int index = 0;
			
			for (Iterator<SubTask> iterator = subTasks.iterator(); iterator.hasNext();) {
				SubTask st = iterator.next();
				if(subTaskId.equals(st.getId())){
					subTask = st;
					break;
				}
				index++;
				
			}
			if(subTask ==null) throw new RuntimeException("找不到对象");
			subTask.setStatus(SubTask.STATUS_COMPLETE);
			subTasks.remove(index);
			subTasks.add(index,subTask);
			
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_COMPLETE_SUB_TASK);
			log.setSummary(subTask.getName());
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("SUB_TASKS",JsonUtil.list2JSON(subTasks));
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		return subTask;
	}
	
	public SubTask redoSubTask(String taskId,String subTaskId,WebUser user) throws Exception{
		SubTask subTask = null;
		try {
			Task task = (Task) doView(taskId);
			beginTransaction();
			
			List<SubTask> subTasks = (List<SubTask>) task.getSubTaskList();
			int index = 0;
			
			for (Iterator<SubTask> iterator = subTasks.iterator(); iterator.hasNext();) {
				SubTask st = iterator.next();
				if(subTaskId.equals(st.getId())){
					subTask = st;
					break;
				}
				index++;
				
			}
			if(subTask ==null) throw new RuntimeException("找不到对象");
			subTask.setStatus(SubTask.STATUS_UNCOMPLETE);
			subTasks.remove(index);
			subTasks.add(index,subTask);
			
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_REDO_SUB_TASK);
			log.setSummary(subTask.getName());
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("SUB_TASKS",JsonUtil.list2JSON(subTasks));
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		return subTask;
	}
	
	public void createRemark(String taskId,Remark remark,WebUser user) throws Exception{
		try {
			beginTransaction();
			//插入日志
			Task task = (Task) doView(taskId);
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_CREATE_REMAEK);
			log.setSummary(remark.getContent());
			logs.add(log);
			
			Collection<Remark> remarks = task.getRemarkList();
			remarks.add(remark);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("REMARK",JsonUtil.list2JSON(remarks));
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	public void removeRemark(String taskId,String remarkId,WebUser user) throws Exception{
		try {
			Task task = (Task) doView(taskId);
			beginTransaction();
			Collection<Remark> remarks = task.getRemarkList();
			Remark po = null;
			for(Remark remark : remarks){
				if(remarkId.equals(remark.getId())){
					po = remark;
					break;
				}
			}
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_DELETE_REMAEK);
			log.setSummary(po.getContent());
			logs.add(log);
			
			remarks.remove(po);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("REMARK",JsonUtil.list2JSON(remarks));
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	public Remark updateRemark(String taskId,String remarkId,String remarkContent,WebUser user) throws Exception{
		Remark remark = null;
		try {
			Task task = (Task) doView(taskId);
			beginTransaction();
			
			List<Remark> remarks = (List<Remark>) task.getRemarkList();
			int index = 0;
			
			for (Iterator<Remark> iterator = remarks.iterator(); iterator.hasNext();) {
				Remark st = iterator.next();
				if(remarkId.equals(st.getId())){
					remark = st;
					break;
				}
				index++;
				
			}
			if(remark ==null) throw new RuntimeException("找不到对象");
			remark.setContent(remarkContent);
			remarks.remove(index);
			remarks.add(index,remark);
			
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_UPDATE_REMAEK);
			log.setSummary(remarkContent);
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("REMARK",JsonUtil.list2JSON(remarks));
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		return remark;
	}

	/**
	 * 更新任务日志
	 * @param taskId
	 * 		任务主键
	 * @param logs
	 * 		日志内容（JSON）
	 * @throws Exception
	 */
	private void updateLogs(String taskId,String logs) throws Exception {
		try {
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("LOGS", logs);
			getDAO().update(items, taskId);
		} catch (Exception e) {
			throw e;
		}
	}

	public Collection<Task> queryMyTasks(String name, int status, String currDate,
			String dateRangeType, WebUser user) throws Exception {
		return ((TaskDAO)getDAO()).queryMyTasks(name, status, currDate, dateRangeType, user);
	}

	public Tag addTag(String taskId, String tagName, WebUser user) throws Exception {
		Tag tag = new Tag();
		tag.setName(tagName);
		try {
			Task task = (Task) doView(taskId);
			beginTransaction();
			
			String tags = StringUtil.isBlank(task.getTags())? tagName : task.getTags()+", "+tagName+" ";
			
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_ADD_TAG);
			log.setSummary(tagName);
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("TAGS",tags);
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		return tag;
		
	}
	
	public void removeTag(String taskId, String tagName, WebUser user) throws Exception {
		try {
			Task task = (Task) doView(taskId);
			beginTransaction();
			
			StringBuilder tags = new StringBuilder();
			
			String[] arr = task.getTags().split(",");
			for (int i = 0; i < arr.length; i++) {
				if(!tagName.equals(arr[i].trim())){
					tags.append(arr[i]).append(",");
				}
			}
			if(tags.length()>0){
				tags.setLength(tags.length() -1);
			}
			
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_DELETE_TAG);
			log.setSummary(tagName);
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("TAGS",tags.toString());
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		
	}
	
	
	public Task setProject(String taskId,String projectId,String projectName,WebUser user) throws Exception{
		Task task = (Task) doView(taskId);
		try {
			String oldProjectId = task.getProjectId();
			
			task.setProjectId(projectId);
			task.setProjectName(projectName);
			beginTransaction();
			
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_SET_PROJECT);
			log.setSummary(projectName);
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("PROJECT_ID",projectId);
			items.put("PROJECT_NAME",projectName);
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			
			//更新任务项目数
			ProjectProcess projectProcess = new ProjectProcessBean();
			projectProcess.updateTasksTotal(oldProjectId);
			if(Task.STATUS_COMPLETE == task.getStatus()){
				projectProcess.updateTasksTotal(oldProjectId);
			}
			projectProcess.updateTasksTotal(task.getProjectId());
			if(Task.STATUS_COMPLETE == task.getStatus()){
				projectProcess.updateTasksTotal(task.getProjectId());;
			}
			
			
			
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		return task;
	}
	
	public void removeProject(String taskId,WebUser user) throws Exception{
		Task task = (Task) doView(taskId);
		try {
			beginTransaction();
			
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_DELETE_PROJECT);
			log.setSummary(task.getProjectName());
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("PROJECT_ID",null);
			items.put("PROJECT_NAME",null);
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			
			ProjectProcess projectProcess = new ProjectProcessBean();
			
			if(Task.STATUS_COMPLETE == task.getStatus()){
				projectProcess.updateTasksTotal(task.getProjectId());
			}
			
			projectProcess.updateTasksTotal(task.getProjectId());
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	}
	
	public Collection<Follower> getFollowersByTask(String taskId)throws Exception{
		return ((TaskDAO)getDAO()).queryFollowersByTask(taskId);
	}

	public Task updateTaskExecutor(String taskId, String executorId,
			String executorName, WebUser user) throws Exception {
		Task task = (Task) doView(taskId);
		//获取旧执行人的Id
		String old_tasker_id = task.getExecutorId();
		try {
			task.setExecutorId(executorId);
			task.setExecutor(executorName);
			beginTransaction();
			
			//插入日志
			Collection<Log> logs = task.getLogList();
			Log log = new Log(user);
			log.setOperationType(Log.TYPE_UPDATE_EXECUTOR);
			log.setSummary(executorName);
			logs.add(log);
			
			Map<String, Object> items = new HashMap<String, Object>();
			items.put("EXECUTER_ID",executorId);
			items.put("EXECUTER",executorName);
			items.put("LOGS",JsonUtil.list2JSON(logs));
			
			((TaskDAO)getDAO()).update(items, taskId);
			
			//　执行者修改后通知发送
		    taskNotificationService.updateExecutor(task,old_tasker_id,user);
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		return task;
	}
	
	public Collection<Task> queryMyFollowTasks(String name,int status,String currDate,String dateRangeType,int page,int lines,WebUser user) throws Exception{
		return ((TaskDAO)getDAO()).queryMyFollowTasks(name, status, currDate, dateRangeType, page,lines,user);
	}
	
	
	public DataPackage<Task> queryTasksByProject(String projectId, String name,
			int status, int level, String executer, String creater,String currDate, String dateRangeType, String tag ,String orderName,String orderBy,int page,int lines, String overdueStatus, WebUser user)
			throws Exception {
		return ((TaskDAO)getDAO()).queryByProject(projectId, name, status, level, executer,creater, currDate, dateRangeType,tag, orderName,orderBy,page, lines, overdueStatus, user);
	}

	public DataPackage<Task> queryTasksByProject(String projectId, String name,Integer[] status, Integer[] levels, String[] executers, String[] creaters, String currDate, String dateRangeType, String[] tags, String orderName,String orderBy,int page,int lines, String overdueStatus, WebUser user)throws Exception{
		return ((TaskDAO)getDAO()).queryByProject(projectId, name, status, levels, executers, creaters, currDate, dateRangeType, tags, orderName, orderBy, page, lines, overdueStatus, user);
	}
	
	public List<Task> listTasksByProject(String projectId, String name,Integer[] status, Integer[] levels, String[] executers, String[] creaters, String currDate, String dateRangeType, String[] tags, String orderName,String orderBy, String overdueStatus, WebUser user)throws Exception{
		List<String> projectIds=new ArrayList<String>();
		if(!StringUtil.isBlank(projectId)){
			projectIds.add(projectId);
		}else{
			//如果projectId为空，则查询全部参与的项目(已经打开的项目)
			ProjectProcess projectProcess=new ProjectProcessBean();
			Collection<Project> myProjects = projectProcess.doQueryMyProject("", 0, user);
			for (Project project : myProjects) {
				if(!project.isClosed())
					projectIds.add(project.getId());
			}
		}
		return ((TaskDAO)getDAO()).listByProjects(projectIds, name, status, levels, executers, creaters, currDate, dateRangeType, tags, orderName, orderBy, overdueStatus, user);
	}

	
	public Collection<Task> queryTasksByTag(String tagName, String name,
			int status, String currDate, String dateRangeType,int page,int lines, WebUser user)
			throws Exception {
		return ((TaskDAO)getDAO()).queryTasksByTag(tagName, name, status, currDate, dateRangeType,page,lines, user);
	}

	public Collection<Task> queryMyTasks4CalendarView(String name, int status,
			String startDate, String endDate, WebUser user) throws Exception {
		return ((TaskDAO)getDAO()).queryMyTasks4CalendarView(name, status, startDate, endDate, user);
	}

	public Collection<Task> queryMyEntrustTasks(String name, int status,
			String currDate, String dateRangeType, int page, int lines,
			WebUser user) throws Exception {
		return ((TaskDAO)getDAO()).queryMyEntrustTasks(name, status, currDate, dateRangeType, page,lines,user);
	}
	
	public void addAttachment(String id,String attachjson) throws Exception{
		((TaskDAO)getDAO()).addAttachment(id, attachjson);
	}

	public Task deleteAttachment(String id,String key) throws Exception{
		return ((TaskDAO)getDAO()).deleteAttachment(id, key);
	}
	
	@Override
	public Task doUpdateTaskStatus(String taskId, int status, WebUser user)
			throws Exception {
		
		Task task = (Task) doView(taskId);
		String status_complete = Task.STATUS_MAP.get(Task.STATUS_COMPLETE);
		String status_reject = Task.STATUS_MAP.get(Task.STATUS_REJECT);
		
		String old_Status = Task.STATUS_MAP.get(task.getStatus());
		String new_Status = Task.STATUS_MAP.get(status);
		try {
			
			beginTransaction();
			//插入日志
			Collection<Log> logs = task.getLogList();
			logs.add(new Log(task, "status",Integer.toString(status), user));
			task.setLogs(JsonUtil.list2JSON(logs));
			
			//更新任务
			task.setStatus(status);
			task = (Task) getDAO().update(task);
			
			//增加/减少项目数
			if(!StringUtil.isBlank(task.getProjectId())){
				if((!old_Status.equals(status_complete) && !old_Status.equals(status_reject)) && (new_Status.equals(status_complete) || new_Status.equals(status_reject))){ // 逻辑：非完成状态转化为 完成状态  ，项目完成数加一
					ProjectProcess projectProcess = new ProjectProcessBean();
					projectProcess.updateTasksTotal(task.getProjectId());
				}else if((old_Status.equals(status_complete) || old_Status.equals(status_reject))  && (!new_Status.equals(status_complete) && !new_Status.equals(status_reject))){  // 逻辑：完成状态转化为非完成状态  ，项目完成数减一
					ProjectProcess projectProcess = new ProjectProcessBean();
					projectProcess.updateTasksTotal(task.getProjectId());
				}
			}
			
			String Activity_Field = null;
			Field declaredField = null;
			int Activity_Value = 0 ;
			
			//获取旧状态相应的Activity字段
			Activity_Field = "TYPE_"+old_Status.substring(7)+"_TASK";
			declaredField = Activity.class.getDeclaredField(Activity_Field);
			Activity_Value = (Integer) declaredField.get(Activity_Field);
			String old_Status_ = Activity.Type_Map.get(Activity_Value);
			
			//获取新状态相应的Activity字段信息
			Activity_Field = "TYPE_"+new_Status.substring(7)+"_TASK";
			declaredField = Activity.class.getDeclaredField(Activity_Field);
			Activity_Value = (Integer) declaredField.get(Activity_Field);
			String new_Status_ = Activity.Type_Map.get(Activity_Value);
			
			//级联创建Activity
			ActivityProcess activityProcess = new ActivityProcessBean();
			Activity activity = new Activity(task.getId(), Activity_Value, user);
			activityProcess.doCreate(activity);
		
			if("STATUS_COMPLETE".equalsIgnoreCase(new_Status)){
				taskNotificationService.complate(task, user);
			}else{
				taskNotificationService.updateTaskStatus(task,old_Status_,new_Status_, user);
			}
			
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
		
		return task;
	}

	@Override
	public DataPackage<Task> queryTasks(String projectId, String taskName,Integer status, Integer level, String executerId, String createrId,String currDate, String dateRangeType, String tag,String orderName, String orderBy, Integer page, Integer lines,String overdueStatus, WebUser user) throws Exception{
		return ((TaskDAO)getDAO()).queryTasks(projectId, taskName, status, level, executerId,createrId, currDate, dateRangeType,tag, orderName,orderBy,page, lines, overdueStatus, user);
	}

	@Override
	public Task doUpdate(Task vo,WebUser user) throws Exception {
		ProjectProcess projectProcess = null ;
		Task task = (Task) doView(vo.getId());
		
		if (task != null) {
	    	
	    String old_ExecutorId = ""; // 原执行者
	    Date old_EndTime = null ;   // 原结束时间
		try {
			beginTransaction();
			//获取日志
			Collection<Log> logs = task.getLogList();
			
			if(!StringUtil.isBlank(vo.getName()) && !vo.getName().equals(task.getName())){//名称修改
				logs.add(new Log(task, "name", vo.getName(), user));
			}
			
			if(!StringUtil.isBlank(vo.getExecutorId()) && !vo.getExecutorId().equals(task.getExecutorId())){//修改执行人
				old_ExecutorId = task.getExecutorId();
				logs.add(new Log(task, "executor", vo.getExecutorId()+","+vo.getExecutor(), user));
			}
			
			if(!compareTime(vo.getStartDate(),task.getStartDate())){//修改开始时间
				if(vo.getStartDate() == null){
					logs.add(new Log(task, "startDate", "", user));
				}else{
					try {
						SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
						String startDate = sdf.format(vo.getStartDate());
						logs.add(new Log(task, "startDate", startDate, user));
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
			
			if(!compareTime(vo.getEndDate(),task.getEndDate())){//修改结束时间
				old_EndTime = task.getEndDate();
				if(vo.getEndDate() == null){
					logs.add(new Log(task, "endDate", "", user));
				}else{
					try {
						SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
						String endDate = sdf.format(vo.getEndDate());
						logs.add(new Log(task, "endDate", endDate, user));
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
			
			if(vo.getLevel() != task.getLevel()){//修改级别
				logs.add(new Log(task, "level",String.valueOf(vo.getLevel()) , user));
			}
			
			String old_Status = Task.STATUS_MAP.get(task.getStatus());
			String new_Status = Task.STATUS_MAP.get(vo.getStatus());
			String old_StatusLable ="";
			String new_StatusLable ="";
			
			if(!old_Status.equals(new_Status)){ //修改状态
				
				String status_complete = Task.STATUS_MAP.get(Task.STATUS_COMPLETE);
				String status_reject = Task.STATUS_MAP.get(Task.STATUS_REJECT);
				
				//增加/减少完成数目
				if(!StringUtil.isBlank(task.getProjectId())){
					if((!old_Status.equals(status_complete) && !old_Status.equals(status_reject)) && (new_Status.equals(status_complete) || new_Status.equals(status_reject))){ // 逻辑：非完成状态转化为 完成状态  ，项目完成数加一
						if(projectProcess == null){
							projectProcess = new ProjectProcessBean();
						}
						projectProcess.updateTasksTotal(task.getProjectId());
					}else if((old_Status.equals(status_complete) || old_Status.equals(status_reject))  && (!new_Status.equals(status_complete) && !new_Status.equals(status_reject))){  // 逻辑：完成状态转化为非完成状态  ，项目完成数减一
						if(projectProcess == null){
							projectProcess = new ProjectProcessBean();
						}
						projectProcess.updateTasksTotal(task.getProjectId());
					}
				}
				
				String Activity_Field = null;
				Field declaredField = null;
				int Activity_Value = 0 ;
				
				//获取旧状态相应的Activity字段
				Activity_Field = "TYPE_"+old_Status.substring(7)+"_TASK";
				declaredField = Activity.class.getDeclaredField(Activity_Field);
				Activity_Value = (Integer) declaredField.get(Activity_Field);
				old_StatusLable = Activity.Type_Map.get(Activity_Value);
				
				//获取新状态相应的Activity字段信息
				Activity_Field = "TYPE_"+new_Status.substring(7)+"_TASK";
				declaredField = Activity.class.getDeclaredField(Activity_Field);
				Activity_Value = (Integer) declaredField.get(Activity_Field);
				new_StatusLable = Activity.Type_Map.get(Activity_Value);
				
				
				logs.add(new Log(task, "status",Integer.toString(vo.getStatus()), user));
			}
			
			
	    	if(!StringUtil.isBlank(vo.getAttachment()) && !vo.getAttachment().equals(task.getAttachment())){ //修改附件
	    		task.setAttachment(vo.getAttachment());
	    	}
	    	
	    	if(!StringUtil.isBlank(vo.getTags()) && !vo.getTags().equals(task.getTags())){//更新标签
	    		Log log = new Log(user);
				log.setOperationType(Log.TYPE_ADD_TAG);
				log.setSummary(vo.getTags());
				logs.add(log);
				task.setTags(vo.getTags());
	    	}
	    	
	    	String oldProjectId = "" ;
	    	String newProjectId = "" ;
	    	if(!vo.getProjectId().equals(task.getProjectId())){ //修改项目
	    		newProjectId = vo.getProjectId();
	    		oldProjectId = task.getProjectId();
	    		//更新task
	    		task.setProjectId(vo.getProjectId());
		    	task.setProjectName(vo.getProjectName());
	    	}
	    	
	    	
			task.setLogs(JsonUtil.list2JSON(logs));
			task = (Task) getDAO().update(task);
			
			if(!oldProjectId.equals(newProjectId)){
				if(projectProcess == null){
	    			projectProcess = new ProjectProcessBean();
	    		}
	    		//1.更新原有项目的任务数量
	    		projectProcess.updateTasksTotal(oldProjectId);
	    		if(task.getStatus() == Task.STATUS_COMPLETE){
	    			projectProcess.updateTasksTotal(oldProjectId);
	    		}
	    		//2.更新现有项目的任务数量
	    		projectProcess.updateTasksTotal(task.getProjectId());
		    	if(task.getStatus() == Task.STATUS_COMPLETE){
	    			projectProcess.updateTasksTotal(task.getProjectId());
	    		}
			}
			
			//　执行修改后通知发送
			if(!StringUtil.isBlank(old_ExecutorId) && !old_ExecutorId.equals(task.getExecutorId())){//修改执行人
				taskNotificationService.updateExecutor(task,old_ExecutorId,user);
			}
			if(!compareTime(task.getEndDate(),old_EndTime)){//修改结束时间
				taskNotificationService.updateTime(task, user);
			}
			if(!old_Status.equals(new_Status)){ //状态
				if("STATUS_COMPLETE".equalsIgnoreCase(new_Status)){
					taskNotificationService.complate(task, user);
				}else{
					taskNotificationService.updateTaskStatus(task,old_StatusLable,new_StatusLable, user);
				}
			}
			commitTransaction();
		} catch (Exception e) {
			rollbackTransaction();
			e.printStackTrace();
			throw e;
		}
	 }
		return task;
	}
	
	private Boolean compareTime(Date d1 ,Date d2){
		long time1 = 0 ;
		long time2 = 0 ; 
		
		if(d1 != null){
			time1 = d1.getTime();
		}
		if( d2 != null){          
			time2 = d2.getTime(); 
		}
		return time1 == time2 ;
	}
	
	@Override
	public void excelExportTask(OutputStream outputStream,DataPackage<Task> datas)throws Exception{
		exportExcel(outputStream,datas);
	}
	
	@Override
	public int excelImportTask(File file,Project project,WebUser webUser)throws Exception{
		Workbook workbook = WorkbookFactory.create(new FileInputStream(file));

		Map<String,List<Object>> datas = getDatasFromExcel(workbook,project,webUser);
		
		//如果存在错误信息，则把错误信息当异常抛出。
		if(datas.get("error") != null){
			StringBuffer errorMessage = new StringBuffer();
			for(Object obj:datas.get("error")){
				errorMessage.append(obj.toString()).append("\n");
			}
			throw new Exception(errorMessage.toString());
		}
		try{
			beginTransaction();
			List<Object> taskArray = datas.get("task");
			for(int i = 0; i < taskArray.size(); ++i){
				Task task = (Task) taskArray.get(i);
				doCreate(task,webUser);
			}
			commitTransaction();
			return taskArray.size();
		}catch(Exception e){
			rollbackTransaction();
			e.printStackTrace();
		} finally {
			if(file.getPath().contains("uploads") && file.exists()){
				file.delete();
			}
		}
		return 0;
	}
	
	private Map<String,List<Object>> getDatasFromExcel(Workbook workbook, Project project, WebUser webUser) throws Exception{
		Map<String,List<Object>> rtn = new HashMap<String,List<Object>>();
		List<Object> errorArray = new ArrayList<Object>();
		List<Object> taskArray = new ArrayList<Object>();
		ProjectProcess projectProcess = new ProjectProcessBean();

		TagProcess tagProcess = new TagProcessBean();
		Collection<Tag> tags = tagProcess.queryTagsByProjectId(project.getId());
		
		Set<String> tmpTagSet = new HashSet<String>();
		for(Tag tag : tags){
			tmpTagSet.add(tag.getName());
		}
		
		Sheet sheet = workbook.getSheetAt(0);
		for(int i = 2; i < sheet.getPhysicalNumberOfRows(); ++i){
			Task task = new Task();
			task.setId(Sequence.getUUID());
			task.setProjectId(project.getId());
			task.setProjectName(project.getName());
			Row row = sheet.getRow(i);
			if(row == null){
				continue;
			}
			
			int j = 0;
			//任务名称校验
			String name = "";
			if(row.getCell(j) != null && row.getCell(j).getCellTypeEnum() == CellType.NUMERIC){
				name = StringUtil.valueOf(row.getCell(j).getNumericCellValue());
			} else if(row.getCell(j) != null && row.getCell(j).getCellTypeEnum() == CellType.STRING){
				name = row.getCell(j).getStringCellValue();
			} else {
				//任务名称不存在，出错
				continue;
			}
			task.setName(name);
			 
			//优先级校验
			++j;
			Integer level = 0;
			if(row.getCell(j) == null || row.getCell(j).getCellTypeEnum() != CellType.NUMERIC){
				 //优先级不存在，使用默认值
				task.setLevel(Task.LEVEL_NOT_IMPORTANT_AND_NOT_URGET);
			} else {
				try{
					level = (int)row.getCell(j).getNumericCellValue();
					if(level < 0 || level > 3){
						errorArray.add((char)(j + 65) + ((i + 1) + "单元格出错，优先级应为数字代码。"));
					}
					task.setLevel(level);
				}catch(Exception e){
					errorArray.add((char)(j + 65) + ((i + 1) + "单元格出错，优先级应为数字代码。"));
				}
			}
			
			//执行人校验
			++j;
			String executer = "";
			if(row.getCell(j) != null && row.getCell(j).getCellTypeEnum() == CellType.NUMERIC){
				executer = StringUtil.valueOf(row.getCell(j).getNumericCellValue());
				Member member = projectProcess.queryMemberByName4Project(project.getId(), executer);
				if(member != null){
					task.setExecutorId(member.getUserId());
					task.setExecutor(member.getUserName());
				} else {
					//成员不存在，出错
					errorArray.add((char)(j + 65) + ((i + 1) + "单元格出错，执行人\"" + executer + "\"不存在该项目内。"));
				}
			} else if(row.getCell(j) != null && row.getCell(j).getCellTypeEnum() == CellType.STRING){
				executer = row.getCell(j).getStringCellValue();
				Member member = projectProcess.queryMemberByName4Project(project.getId(), executer);
				if(member != null){
					task.setExecutorId(member.getUserId());
					task.setExecutor(member.getUserName());
				} else {
					//成员不存在，出错
					errorArray.add((char)(j + 65) + ((i + 1) + "单元格出错，执行人\"" + executer + "\"不存在该项目内。"));
				}
			} else {
				task.setExecutorId(webUser.getId());
				task.setExecutor(webUser.getName());
			}

			//创建人为导入任务操作者
			task.setCreatorId(webUser.getId());
			task.setCreator(webUser.getName());
			
			//创建时间为当前时间
			Date createDate = DateUtil.getToday();
			task.setCreateDate(createDate);
			
			//开始时间校验
			++j;
			Date startDate = DateUtil.getToday();
			if(row.getCell(j) == null){
				 //开始日期不存在，使用当前日期
				task.setStartDate(startDate);
			} else {
				try{
					startDate = row.getCell(j).getDateCellValue();
					task.setStartDate(startDate);
				}catch(Exception e){
					//日期格式错误
					errorArray.add((char) (j + 65) + ((i + 1) + "单元格出错，日期\""
							+ row.getCell(j).getStringCellValue() + "\"格式不正确。"));
				}
			}
			
			//结束时间校验
			++j;
			Date endDate = null;
			if(row.getCell(j) == null){
				 //结束日期不存在，使用空值
			} else {
				try{
					endDate = row.getCell(j).getDateCellValue();
					task.setEndDate(endDate);
				}catch(Exception e){
					//日期格式错误
					errorArray.add((char) (j + 65) + ((i + 1) + "单元格出错，日期\""
							+ row.getCell(j).getStringCellValue() + "\"格式不正确。"));
				}
			}

			//备注校验
			++j;
			Remark remark = new Remark();
			if(row.getCell(j) != null && row.getCell(j).getCellTypeEnum() == CellType.NUMERIC){
				String remarkContent = StringUtil.valueOf(row.getCell(j).getNumericCellValue());
				remark.setContent(remarkContent);
				remark.setCreateDate(DateUtil.getToday());
				remark.setId(Sequence.getUUID());
				remark.setUserId(webUser.getId());
				remark.setCreateRemarkUser(webUser.getName());
				Collection<Remark> remarks = task.getRemarkList();
				remarks.add(remark);
				task.setRemark(JsonUtil.list2JSON(remarks));
			} else if(row.getCell(j) != null && row.getCell(j).getCellTypeEnum() == CellType.STRING){
				String remarkContent = row.getCell(j).getStringCellValue();
				remark.setContent(remarkContent);
				remark.setCreateDate(DateUtil.getToday());
				remark.setId(Sequence.getUUID());
				remark.setUserId(webUser.getId());
				remark.setCreateRemarkUser(webUser.getName());
				Collection<Remark> remarks = task.getRemarkList();
				remarks.add(remark);
				task.setRemark(JsonUtil.list2JSON(remarks));
			}
			
			//标签校验
			++j;
			if(row.getCell(j) != null && row.getCell(j).getCellTypeEnum() == CellType.NUMERIC){
				String importTagString = StringUtil.valueOf(row.getCell(j).getNumericCellValue());
				try{
					if(!tmpTagSet.contains(importTagString)){
						throw new Exception();
					}
				} catch (Exception e){
					errorArray.add((char) (j + 65) + ((i + 1) + "单元格出错，标签\""
							+ importTagString
							+ "\"不存在项目\"" + project.getName() + "\"内。"));
				}
				task.setTags(importTagString);
			} else if(row.getCell(j) != null && row.getCell(j).getCellTypeEnum() == CellType.STRING){
				String importTagString = row.getCell(j).getStringCellValue();
				for(String tag : importTagString.split(",")){
					try{
						if(!tmpTagSet.contains(tag.trim())){
							throw new Exception();
						}
					} catch (Exception e){
						errorArray.add((char) (j + 65) + ((i + 1) + "单元格出错，标签\""
								+ tag
								+ "\"不存在项目\"" + project.getName() + "\"内。"));
					}
				}
				task.setTags(importTagString);
			}
			taskArray.add(task);
		}
		
		if(errorArray.size() == 0){
			rtn.put("task", taskArray);
		} else {
			rtn.put("error", errorArray);
		}
		return rtn;
	}

	private void exportExcel(OutputStream outputStream,DataPackage<Task> datas) {
		try {
			Collection<Task> taskList = datas.getDatas();
			String[] titles = {"任务名称","优先级","创建人","创建时间","开始时间","截止时间","标签","执行人","备注","所属项目","状态"};
			//1、创建工作簿
			HSSFWorkbook workbook = new HSSFWorkbook();
			//1.1、创建头标题样式
			HSSFCellStyle style1 = createCellStyle(workbook, (short)13);
			
			//1.2、创建列标题样式
			HSSFCellStyle style2 = createCellStyle(workbook, (short)12);
			
			//1.3、创建合并单元格对象//起始行号，结束行号，起始列号，结束列号
			CellRangeAddress cellRangeAddress_1 = new CellRangeAddress(0, 0, 0, titles.length - 1);
			
			//2、创建工作表
			HSSFSheet sheet = workbook.createSheet("任务列表");
			//2.1、设置合并单元格对象
			sheet.addMergedRegion(cellRangeAddress_1);
			//2.2、设置默认列宽
			sheet.setDefaultColumnWidth(25);
			
			//3、创建行
			//3.1、创建头标题行并且设置头标题和样式
			HSSFRow row1 = sheet.createRow(0);
			HSSFCell _cell = row1.createCell(0);
			_cell.setCellStyle(style1);
			_cell.setCellValue("任务列表");
			
			//3.2、创建列标题行并且设置列标题和样式
			HSSFRow row2 = sheet.createRow(1);
			
			for(int i = 0; i < titles.length; ++i){
				HSSFCell cell = row2.createCell(i);
				cell.setCellStyle(style2);
				cell.setCellValue(titles[i]);
			}
			
			//4、创建单元格（写入数据库中的任务列表）
			if(taskList != null){
				int rowIndex = 2 ;
				for(Task task : taskList){
					HSSFRow row = sheet.createRow(rowIndex++);
					row.createCell(0).setCellValue(task.getName());
					row.createCell(1).setCellValue(Task.LEVEL_MAP_MAPPING.get(task.getLevel()));
					row.createCell(2).setCellValue(task.getCreator());
					row.createCell(3).setCellValue(DateUtil.getDateStrC(task.getCreateDate()));
					row.createCell(4).setCellValue(DateUtil.getDateStrC(task.getStartDate()));
					row.createCell(5).setCellValue(DateUtil.getDateStrC(task.getEndDate()));
					row.createCell(6).setCellValue(task.getTags());
					row.createCell(7).setCellValue(task.getExecutor());
					StringBuffer remarks = new StringBuffer();
					Collection<Remark> remarkList = task.getRemarkList();
					for(Remark remark:remarkList){
						remarks.append(remark.getCreateRemarkUser())
						.append(": ")
						.append(remark.getContent())
						.append("\n");
					}
					row.createCell(8).setCellValue(remarks.toString());
					row.createCell(9).setCellValue(task.getProjectName());
					row.createCell(10).setCellValue(Task.STATUS_MAP_MAPPING.get(task.getStatus()));
				}
			}
			
			//5、输出
			workbook.write(outputStream);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * 创建单元格样式
	 * @param workbook 工作簿
	 * @param fontSize 字体大小
	 * @return 单元格样式
	 */
	private HSSFCellStyle createCellStyle(HSSFWorkbook workbook, short fontSize){
		HSSFCellStyle style = workbook.createCellStyle();
		style.setAlignment(HorizontalAlignment.CENTER);//水平居中
		style.setVerticalAlignment(VerticalAlignment.CENTER);//垂直居中
		//1.1.1、创建头标题字体
		HSSFFont font = workbook.createFont();
		font.setBold(true);//字体加粗
		font.setFontHeightInPoints(fontSize);
		//在样式中加载字体
		style.setFont(font);
		return style;
	}
	
	//添加接口实现：与dymaticReport结合
	public List<TaskUserCount> listTaskUserCount(String projectId,WebUser user) throws Exception{
		List<Task> listTask=listTasksByProject(projectId, null, null, null, null, null, null, null, null, null, null, null, user);
		Map<String,Map<String,Integer>> maps=new HashMap<String,Map<String,Integer>>();
		List<TaskUserCount> litsTaskUserCount=new ArrayList<TaskUserCount>();
		for (Task task : listTask) {
			//判断是否有值
			Map<String,Integer> map=maps.get(task.getExecutor());
			if(map==null){
				//没值则创建
				map=new HashMap<String,Integer>();
				maps.put(task.getExecutor(), addCount(map, task));
			}else{
				maps.put(task.getExecutor(), addCount(map, task));
			}//有则进行添加
				
		}
		Set<String> keys=   maps.keySet();
  	    Iterator<String> it=keys.iterator();
		//进行listTask赋值
		while(it.hasNext()){
			
			String key=it.next();
			Map<String,Integer> mapTemp=maps.get(key);
			//状态
			int newCount=0;
			int progressCount=0;
			int resolvedCount=0;
			int completeCount=0;
			int rejectCount=0;
			int createCount=0;
			//用三元运算符判断(状态数)
			newCount=mapTemp.get("newCount")!=null?mapTemp.get("newCount"):0;
			progressCount=mapTemp.get("progressCount")!=null?mapTemp.get("progressCount"):0;
			resolvedCount=mapTemp.get("resolvedCount")!=null?mapTemp.get("resolvedCount"):0;
			completeCount=mapTemp.get("completeCount")!=null?mapTemp.get("completeCount"):0;
			rejectCount=mapTemp.get("rejectCount")!=null?mapTemp.get("rejectCount"):0;
			createCount=mapTemp.get("createCount")!=null?mapTemp.get("createCount"):0;
			//优先级
			int levelNotImportantAndNotUrget=0;
			int levelNotImportantAndUrget=0;
			int levelImportantAndNotUrget=0;
			int levelImportantAndUrget=0;
			//用三元运算符判断(优先级)
			levelNotImportantAndNotUrget=mapTemp.get("levelNotImportantAndNotUrget")!=null?mapTemp.get("levelNotImportantAndNotUrget"):0;
			levelNotImportantAndUrget=mapTemp.get("levelNotImportantAndUrget")!=null?mapTemp.get("levelNotImportantAndUrget"):0;
			levelImportantAndNotUrget=mapTemp.get("levelImportantAndNotUrget")!=null?mapTemp.get("levelImportantAndNotUrget"):0;
			levelImportantAndUrget=mapTemp.get("levelImportantAndUrget")!=null?mapTemp.get("levelImportantAndUrget"):0;
			
			TaskUserCount taskUserCount=new TaskUserCount();
			taskUserCount.setExecutor(key);
			taskUserCount.setNewCount(newCount);
			taskUserCount.setProgressCount(progressCount);
			taskUserCount.setResolvedCount(resolvedCount);
			taskUserCount.setComplete(completeCount);
			taskUserCount.setReject(rejectCount);
			taskUserCount.setCreateCount(createCount);
			taskUserCount.setLevelNotImportantAndNotUrget(levelNotImportantAndNotUrget);
			taskUserCount.setLevelNotImportantAndUrget(levelNotImportantAndUrget);
			taskUserCount.setLevelImportantAndNotUrget(levelImportantAndNotUrget);
			taskUserCount.setLevelImportantAndUrget(levelImportantAndUrget);
			
			litsTaskUserCount.add(taskUserCount);
			
		}
		return litsTaskUserCount;
	}
	
	private Map<String,Integer> addCount(Map<String,Integer> map,Task task){
		
		//执行人状态数
		switch(task.getStatus()){
			case Task.STATUS_NEW:
				int newCount=0;
				if(map.get("newCount")!=null)
					newCount=map.get("newCount");
				map.put("newCount", ++newCount);
				break;
			case Task.STATUS_IN_PROGRESS:
				int progressCount=0;
				if(map.get("progressCount")!=null)
					progressCount=map.get("progressCount");
				map.put("progressCount", ++progressCount);
				break;
			case Task.STATUS_RESOLVED:
				int resolvedCount=0;
				if(map.get("resolvedCount")!=null)
					resolvedCount=map.get("resolvedCount");
				map.put("resolvedCount", ++resolvedCount);
				break;
			case Task.STATUS_COMPLETE:
				int completeCount=0;
				if(map.get("completeCount")!=null)
					completeCount=map.get("completeCount");
				map.put("completeCount", ++completeCount);
				break;
			case Task.STATUS_REJECT:
				int rejectCount=0;
				if(map.get("rejectCount")!=null)
					rejectCount=map.get("rejectCount");
				map.put("rejectCount", ++rejectCount);
				break;
		}
		if(task.getCreator().equals(task.getExecutor())){
			int createCount=0;
			if(map.get("createCount")!=null)
				createCount=map.get("createCount");
			map.put("createCount", ++createCount);
		}
		//执行人等级数
		//如果状态已经完成或者作废，则不进行优先级计数
		if(task.getStatus()!=Task.STATUS_COMPLETE && task.getStatus()!=Task.STATUS_REJECT){
			switch(task.getLevel()){
			case Task.LEVEL_NOT_IMPORTANT_AND_NOT_URGET:
				int levelNotImportantAndNotUrget=0;
				if(map.get("levelNotImportantAndNotUrget")!=null)
					levelNotImportantAndNotUrget=map.get("levelNotImportantAndNotUrget");
				map.put("levelNotImportantAndNotUrget", ++levelNotImportantAndNotUrget);
				break;
			case Task.LEVEL_NOT_IMPORTANT_AND_URGET:
				int levelNotImportantAndUrget=0;
				if(map.get("levelNotImportantAndUrget")!=null)
					levelNotImportantAndUrget=map.get("levelNotImportantAndUrget");
				map.put("levelNotImportantAndUrget", ++levelNotImportantAndUrget);
				break;
			case Task.LEVEL_IMPORTANT_AND_NOT_URGET:
				int levelImportantAndNotUrget=0;
				if(map.get("levelImportantAndNotUrget")!=null)
					levelImportantAndNotUrget=map.get("levelImportantAndNotUrget");
				map.put("levelImportantAndNotUrget", ++levelImportantAndNotUrget);
				break;
			case Task.LEVEL_IMPORTANT_AND_URGET:
				int levelImportantAndUrget=0;
				if(map.get("levelImportantAndUrget")!=null)
					levelImportantAndUrget=map.get("levelImportantAndUrget");
				map.put("levelImportantAndUrget", ++levelImportantAndUrget);
				break;
			}
		}
		return map;
	}
}
