package com.bcxin.tenant.bcx.domains.services.impls;

import com.alibaba.fastjson.JSON;
import com.bcxin.tenant.bcx.domains.entities.MetaModuleEntity;
import com.bcxin.tenant.bcx.domains.entities.MetaPageEntity;
import com.bcxin.tenant.bcx.domains.entities.MetaPageVersionEntity;
import com.bcxin.tenant.bcx.domains.repositories.MetaPageRepository;
import com.bcxin.tenant.bcx.domains.repositories.MetaPageVersionRepository;
import com.bcxin.tenant.bcx.domains.repositories.criterias.MetaPageSearchCriteria;
import com.bcxin.tenant.bcx.domains.services.MetaModuleService;
import com.bcxin.tenant.bcx.domains.services.MetaPageService;
import com.bcxin.tenant.bcx.domains.services.commands.DeleteDataCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.BatchLogicalDeletePageCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.CreateMetaPageCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.LogicalDeletePageCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.SavePageFormJsonCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.UpdateMetaPageCommand;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.TenantContext;
import com.bcxin.tenant.bcx.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.bcx.infrastructures.UnitWork;
import com.bcxin.tenant.bcx.infrastructures.components.IdWorker;
import com.bcxin.tenant.bcx.infrastructures.components.JsonProvider;
import com.bcxin.tenant.bcx.infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.bcx.infrastructures.snapshots.metas.MetaPageStructureSettingSnapshot;
import com.bcxin.tenant.bcx.infrastructures.utils.StringUtil;
import org.springframework.stereotype.Service;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 页面元数据服务实现
 */
@Service
public class MetaPageServiceImpl implements MetaPageService {
    private final MetaPageRepository pageRepository;
    private final MetaPageVersionRepository versionRepository;
    private final MetaModuleService moduleService;
    private final UnitWork unitWork;
    private final IdWorker idWorker;
    private final JsonProvider jsonProvider;

    public MetaPageServiceImpl(MetaPageRepository pageRepository, 
                              MetaPageVersionRepository versionRepository,
                              MetaModuleService moduleService,
                              UnitWork unitWork, 
                              IdWorker idWorker,
                              JsonProvider jsonProvider) {
        this.pageRepository = pageRepository;
        this.versionRepository = versionRepository;
        this.moduleService = moduleService;
        this.unitWork = unitWork;
        this.idWorker = idWorker;
        this.jsonProvider = jsonProvider;
    }

    @Override
    public String dispatch(CreateMetaPageCommand command) {
        command.validate();

        if (StringUtil.isEmpty(command.getName())) {
            throw new ArgumentTenantException("页面名称不能为空");
        }

        if (command.getPageType() == null) {
            throw new ArgumentTenantException("页面类型不能为空");
        }

        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel==null) {
            throw new UnAuthorizedTenantException();
        }

        // 如果提供了模块ID，验证模块是否存在并获取项目ID
        String projectId=null;
        if (Objects.nonNull(command.getModuleId())) {
            // 验证模块是否存在并获取项目ID
            MetaModuleEntity module = this.moduleService.getById(command.getModuleId());
            if (module == null) {
                throw new ArgumentTenantException("模块不存在: " + command.getModuleId());
            }
            projectId = module.getProjectId();
        }

        if(StringUtil.isEmpty(projectId)) {
            throw new BadTenantException("项目无效");
        }

        // 验证和规范化formJson
        String validatedFormJson = validateAndNormalizeJson(command.getFormJson());
        String id = String.valueOf(this.idWorker.getNextId());
        MetaPageEntity entity = MetaPageEntity.create(
                id,
                projectId,  // 使用从模块获取的项目ID
                command.getModuleId(),
                command.getName(),
                command.getNote(),
                command.getSlug(),
                command.getLogo(),
                command.getPageType(),
                command.getBusinessType(),
                command.getTableName(),
                command.isEnableBPMFlow(),
                command.isEnableAnonymousAccess(),
                validatedFormJson,  // 使用验证后的JSON
                userModel.getUserType(),
                userModel.getOrganizationId(),
                userModel.getEmployeeId(),
                userModel.getName(),
                command.isSync2Identity(),
                command.getBeginTime(),
                command.getEndTime(),
                command.getAttachment(),
                command.getLimitedCount()
        );
        // 字段值由AOP切面自动填充

        entity.setVersion(1L); // 初始版本为1
        entity.setDeleted(false); // 默认未删除

        Timestamp now = new Timestamp(System.currentTimeMillis());
        entity.setCreatedTime(now);
        entity.setLastUpdatedTime(now);


        MetaPageVersionEntity versionEntity = null;
        // 创建初始版本记录
        if (!StringUtil.isEmpty(validatedFormJson)) {
             versionEntity = MetaPageVersionEntity.create(
                     String.valueOf(this.idWorker.getNextId()),
                     id,
                     1L,
                     validatedFormJson,
                     entity.getFormSettingJson(),
                     now
             );

        }

        MetaPageVersionEntity finalVersionEntity = versionEntity;
        this.unitWork.executeTran(() -> {
            this.pageRepository.insert(entity);
            if(finalVersionEntity !=null) {
                this.versionRepository.insert(finalVersionEntity);
            }
        });

        return id;
    }

    @Override
    public void dispatch(UpdateMetaPageCommand command) {
        command.validate();

        MetaPageEntity entity = getByIdIncludeDeleted(command.getId());
        if (entity == null || entity.isDeleted()) {
            throw new NoFoundTenantException();
        }
        this.unitWork.executeTran(() -> {
            // 更新字段
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
            entity.change(
                    command.getName() != null ? command.getName() : entity.getName(),
                    command.getNote() != null ? command.getNote() : entity.getNote(),
                    command.getSlug() != null ? command.getSlug() : entity.getSlug(),
                    command.getLogo() != null ? command.getLogo() : entity.getLogo(),
                    command.getPageType() != null ? command.getPageType() : entity.getPageType(),
                    command.getBusinessType() != null ? command.getBusinessType() : entity.getBusinessType(),
                    command.getTableName() != null ? command.getTableName() : entity.getTableName(),
                    command.getEnableBPMFlow(),
                    command.getEnableAnonymousAccess(),
                    command.isSync2Identity(),
                    command.getBeginTime(),
                    command.getEndTime(),
                    command.getAttachment(),
                    command.getLimitedCount(),
                    userModel
            );

            this.pageRepository.update(entity);
        });
    }

    @Override
    public void dispatch(SavePageFormJsonCommand command) {
        command.validate();
        // 验证和规范化formJson
        String validatedFormJson = validateAndNormalizeJson(command.getFormJson());

        // 验证和规范化formSettingJson
        String validatedSettingJson = validateAndNormalizeJson(command.getFormSettingJson());
        // 根据ID查询现有实体（包括已删除的）
        MetaPageEntity entity = getByIdIncludeDeleted(command.getId());
        if (entity == null || entity.isDeleted()) {
            throw new NoFoundTenantException();
        }

        // 更新表单字段
        boolean hasChanged = entity.change(
                validatedFormJson,
                validatedSettingJson,
                command.isForMobile(),
                TenantContext.getInstance().getUserContext().get()
        );

        MetaPageVersionEntity versionEntity = null;
        // 如果表单结构或表单设置发生变化，保存新版本
        if (hasChanged) {
            versionEntity = MetaPageVersionEntity.create(
                    String.valueOf(this.idWorker.getNextId()),
                    entity.getId(),
                    entity.getVersion(),
                    entity.getFormJson(),
                    entity.getFormSettingJson(),
                    entity.getLastUpdatedTime()
            );
        }

        MetaPageVersionEntity finalVersionEntity = versionEntity;
        this.unitWork.executeTran(() -> {
            this.pageRepository.update(entity);

            if (finalVersionEntity != null) {
                this.versionRepository.insert(finalVersionEntity);
            }
        });
    }

    @Override
    public void dispatch(DeleteDataCommand command) {
        command.validate();
        this.unitWork.executeTran(() -> {
            // 根据ID查询现有实体（包括已删除的）
            MetaPageEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null || entity.isDeleted()) {
                throw new NoFoundTenantException();
            }

            // 删除页面（版本记录可保留作为历史记录）
            this.pageRepository.delete(command.getId());
        });
    }

    @Override
    public void dispatch(LogicalDeletePageCommand command) {
        command.validate();

        this.unitWork.executeTran(() -> {
            // 根据ID查询现有实体（包括已删除的）
            MetaPageEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null || entity.isDeleted()) {
                throw new NoFoundTenantException();
            }

            // 标记为已删除
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
            entity.changeMarkAsDeleted(userModel);

            // 保存更新
            this.pageRepository.update(entity);
        });
    }

    @Override
    public void dispatch(BatchLogicalDeletePageCommand command) {
        command.validate();

        this.unitWork.executeTran(() -> {
            List<String> ids = command.getIds();
            if (ids == null || ids.isEmpty()) {
                return;
            }
            
            // 1. 批量查询所有entities（一次查询）
            List<MetaPageEntity> entities = this.pageRepository.getByIds(ids);
            
            // 2. 过滤出未删除的validIds
            List<String> validIds = new ArrayList<>();
            
            for (MetaPageEntity entity : entities) {
                if (entity != null && !entity.isDeleted()) {
                    validIds.add(entity.getId());
                }
            }
            
            // 如果没有有效的ID需要删除，直接返回
            if (validIds.isEmpty()) {
                return;
            }
            
            // 3. 批量更新所有entities为已删除（一次更新）
            Timestamp now = new Timestamp(System.currentTimeMillis());
            this.pageRepository.batchMarkAsDeleted(validIds, now);
        });
    }

    @Override
    public EntityCollection<MetaPageEntity> search(MetaPageSearchCriteria criteria) {
        return this.pageRepository.search(criteria);
    }

    /**
     * 获取页面实体（包括已删除的记录）
     * 仅供内部使用
     */
    private MetaPageEntity getByIdIncludeDeleted(String id) {
        return this.pageRepository.getById(id);
    }

    /**
     * 验证和规范化JSON字符串
     * @param json 原始JSON字符串
     * @return 验证后的JSON字符串，如果为空或无效则返回null
     */
    private String validateAndNormalizeJson(String json) {
        // 如果为null或空字符串或仅包含空白字符，返回null
        if (json == null || json.trim().isEmpty()) {
            return null;
        }
        
        try {
            // 验证JSON格式是否正确
            JSON.parse(json);
            return json;
        } catch (Exception e) {
            // 不传递原始异常对象，避免Dubbo序列化问题
            throw new IllegalArgumentException("无效的JSON格式: " + e.getMessage());
        }
    }
}

