package com.bcxin.backend.jobs;

import com.bcxin.backend.certificateSignatures.CertificateSignatureProvider;
import com.bcxin.backend.documentSignatrures.SignatureConfigProperty;
import com.bcxin.backend.tasks.TaskAbstract;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Collection;

/**
 * 资格认证签章功能定时任务
 */
@Component
@ConditionalOnProperty(prefix = "myapps.signature",name = "enable",havingValue = "true")
public class CertificateSignatureScheduleJob extends ScheduleJobAbstract {
    private final static Logger logger = LoggerFactory.getLogger(CertificateSignatureScheduleJob.class);
    private final SignatureConfigProperty signatureConfigProperty;

    public CertificateSignatureScheduleJob(SignatureConfigProperty signatureConfigProperty) {
        this.signatureConfigProperty = signatureConfigProperty;
    }

    /**
     * 步骤一、将待签章数据放入队列
     */
    @Scheduled(fixedRate = 1 * 60 * 1000)
    public void step1_prepareData() {
        CertificateSignatureProvider certificateSignatureProvider = this.getBean(
                signatureConfigProperty.getAreaCode(),
                CertificateSignatureProvider.class
        );

        if (certificateSignatureProvider == null) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getAreaCode());
            return;
        }

        certificateSignatureProvider.step1_prepareData();
    }


    /**
     * 步骤二、生成待签章的PDF
     */
    @Scheduled(cron = "${myapps.signature.create_pdf_schedule}") // 测试环境5分钟一次
    public void step2_generatePdf() {
        CertificateSignatureProvider certificateSignatureProvider = this.getBean(
                signatureConfigProperty.getAreaCode(),
                CertificateSignatureProvider.class
        );

        if (certificateSignatureProvider == null) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getAreaCode());
            return;
        }

        certificateSignatureProvider.step2_generatePdf();
    }

    /**
     * 步骤三、拷贝PDF文件到IN目录并执行签章操作; 对应于签章的
     * convertDto.getTempUrl().replace("/temp", "/in") 与 json.put("in", "/data/share/in"); 这边在服务器上是有映射关系的.
     * 签章后的目录为: json.put("out", "/data/share/out"); 对应于我们的convertDto.getTempUrl().replace("/temp", "/out")
     *
     * 其中: convertDto.getTempUrl() 为rootPath + /uploads/pdf/dianziqianzhang/temp
     */
    @Scheduled(cron = "${myapps.signature.cgi_schedule}") // 测试环境3分钟一次
    public void step3_doSignature() {
        CertificateSignatureProvider certificateSignatureProvider = this.getBean(
                signatureConfigProperty.getAreaCode(),
                CertificateSignatureProvider.class
        );

        if (certificateSignatureProvider == null) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getAreaCode());
            return;
        }
        certificateSignatureProvider.step3_doSignature();
    }


    /**
     * <b> 步骤四: 针对签章完成的操作; 扫描解析out目录的文件, 复制到业务目录并更改签章状态 </b>
     */
    @Scheduled(cron = "${myapps.signature.move_schedule}") // 测试环境3分钟一次
    public void step4_doSignature() {
        CertificateSignatureProvider certificateSignatureProvider = this.getBean(signatureConfigProperty.getAreaCode(),
                CertificateSignatureProvider.class
        );

        if (certificateSignatureProvider == null) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getAreaCode());
            return;
        }
        certificateSignatureProvider.step4_moveOutDocument2BusinessDirAndFinishSignature();
    }

    @Scheduled(cron = "${myapps.signature.change_schedule}") // 测试环境1小时一次
    public void step5_doneForExceptionData() {
        CertificateSignatureProvider certificateSignatureProvider = this.getBean(signatureConfigProperty.getAreaCode(),
                CertificateSignatureProvider.class
        );

        if (certificateSignatureProvider == null) {
            logger.error("找不到符合实现:{}", signatureConfigProperty.getAreaCode());
            return;
        }
        certificateSignatureProvider.step5_doneForExceptionData();
    }


    /**
     * ####################################################################开始#############文书签章功能
     */
    /**
     * <b> 检索证书管理记录生成符合条件的SignatureQueues签章记录 </b>
     * 5分钟执行一次
     * @author ZXF
     * @create 2022/10/20 0020 17:28
     * @version
     * @注意事项 </b>
     */
    //@Scheduled(fixedRate = 1 * 60 * 1000)
    @Deprecated
    public void signatureToCreate() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("execute.signature.pdf.job.create_signature");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    /**
     * <b> 模板加载生成pdf证书 </b>
     * 5分钟执行一次
     * @author ZXF
     * @create 2022/10/20 0020 17:28
     * @version
     * @注意事项 </b>
     */
    //@Scheduled(cron = "${myapps.signature.create_pdf_schedule}") // 测试环境5分钟一次
    @Deprecated
    public void signatureToPDF() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("execute.signature.pdf.job.create_pdf");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    /**
     * <b> 临时目录pdf剪切到in目录触发签章，结束电子签章流程 </b>
     * @author ZXF
     * @create 2023/01/07 0007 11:47
     * @version
     * @注意事项 </b>
     */
    //@Scheduled(cron = "${myapps.signature.move_schedule}") // 测试环境3分钟一次
    @Deprecated
    public void signatureMovePDF() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("execute.signature.pdf.job.move");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    /**
     * <b> 临时目录pdf剪切到in目录触发签章 </b>
     * @author ZXF
     * @create 2023/01/07 0007 11:47
     * @version
     * @注意事项 </b>
     */
    //@Scheduled(cron = "${myapps.signature.cgi_schedule}") // 测试环境3分钟一次
    @Deprecated
    public void signQuicklyExCGI() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("execute.signature.pdf.job.cgi");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    /**
     * <b> in目录下的文件是否在out目录下存在，存在就删除in目录对应文件 </b>
     * @author ZXF
     * @create 2023/01/07 0007 11:47
     * @version
     * @注意事项 </b>
     */
    //@Scheduled(cron = "${myapps.signature.del_schedule}") // 测试环境30秒一次
    @Deprecated
    public void signatureDelInPDF() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("execute.signature.pdf.job.del");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }

    /**
     * <b> 扫描签章表中状态（status）为3（已生成pdf待签章的数据）且当前时间大于上一次更新时间（last_processed_time 3小时以上,将其状态改为0 </b>
     * @author LYM
     * @create 2023/05/06 0001 10:27
     * @version
     * @注意事项 </b>
     */
    //@Scheduled(cron = "${myapps.signature.change_schedule}") // 测试环境1小时一次
    @Deprecated
    public void signatureChangeStatus() {
        Collection<TaskAbstract> taskAbstracts = this.getTasks("execute.signature.pdf.job.change");
        taskAbstracts.forEach(task -> {
            task.run();
        });
    }
}
