package com.bcxin.platform.service.oauth;

import com.bcxin.platform.util.ByteAndObject;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.concurrent.TimeUnit;

@Service
public class RedisUtil {

	@Resource
	protected RedisTemplate<String, Object> redisTemplate;

	public final static int CAHCETIME = 60;// 默认缓存时间 60S
	public final static int CAHCEHOUR = 60 * 60;// 默认缓存时间 1hr
	public final static int CAHCEDAY = 60 * 60 * 24;// 默认缓存时间 1Day
	public final static int CAHCEWEEK = 60 * 60 * 24 * 7;// 默认缓存时间 1week
	public final static int CAHCEMONTH = 60 * 60 * 24 * 7 * 30;// 默认缓存时间 1month


	@Value("${spring.redisPrefixKey}")
	public String REDIS_PREFIX_KEY="PASP";

	/**
	 * Redis序列化
	 * 
	 * @return
	 */
	protected RedisSerializer<String> getRedisSerializer() {
		return redisTemplate.getStringSerializer();
	}

	public boolean putCache(String key, Object value, int seconds) {
		final byte[] bkey = (REDIS_PREFIX_KEY+":"+key).getBytes();
		final byte[] bvalue = ByteAndObject.ObjectToByte(value);
		boolean result = redisTemplate.execute(new RedisCallback<Boolean>() {
			@Override
			public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
				if (seconds == 0) {
					return connection.setNX(bkey, bvalue);
				} else {
					connection.setNX(bkey, bvalue);
					return connection.expire(bkey, TimeUnit.SECONDS.toSeconds(seconds));
				}
			}
		});
		return result;
	}

	public Object getCache(String key) {

		if (key == null) {
			return null;
		}
		byte[] result = redisTemplate.execute(new RedisCallback<byte[]>() {
			@Override
			public byte[] doInRedis(RedisConnection connection) throws DataAccessException {
				return connection.get((REDIS_PREFIX_KEY+":"+key).getBytes());
			}
		});
		if (result == null) {
			return null;
		}
		Object obj = ByteAndObject.ByteToObject(result);

		return obj;
	}

	public Long delete(String key) {
		final byte[] bkey = (REDIS_PREFIX_KEY+":"+key).getBytes();
		return redisTemplate.execute(new RedisCallback<Long>() {
			@Override
			public Long doInRedis(RedisConnection connection) throws DataAccessException {
				return connection.del(bkey);
			}
		});
	}

	/**
	 * 删除缓存
	 *
	 * @param key 可以传一个值 或多个
	 */
	public void del(String... key) {
		if (key != null && key.length > 0) {
			if (key.length == 1) {
				redisTemplate.delete(key[0]);
			} else {
				redisTemplate.delete(CollectionUtils.arrayToList(key));
			}
		}
	}

	/**
	 * 判断key是否存在
	 *
	 * @param key 键
	 * @return true 存在 false不存在
	 */
	public boolean hasKey(String key) {
		try {
			return redisTemplate.hasKey(key);
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * 指定缓存失效时间
	 *
	 * @param key  键
	 * @param time 时间(秒)
	 * @return
	 */
	public boolean expire(String key, long time) {
		try {
			if (time > 0) {
				redisTemplate.expire(key, time, TimeUnit.SECONDS);
			}
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}

	public boolean putStringCache(String key, String value, int seconds) {

		final byte[] bkey = (REDIS_PREFIX_KEY+":"+key).getBytes();
		final byte[] bvalue = value.getBytes();
		boolean result = redisTemplate.execute(new RedisCallback<Boolean>() {
			@Override
			public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
				if (seconds == 0) {
					return connection.setNX(bkey, bvalue);
				} else {
					connection.setNX(bkey, bvalue);
					return connection.expire(bkey, TimeUnit.SECONDS.toSeconds(seconds));
				}

			}
		});
		return result;
	}

	public String getStringCache(String key) {
		String obj = null;
		try {

			if (key == null) {
				return null;
			}
			byte[] result = redisTemplate.execute(new RedisCallback<byte[]>() {
				@Override
				public byte[] doInRedis(RedisConnection connection) throws DataAccessException {
					return connection.get((REDIS_PREFIX_KEY+":"+key).getBytes());
				}
			});
			if (result == null) {
				return null;
			}
			obj = new String(result, "GB2312");

		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}

		return obj;
	}

	public boolean putListCache(String key, List<?> list, int seconds) {
		final byte[] bkey = (REDIS_PREFIX_KEY+":"+key).getBytes();
		final byte[] bvalue = ByteAndObject.serializeList(list);
		boolean result = redisTemplate.execute(new RedisCallback<Boolean>() {
			@Override
			public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
				if (seconds == 0) {
					return connection.setNX(bkey, bvalue);
				} else {
					connection.setNX(bkey, bvalue);
					return connection.expire(bkey, TimeUnit.SECONDS.toSeconds(seconds));
				}
			}
		});
		return result;
	}

	public List<?> getListCache(String key) {
		if (key == null) {
			return null;
		}
		byte[] result = redisTemplate.execute(new RedisCallback<byte[]>() {
			@Override
			public byte[] doInRedis(RedisConnection connection) throws DataAccessException {
				return connection.get((REDIS_PREFIX_KEY+":"+key).getBytes());
			}
		});
		if (result == null) {
			return null;
		}
		List<?> list = ByteAndObject.unserializeList(result);
		return list;
	}

	/**
	 * redis防重复提交
	 *
	 * @param id
	 * @return
	 */
	public boolean getRedisRepetition(String id) {
		//每次访问 + 1 (可以实现计数器)
		Long count = redisTemplate.opsForValue().increment("repeat_submit:" + id, 1);
		if (count == 1) {
			redisTemplate.expire("repeat_submit:" + id, 5, TimeUnit.SECONDS);
			return true;
		}
		return false;
	}

	/**
	 * <b> 根据key存取值 </b>
	 * @author ZXF
	 * @create 2023/05/10 0010 15:02
	 * @version
	 * @注意事项 </b>
	 */
	public boolean setNx(String key, String value, long timeout) {
		return redisTemplate
				.opsForValue()
				.setIfAbsent(key, value, timeout, TimeUnit.SECONDS);
	}
}
