package com.bcxin.platform.service.oauth;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.bcxin.obpm.domain.ObpmUser;
import com.bcxin.obpm.mapper.ObpmDepartmentMapper;
import com.bcxin.obpm.mapper.ObpmUserMapper;
import com.bcxin.platform.common.constant.Constants;
import com.bcxin.platform.common.exception.BusinessException;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.common.utils.StringUtils;
import com.bcxin.platform.domain.company.PerBaseInfo;
import com.bcxin.platform.dto.oauth.AccessToken;
import com.bcxin.platform.dto.oauth.OAuthTokenxRequest;
import com.bcxin.platform.mapper.company.ComDepartMapper;
import com.bcxin.platform.mapper.company.PerBaseInfoMapper;
import com.bcxin.platform.service.system.ISysConfigService;
import com.bcxin.platform.util.AESUtil;
import com.bcxin.platform.util.constants.CommonConst;
import com.github.pagehelper.util.StringUtil;
import org.apache.commons.codec.binary.Hex;
import org.apache.oltu.oauth2.as.issuer.MD5Generator;
import org.apache.oltu.oauth2.as.issuer.OAuthIssuer;
import org.apache.oltu.oauth2.as.issuer.OAuthIssuerImpl;
import org.apache.oltu.oauth2.as.response.OAuthASResponse;
import org.apache.oltu.oauth2.common.error.OAuthError;
import org.apache.oltu.oauth2.common.exception.OAuthProblemException;
import org.apache.oltu.oauth2.common.exception.OAuthSystemException;
import org.apache.oltu.oauth2.common.message.OAuthResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * OAuthToken2.0 客户端授权
 *
 * @author zhangye
 * @version 1.0
 */
@Service
@Transactional("transactionManager")
public class OAuthServiceImpl implements OAuthService {

    private static Logger logger = LoggerFactory.getLogger(OAuthServiceImpl.class);

    @Resource
    private PerBaseInfoMapper perBaseInfoMapper;

    @Resource
    private RedisUtil redisUtil;

    @Autowired
    private ISysConfigService sysConfigService;

    @Resource
    private ObpmUserMapper obpmUserMapper;

    @Resource
    private ObpmDepartmentMapper obpmDepartmentMapper;

    @Resource
    private ComDepartMapper comDepartMapper;


    /**
     * 获取客户端授权AccessToken
     */
    @Override
    public Result saveDispatch(HttpServletRequest request, HttpServletResponse response, String clientType)
            throws Exception {

        OAuthTokenxRequest oauthRequest = new OAuthTokenxRequest(request);
        Map<Boolean, OAuthResponse> handleResult = handle(oauthRequest,clientType);

        if (handleResult.containsKey(true)) {
            return Result.success(Result.SUCCESS_MSG,
                    handleResult.get(true) == null ? null : JSON.parse(handleResult.get(true).getBody()));
        } else {
            return Result.fail(handleResult.get(false) == null ? "参数错误！" : JSON.parseObject(handleResult.get(false).getBody()).getString("error_description"));
        }
    }

    /**
     * 校验客户端参数，验证成功返回AccessToken
     *
     * @return
     * @throws OAuthProblemException
     * @throws OAuthSystemException
     */
    private Map<Boolean, OAuthResponse> handle(OAuthTokenxRequest oauthRequest, String clientType) throws OAuthProblemException, OAuthSystemException {

        Map<Boolean, OAuthResponse> result = new HashMap<Boolean, OAuthResponse>();
        OAuthResponse validateResponse = null;


        String userName = oauthRequest.getUsername();
        String passWord = oauthRequest.getPassword();
        String publicKey = oauthRequest.getPublicKey();

        if(StringUtils.isEmpty(userName) || StringUtils.isEmpty(passWord) || StringUtils.isEmpty(publicKey)){
            validateResponse = OAuthResponse.errorResponse(HttpServletResponse.SC_BAD_REQUEST)
                    .setError(OAuthError.TokenResponse.INVALID_GRANT).setErrorDescription("参数错误").buildJSONMessage();

            result.put(false, validateResponse);
            return result;
        }

        try {
            byte[] bytes = publicKey.getBytes();
            byte[] data = Hex.decodeHex(passWord.toCharArray());
            byte[] s = new byte[0];
            s = AESUtil.AES_CBC_Decrypt(data, bytes, bytes);
            passWord = new String(s);
            if(!userName.equals(passWord)){
                validateResponse = OAuthResponse.errorResponse(HttpServletResponse.SC_BAD_REQUEST)
                        .setError(OAuthError.TokenResponse.INVALID_GRANT).setErrorDescription("用户或者密码错误").buildJSONMessage();
                result.put(false, validateResponse);
                return result;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        PerBaseInfo perBaseInfo = perBaseInfoMapper.selectPerBaseInfoByTlkId(userName);
        if(perBaseInfo == null){
            validateResponse = OAuthResponse.errorResponse(HttpServletResponse.SC_BAD_REQUEST)
                    .setError(OAuthError.TokenResponse.INVALID_GRANT).setErrorDescription("用户不存在").buildJSONMessage();

            result.put(false, validateResponse);
            return result;
        }

        OAuthResponse oAuthResponse = handleAfterValidation(oauthRequest.getRedirectURI(),clientType, perBaseInfo);
        result.put(true, oAuthResponse);
        return result;
    }

    private OAuthResponse handleAfterValidation(String redirectURI, String clientType,PerBaseInfo perBaseInfo) throws OAuthProblemException, OAuthSystemException {
        AccessToken accessToken = retrieveAccessToken(clientType,perBaseInfo);
        OAuthResponse tokenResponse = createTokenResponse(redirectURI,accessToken);
        return tokenResponse;

    }

    private OAuthResponse createTokenResponse(String redirectURI,AccessToken accessToken)
            throws OAuthSystemException {
        final OAuthASResponse.OAuthTokenResponseBuilder builder = OAuthASResponse
                .tokenResponse(HttpServletResponse.SC_OK).location(redirectURI)
                .setAccessToken(accessToken.getTokenId())
                .setExpiresIn(String.valueOf(accessToken.currentTokenExpiredSeconds()))
                .setTokenType(accessToken.getTokenType());
        final String refreshToken = accessToken.getRefreshToken();
        if (StringUtils.isNotEmpty(refreshToken)) {
            builder.setRefreshToken(refreshToken);
        }
        if (StringUtils.isNotEmpty(redirectURI)) {
            builder.setParam("redirect_uri", redirectURI);
        }
        return builder.buildJSONMessage();
    }

    @Override
    public AccessToken getAccessToken(String clientType, PerBaseInfo perBaseInfo) throws OAuthSystemException {
        String loginId = perBaseInfo.getPerId()+"";
        AccessToken accessToken = null;
        // 用客户端id和用户账号获取accessToken
        String value = redisUtil.getStringCache(loginId);
        // 获取accessToken
        if (StringUtils.isNotEmpty(value)) {
            accessToken = (AccessToken) redisUtil.getCache(value);
            // 只允许一个账号登陆
            if (accessToken != null) {
                return accessToken;
            }
        }
        accessToken = createAccessToken(perBaseInfo,clientType);
        return accessToken;
    }

    /**
     * 验证AccessToken是否已经存在，不存在则新创建
     */
    private AccessToken retrieveAccessToken(String clientType, PerBaseInfo perBaseInfo) throws OAuthSystemException {
        String loginId = perBaseInfo.getPerId()+"";
        AccessToken accessToken = null;
        // 用客户端id和用户账号获取accessToken
        String value = redisUtil.getStringCache(loginId);
        // 获取accessToken
        if (StringUtils.isNotEmpty(value)) {
            accessToken = (AccessToken) redisUtil.getCache(value);
            // 只允许一个账号登陆
            if (accessToken != null) {
                redisUtil.delete(loginId);
                redisUtil.delete(accessToken.getTokenId());
            }
        }
        accessToken = createAccessToken(perBaseInfo,clientType);
        return accessToken;
    }

    /**
     * 创建AccessToken
     *
     * @param perBaseInfo
     * @param clientType
     * @return
     * @throws OAuthSystemException
     */
    private AccessToken createAccessToken(PerBaseInfo perBaseInfo,String clientType) throws OAuthSystemException {
        // access_token生成器
        OAuthIssuer oauthIssuer = new OAuthIssuerImpl(new MD5Generator());
        // access_token
        String accessTokenString = oauthIssuer.accessToken();
        AccessToken accessToken = null;
        int seconds = 0; // 过期时间
        // IM 客户端KEY
        if (clientType.equals(CommonConst.CLIENTTYPE_WEB)) {
            // WEB端
            seconds = AccessToken.ACCESS_TOKEN_WEB_VALIDITY_SECONDS;
        } else if (clientType.equals(CommonConst.CLIENTTYPE_MOBILE)) {
            // 手机端
            seconds = AccessToken.ACCESS_TOKEN_MOBILE_VALIDITY_SECONDS;
        }
        // 根据用户ID获取公司ID
        accessToken = new AccessToken();
        accessToken.setComId(perBaseInfo.getComId().toString());
        accessToken.setPerId(perBaseInfo.getPerId().toString());

        ObpmUser obpmUser = obpmUserMapper.selectObpmUserById(perBaseInfo.getTlkPerId());

        if(obpmUser != null){
            if(Constants.TRUE.equals(obpmUser.getISDOMAINUSER())){
                accessToken.setDomainAdmin(true);
            }else{
                List<String> tlkDepartIds = obpmDepartmentMapper.getAdminDepartIds(perBaseInfo.getTlkPerId());
                if(tlkDepartIds.size() > 0) {
                    List<String> departIds = comDepartMapper.findByTlkIds(tlkDepartIds);
                    accessToken.setAdminOrgIds(JSON.toJSONString(departIds));
                }
            }
        }

        //accessToken.setUserId(perBaseInfo.getUserId().toString());
        accessToken.setName(perBaseInfo.getName());
        accessToken.setTokenId(accessTokenString);
        accessToken.setTokenExpiredSeconds(seconds);
        accessToken.setRefreshToken(oauthIssuer.refreshToken());
        // 客户端id和用户名作为key access_token作为value存入redis
        redisUtil.putStringCache(perBaseInfo.getPerId()+"", accessTokenString, seconds);
        redisUtil.putCache(accessTokenString, accessToken, seconds);
        return accessToken;
    }

    /**
     * 获取AccessToken信息
     */
    @Override
    public AccessToken getAccessToken(String access_token) {
        AccessToken accessToken = (AccessToken)redisUtil.getCache(access_token);
        if (accessToken != null) {
            // accessToken过期
            if (accessToken.tokenExpired()) {
                accessToken = null;
            }
        }
        return accessToken;
    }

    /**
     * 获取AccessToken信息 v5 token
     */
    @Override
    public AccessToken getAccessToken(String access_token,boolean userV5Token) {
        AccessToken accessToken = getAccessToken(access_token);
        if (accessToken == null && userV5Token) {
            try {
                String v5Url = sysConfigService.selectConfigByKey(Constants.V5_URL);
                if(StringUtil.isEmpty(v5Url)){
                    return null;
                }

                v5Url += "?accessToken=" + access_token;
                System.err.println("====>getAccessToken.v5Url:"+v5Url);
                String result = HttpUtil.get(v5Url);
                System.err.println("====>getAccessToken.result:"+result);
                Result tlkResult = JSON.parseObject(result, Result.class);
                Map resultMap = JSON.parseObject(tlkResult.getData().toString(), Map.class);
                String tlkUserId = !ObjectUtil.isNull(resultMap.get("id")) ? resultMap.get("id").toString() : "";
                PerBaseInfo perBaseInfo = perBaseInfoMapper.selectPerBaseInfoByTlkId(tlkUserId);
                /*if(perBaseInfo == null){
                    try{
                        obpmDataSyncService.getV5InfoByTlkPerId(tlkUserId);
                    }catch (Exception e) {
                        e.printStackTrace();
                        throw new BusinessException("用户不存在：" + tlkUserId);
                    }
                }
                perBaseInfo = perBaseInfoMapper.selectPerBaseInfoByTlkId(tlkUserId);*/
                if(perBaseInfo == null){
                    throw new BusinessException("用户不存在：" + tlkUserId);
                }
                // 根据用户ID获取公司ID
                accessToken = new AccessToken();
                accessToken.setComId(perBaseInfo.getComId().toString());
                accessToken.setPerId(perBaseInfo.getPerId().toString());

                ObpmUser obpmUser = obpmUserMapper.selectObpmUserById(perBaseInfo.getTlkPerId());

                if(obpmUser != null){
                    if(Constants.TRUE.equals(obpmUser.getISDOMAINUSER())||"1".equals(obpmUser.getISDOMAINUSER())){
                        accessToken.setDomainAdmin(true);
                    }else{
                        List<String> tlkDepartIds = obpmDepartmentMapper.getAdminDepartIds(perBaseInfo.getTlkPerId());
                        if(tlkDepartIds.size() > 0) {
                            List<String> departIds = comDepartMapper.findByTlkIds(tlkDepartIds);
                            accessToken.setAdminOrgIds(JSON.toJSONString(departIds));
                        }
                    }
                }

                //accessToken.setUserId(perBaseInfo.getUserId().toString());
                accessToken.setName(perBaseInfo.getName());
                accessToken.setTokenExpiredSeconds(AccessToken.ACCESS_TOKEN_WEB_VALIDITY_SECONDS);
                accessToken.setTokenId(access_token);
                // 客户端id和用户名作为key access_token作为value存入redis
                redisUtil.putStringCache(perBaseInfo.getPerId()+"", access_token, accessToken.getTokenExpiredSeconds());
                redisUtil.putCache(access_token, accessToken, accessToken.getTokenExpiredSeconds());
            }catch (Exception e){
                e.printStackTrace();
                logger.error("v5_access_token错误！");
                return null;
            }
        }
        return accessToken;
    }

    public static void main(String[] args) throws Exception {
        String userName = "15601000000";
        String passWord = "86b955d41be7a1ce4b8db67da688103e";
        String publicKey = "wdMFAbNpSmRJEmOl";
        if (StringUtils.isNotEmpty(publicKey)) {
            byte[] bytes = publicKey.getBytes();

            byte[] data = Hex.decodeHex(passWord.toCharArray());
            byte[] s = AESUtil.AES_CBC_Decrypt(data, bytes, bytes);
            passWord = new String(s);
            System.out.println(passWord);
        } else {
        }
    }

}
