package com.bcxin.platform.service.grant;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.bcxin.platform.common.constant.DictConst;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.IdWorker;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.common.utils.StringUtils;
import com.bcxin.platform.domain.grant.ComBhGrantDetail;
import com.bcxin.platform.domain.grant.ComBhGrantInfo;
import com.bcxin.platform.domain.grant.ComBhGrantOperRecord;
import com.bcxin.platform.dto.grant.ComBhGrantInfoDTO;
import com.bcxin.platform.mapper.grant.ComBhGrantDetailMapper;
import com.bcxin.platform.mapper.grant.ComBhGrantInfoMapper;
import com.bcxin.platform.mapper.grant.ComBhGrantOperRecordMapper;
import com.bcxin.platform.service.common.CommonService;
import com.bcxin.platform.util.*;
import com.bcxin.platform.util.codes.BhPaymentServiceConst;
import com.bcxin.platform.util.constants.CommonConst;
import com.bcxin.platform.util.excel.ExcelEnum;
import com.bcxin.platform.util.excel.ExcelUtil;
import com.bcxin.platform.util.file.BcxinFileUtils;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.util.StringUtil;
import com.google.common.collect.Maps;
import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.*;

/**
 * 企业渤海薪资发放
 *
 * @author llc
 * @date 2020-08-13
 */
@Service
@Transactional("transactionManager")
public class ComBhGrantServiceImpl implements ComBhGrantService {

    private static final Logger logger = LoggerFactory.getLogger(ComBhGrantServiceImpl.class);

    @Resource
    private ComBhGrantDetailMapper comBhGrantDetailMapper;

    @Resource
    private ComBhGrantInfoMapper comBhGrantInfoMapper;

    @Resource
    private ComBhGrantOperRecordMapper comBhGrantOperRecordMapper;

    @Resource
    private CommonService commonService;

    @Resource
    private IdWorker idWorker;

    private final static String BHGRANTFILETYPE = "4"; // 渤海代发文件类型

    private final static String BHGRANTQDDM = "C5"; // 渤海代发渠道代码


    /**
     * 获取企业渤海薪资代发记录
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-08-14
     */
    @Override
    public Result pageComBhGrantInfoList(ComBhGrantInfoDTO dto) throws V5BusinessException {
        if (dto.getPageNumber() == null) {
            throw new V5BusinessException(Result.ERROR, "第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            throw new V5BusinessException(Result.ERROR, "每页多少行不能为空");
        }
        PageHelper.startPage(dto.getPageNumber(), dto.getPageSize());
        List<Map<String, String>> list = comBhGrantInfoMapper.pageComBhGrantInfoList(dto);
        PageInfoUtils pageInfo = new PageInfoUtils(list);
        return Result.success(Result.SUCCESS_QUERY_MSG, pageInfo);
    }

    /**
     * 新增企业渤海薪资代发
     *
     * @param comBhGrantInfo
     * @return
     * @author llc
     * @date 2020-08-13
     */
    @Override
    public Result saveComBhGrantInfo(ComBhGrantInfo comBhGrantInfo) throws V5BusinessException {
        if (StringUtil.isEmpty(comBhGrantInfo.getTitle())) {
            throw new V5BusinessException(Result.ERROR, "标题不能为空");
        }
        if (StringUtil.isEmpty(comBhGrantInfo.getGrantType())) {
            throw new V5BusinessException(Result.ERROR, "发放类型不能为空");
        }
        comBhGrantInfo.setComBhGrantId(idWorker.nextId());
        comBhGrantInfo.setSumAmount(BigDecimal.ZERO);
        comBhGrantInfo.setGrantCount(0);
        comBhGrantInfo.setTotalSumAmount(BigDecimal.ZERO);
        comBhGrantInfo.setTotalGrantCount(0);
        /*** 插入企业发放信息 ***/
        comBhGrantInfoMapper.insertComBhGrantInfo(comBhGrantInfo);

        /*** 插入操作记录 ***/
        ComBhGrantOperRecord record = new ComBhGrantOperRecord();
        record.setCreateBy(comBhGrantInfo.getCreateBy());
        record.setCreateTime(DateUtil.date());
        record.setComId(comBhGrantInfo.getComId());
        record.setComBhGrantId(comBhGrantInfo.getComBhGrantId());
        record.setComBhGrantOperRecordId(idWorker.nextId());
        record.setOperGrantFuncType(DictConst.OPERGRANTFUNCTYPE_XJFFXX);
        record.setOperResult("新建“" + comBhGrantInfo.getTitle() + "”");
        comBhGrantOperRecordMapper.insertComBhGrantOperRecord(record);

        return Result.success(Result.SUCCESS_MSG, comBhGrantInfo);
    }


    /**
     * 根据id获取企业渤海薪资代发
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-08-13
     */
    @Override
    public Result getComBhGrantInfoById(ComBhGrantInfoDTO dto) throws V5BusinessException {
        if (dto.getComBhGrantId() == null) {
            throw new V5BusinessException(Result.ERROR, "企业渤海发放ID不能为空");
        }
        ComBhGrantInfo comBhGrantInfo = comBhGrantInfoMapper.getComBhGrantInfoById(dto.getComBhGrantId());
        return Result.success(Result.SUCCESS_QUERY_MSG, comBhGrantInfo);
    }

    /**
     * 删除企业渤海薪资代发单
     *
     * @param dto
     * @author llc
     * @date 2020-08-13
     */
    @Override
    public Result deleteComBhGrantInfo(ComBhGrantInfoDTO dto) throws V5BusinessException {
        if (dto.getComBhGrantId() == null) {
            throw new V5BusinessException(Result.ERROR, "代发单ID不能为空");
        }
        ComBhGrantInfo comBhGrantInfo = comBhGrantInfoMapper.getComBhGrantInfoById(dto.getComBhGrantId());
        if (!CommonConst.N.equals(comBhGrantInfo.getCommitStatus())) {
            throw new V5BusinessException(Result.ERROR, "只允许删除待提交的记录");
        }
        /*** 删除薪资代发单 ***/
        comBhGrantInfoMapper.deleteById(dto.getComBhGrantId());

        /*** 删除代发明细 ***/
        comBhGrantDetailMapper.deleteByComBhGrantId(dto.getComBhGrantId());

        /*** 删除操作记录 ***/
        comBhGrantOperRecordMapper.deleteGrantOperRecordByComBhGrantId(dto.getComBhGrantId());

        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * 获取企业渤海薪资发放明细
     *
     * @param dto
     * @author llc
     * @date 2020-08-14
     */
    @Override
    public Result pageComBhGrantDetailList(ComBhGrantInfoDTO dto) throws V5BusinessException {
        if (dto.getPageNumber() == null) {
            throw new V5BusinessException(Result.ERROR, "第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            throw new V5BusinessException(Result.ERROR, "每页多少行不能为空");
        }
        if (dto.getComBhGrantId() == null) {
            throw new V5BusinessException(Result.ERROR, "代发单ID不能为空");
        }
        PageHelper.startPage(dto.getPageNumber(), dto.getPageSize());
        List<Map<String, Object>> comGrantInfoList = comBhGrantDetailMapper.getComBhGrantDetailList(dto);
        PageInfoUtils pageInfo = new PageInfoUtils(comGrantInfoList);
        return Result.success(Result.SUCCESS_QUERY_MSG, pageInfo);
    }


    /***
     * 导出薪资代发模板
     * @param dto
     * @author llc
     * @date 2020-03-19
     * @return
     * @throws V5BusinessException
     */
    @Override
    public Result exportOfflineGrantTemp(ComBhGrantInfoDTO dto, HttpServletResponse response) throws V5BusinessException {
        OutputStream os = null;
        HSSFWorkbook wb = new HSSFWorkbook();
        try {
            String fileName = "_" + "薪资代发模板" + DateUtils.getDate("yyyyMMddHHmmss") + ".xls";
            response.setContentType("application/vnd.ms-excel");
            response.setHeader("Content-Disposition",
                    "attachment;fileName=" + new String(fileName.getBytes("gb2312"), "ISO8859-1"));
            os = response.getOutputStream();
            HSSFSheet sheet1 = wb.createSheet("薪资代发模板");

            /**** excel标题栏 ****/
            String headerTitle = "薪资代发模板填写须知：\n"
                    + " <1>表单列目中带*号为必填字段；\n"
                    + " <2>系统会根据导入人员的身份证号匹配保安员证联名卡账号作为收款账号；若人员没有开通保安员证联名卡，则无法在线发薪；\n"
                    + " <3>一个代发单里的人员（收款账户）只能一条，若需要给某个人员（收款账户）发放多笔资金，请另建代发单发放；\n"
                    + " <4>当需要更新已导入发放数据时，请先在系统页面删除相应记录后，再重新导入；";

            /*** 填充文本样式 ***/
            CellStyle stringStyle = ExcelUtil.getStringStyle(wb);

            /*** 填充必填样式 ***/
            CellStyle style1 = ExcelUtil.getNecessaryTitleStyle(wb);

            /**** 生成sheet1的内容 ***/
            HSSFRow rowFirst = sheet1.createRow(0); // 第一个sheet的第一行为标题
            HSSFFont titleFont = wb.createFont();
            HSSFCellStyle titleStyle = wb.createCellStyle();
            titleFont.setFontHeightInPoints((short) 11);
            titleFont.setColor(HSSFColor.BLUE.index);
            titleFont.setFontName("黑体");
            titleStyle.setFont(titleFont);
            titleStyle.setWrapText(true);
            rowFirst.setHeight((short) (3000));
            sheet1.addMergedRegion(new CellRangeAddress(0, (short) 0, 0, 2));
            HSSFCell titleCell = rowFirst.createCell(0);
            titleCell.setCellValue(headerTitle);
            titleCell.setCellStyle(titleStyle);

            /*** 填充第一行数据 ***/
            HSSFRow rowSecond = sheet1.createRow(1);
            String[] firstArr = {"*姓名", "*身份证", "*发放金额"};
            int column;
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowSecond.createCell(i); // 获取第二行的每个单元格
                if (i == 0 || i == 2) {
                    column = 4000;
                } else {
                    column = 6000;
                }
                sheet1.setColumnWidth(i, column); // 设置每列的列宽
                sheet1.setDefaultColumnStyle(i, stringStyle); // 设置单元格格式 --文本格式
            }

            /*** 填充第二行数据 ***/
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowSecond.createCell(i); // 获取第二行的每个单元格
                cell.setCellValue(firstArr[i]);
                cell.setCellStyle(style1);
            }
            wb.write(os);
            os.close();
            wb.close();
        } catch (Exception e) {
            throw new V5BusinessException(Result.ERROR, e.getMessage());
        } finally {
            try {
                os.close();
                wb.close();
            } catch (IOException e1) {
            }
        }
        return Result.success("导出薪资代发模板数据文件成功");
    }

    /**
     * 导入薪资代发模板
     *
     * @param xfile
     * @param dto
     * @author llc
     * @date 2020-03-19
     */
    @Override
    public Result importOfflineGrantTemp(MultipartFile xfile, ComBhGrantInfoDTO dto)
            throws V5BusinessException {
        if (dto.getComBhGrantId() == null) {
            throw new V5BusinessException(Result.ERROR, "企业发放ID不能为空");
        }
        if (StringUtil.isEmpty(dto.getGrantYears())) {
            throw new V5BusinessException(Result.ERROR, "发放年月不能为空");
        }
        /*** 校验文件格式 ***/
        String fileType = xfile.getOriginalFilename().substring(xfile.getOriginalFilename().lastIndexOf(".") + 1)
                .toLowerCase();// 文件类型
        if (xfile != null && xfile.getSize() > 0) {
            if (!fileType.equals(ExcelEnum.EXCEL_TYPE_XLS.getValue())
                    && !fileType.equals(ExcelEnum.EXCEL_TYPE_XLSX.getValue())) {
                return Result.fail("上传文件格式不正确，不是Excel文件！");
            }
        }
        InputStream inputStream = null;
        Integer importCount = 0;
        List<Map<String, String>> list = new ArrayList<Map<String, String>>(); // 数据库校验的list
        List<Map<String, String>> faildList = new ArrayList<Map<String, String>>(); // 定义失败的list
        List<Map<String, String>> returnList = new ArrayList<Map<String, String>>(); // 定义返回的list
        try {
            inputStream = xfile.getInputStream();
            Workbook workBook = WorkbookFactory.create(inputStream);
            Sheet sheet = workBook.getSheetAt(0); // 读取第一个sheet

            boolean flag = true;
            int i = 2; // 设置初始行第3行
            while (flag) {
                Row row = sheet.getRow(i);
                if (row != null) {
                    Map<String, String> data = new HashMap<String, String>();
                    String errorMsg = ""; // 校验错误信息

                    /*** 姓名 ***/
                    String name = ExcelUtil.getCellValue(row.getCell(0));
                    name = name.trim().replaceAll(" ", "").replaceAll(" ", "").replaceAll(" ", "");
                    data.put("name", name);

                    /*** 证件号码 ***/
                    String idCardNo = ExcelUtil.getCellValue(row.getCell(1));
                    idCardNo = idCardNo.trim().toUpperCase().replaceAll(" ", "").replaceAll(" ", "").replaceAll(" ",
                            "");
                    /*** 因为客户之前的excel身份证号可能存在科学计数法 ***/
                    if (idCardNo.contains("E") && idCardNo.contains(".")) {
                        BigDecimal bd = new BigDecimal(idCardNo);
                        idCardNo = bd.toPlainString();
                    }
                    data.put("idCardNo", idCardNo);

                    /*** 人员类型 ***/
                    String amount = ExcelUtil.getCellValue(row.getCell(2));
                    amount = amount.trim().replaceAll(" ", "").replaceAll(" ", "").replaceAll(" ", "");
                    data.put("amount", amount);

                    data.put("bankCardNo", null); // 为了兼容银行卡号导入发放

                    /***** 校验excel字段 ****/

                    /*** 姓名 ***/
                    if (StringUtil.isEmpty(name)) {
                        errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "姓名不能为空" : " || " + "姓名不能为空");
                    }

                    /*** 证件号码 ***/
                    if (StringUtil.isEmpty(idCardNo)) {
                        errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "证件号码不能为空" : " || " + "证件号码不能为空");
                    }

                    /*** 金额 ***/
                    if (StringUtil.isEmpty(amount)) {
                        errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "发放金额不能为空" : " || " + "发放金额不能为空");
                    } else {
                        /*** 校验金额是否合理 ****/
                        if (!StringUtils.isBigDecimal(amount)) {
                            errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "发放金额有误" : " || " + "发放金额有误");
                        } else {
                            if (new BigDecimal(amount).compareTo(BigDecimal.ZERO) < 1) {
                                errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "发放金额不能为0或负数" : " || " + "发放金额不能为0或负数");
                            }
                        }
                    }
                    if (StringUtil.isNotEmpty(errorMsg)) {
                        data.put("errorMsg", errorMsg);
                        faildList.add(data);
                        returnList.add(data);
                    } else {
                        list.add(data);
                    }
                    importCount++;
                } else {
                    flag = false;
                }
                i++;
            }

            /** 如果客户没上传内容 **/
            if (list.size() == 0 && faildList.size() == 0) {
                return Result.fail("导入模板没有信息");
            }

            for (Map<String, String> map : list) {
                map.put("comBhGrantDetailId", String.valueOf(idWorker.nextId()));
            }

            /**** 处理业务逻辑 ***/
            if (list != null && list.size() > 0) {
                List<Map<String, String>> tempInputPerList = saveOfflineGrantInfo(list, importCount, dto);
                List<Map<String, String>> checkFaildList = new ArrayList<>();
                for (Map<String, String> map : tempInputPerList) {
                    if (map.get("isSuccess").equals(CommonConst.N)) { // 失败人员
                        checkFaildList.add(map);
                    }
                    returnList.add(map);
                }

                /**** 如果校验的导入人员信息有误 ****/
                if (checkFaildList != null && checkFaildList.size() > 0) {
                    faildList.addAll(checkFaildList);
                }

            }

        } catch (Exception e) {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e1) {

                }
            }
            e.printStackTrace();
            throw new V5BusinessException(Result.ERROR, "表格数据有误");
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e1) {

                }
            }
        }

        /*** 如果有失败人员信息,返回所有导入信息 ***/
        String resultMsg = MessageFormat.format("本次导入条数【{0}】,其中校验失败条数:【{1}】", importCount,
                faildList.size());
        if (faildList.size() > 0) {
            String fileUrl = "";
            try {
                fileUrl = buildFaildExcel(returnList);
            } catch (Exception e) {
                e.printStackTrace();
                throw new V5BusinessException(Result.ERROR, "生成错误失败人员文件错误");
            }
            return Result.fail(resultMsg, fileUrl);
        } else {
            return Result.success(resultMsg);
        }
    }

    /**
     * 处理导入线下工资方法
     *
     * @param grantList
     * @param sumCount  导入的总记录
     * @param dto
     * @author llc
     * @date 2020-08-17
     */
    public List<Map<String, String>> saveOfflineGrantInfo(List<Map<String, String>> grantList, Integer sumCount, ComBhGrantInfoDTO dto)
            throws V5BusinessException {
        Long importBatchId = idWorker.nextId();
        /***** 批量插入线下代发导入临时表 ****/

        /**** 数据量太大导致java内存溢出 ,这里分批插入 ****/
        int insertLength = grantList.size();
        int i = 0;
        while (insertLength > 2000) {
            comBhGrantInfoMapper.insertTempOfflineBhGrant(grantList.subList(i, i + 2000), importBatchId);
            i = i + 2000;
            insertLength = insertLength - 2000;
        }
        if (insertLength > 0) {
            comBhGrantInfoMapper.insertTempOfflineBhGrant(grantList.subList(i, i + insertLength), importBatchId);
        }

        /***** 导入逻辑 ***/
        comBhGrantInfoMapper.checkIdCardNoRepeat(importBatchId);

        comBhGrantInfoMapper.checkExistsIdCard(importBatchId);

        comBhGrantInfoMapper.updateCheckPassPerId(importBatchId);

//        comBhGrantInfoMapper.checkIdCardNoInCom(importBatchId, dto.getComId());

        comBhGrantInfoMapper.checkPerExistsJointlyCard(importBatchId);

        comBhGrantInfoMapper.checkExistsRecord(importBatchId, dto.getComBhGrantId());

        /***** 查询本次导入的线下代发记录信息 ****/
        List<Map<String, String>> tempOfflineGrantList = comBhGrantInfoMapper.getTempOfflineBhGrantList(importBatchId);

        /*** 校验通过的代发记录数 ***/
        Integer checkPassCount = 0;
        for (Map<String, String> map : tempOfflineGrantList) {
            if (map.get("isSuccess").equals(CommonConst.Y)) {
                checkPassCount++;
            }
        }

        /*** 代发列表全部校验通过才需要生成代发记录 ***/
        if (sumCount.equals(checkPassCount)) {
            comBhGrantInfoMapper.updateBhGrantInfo(importBatchId, dto.getUpdateBy(), dto.getComBhGrantId(), dto.getGrantYears(), DictConst.GRANTDATASOURCE_DRXXSJ);
            comBhGrantDetailMapper.insertBhGrantDetail(importBatchId, dto.getCreateBy(), dto.getComBhGrantId(), dto.getGrantYears());

            /*** 插入操作记录 ***/
            ComBhGrantOperRecord record = new ComBhGrantOperRecord();
            record.setCreateBy(dto.getCreateBy());
            record.setCreateTime(DateUtil.date());
            record.setComId(dto.getComId());
            record.setComBhGrantId(dto.getComBhGrantId());
            record.setComBhGrantOperRecordId(idWorker.nextId());
            record.setOperGrantFuncType(DictConst.OPERGRANTFUNCTYPE_DRXXSJ);
            record.setOperResult("导入发放数据" + sumCount + "条");
            comBhGrantOperRecordMapper.insertComBhGrantOperRecord(record);
        }

        /***** 删除本次导入的线下代发信息 ****/
        comBhGrantInfoMapper.deleteTempOfflineBhGrantImport(importBatchId);

        return tempOfflineGrantList;
    }


    /**
     * 生成错误列表的excel
     *
     * @param failList
     * @author llc
     * @date 2020-03-19
     */
    private String buildFaildExcel(List<Map<String, String>> failList) throws V5BusinessException {
        FileOutputStream fileOut = null;
        try {
            Date now = new Date();
            String upload_path = commonService.getSystemConfig("SYS_PIC_PATH");
            String dirDate = DateUtil.today(); // 当前日期
            String dirPaht = upload_path + "temp/" + dirDate;
            File dir = new File(dirPaht);
            if (!dir.exists()) {
                dir.mkdirs();
            }

            String fileName = now.getTime() + RandomUtil.randomNumbers(6) + ".xls";
            String filePath = dirPaht + "/" + fileName;
            File file = new File(filePath);
            HSSFWorkbook wb = new HSSFWorkbook();
            HSSFSheet sheet1 = wb.createSheet("薪资代发模板");

            /**** excel标题栏 ****/
            String headerTitle = "薪资代发模板填写须知：\n"
                    + " <1>表单列目中带*号为必填字段；\n"
                    + " <2>系统会根据导入人员的身份证号匹配保安员证联名卡账号作为收款账号；若人员没有开通保安员证联名卡，则无法在线发薪；\n"
                    + " <3>一个代发单里的人员（收款账户）只能一条，若需要给某个人员（收款账户）发放多笔资金，请另建代发单发放；\n"
                    + " <4>当需要更新已导入发放数据时，请先在系统页面删除相应记录后，再重新导入；";

            /*** 填充文本样式 ***/
            CellStyle stringStyle = ExcelUtil.getStringStyle(wb);

            /*** 填充必填样式 ***/
            CellStyle style1 = ExcelUtil.getNecessaryTitleStyle(wb);

            /**** 生成sheet1的内容 ***/
            HSSFRow rowFirst = sheet1.createRow(0); // 第一个sheet的第一行为标题
            HSSFFont titleFont = wb.createFont();
            HSSFCellStyle titleStyle = wb.createCellStyle();
            titleFont.setFontHeightInPoints((short) 11);
            titleFont.setColor(HSSFColor.BLUE.index);
            titleFont.setFontName("黑体");
            titleStyle.setFont(titleFont);
            titleStyle.setWrapText(true);
            rowFirst.setHeight((short) (3000));
            sheet1.addMergedRegion(new CellRangeAddress(0, (short) 0, 0, 2));
            HSSFCell titleCell = rowFirst.createCell(0);
            titleCell.setCellValue(headerTitle);
            titleCell.setCellStyle(titleStyle);

            /*** 填充第一行数据 ***/
            HSSFRow rowSecond = sheet1.createRow(1);
            String[] firstArr = {"*姓名", "*身份证", "*发放金额", "错误信息"};
            int column;
            for (int i = 0; i < firstArr.length; i++) {
                if (i == 0 || i == 2) {
                    column = 4000;
                } else if (i == 1) {
                    column = 6000;
                } else {
                    column = 8000;
                }
                sheet1.setColumnWidth(i, column); // 设置每列的列宽
                sheet1.setDefaultColumnStyle(i, stringStyle); // 设置单元格格式 --文本格式
            }

            /*** 填充第二行数据 ***/
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowSecond.createCell(i); // 获取第二行的每个单元格
                cell.setCellValue(firstArr[i]);
                cell.setCellStyle(style1);
            }
            HSSFRow row = null;
            int rowIndex = 2;
            for (Map<String, String> map : failList) {
                row = sheet1.createRow(rowIndex);
                row.createCell(0).setCellValue(map.get("name"));
                row.createCell(1).setCellValue(map.get("idCardNo"));
                row.createCell(2).setCellValue(map.get("amount"));
                row.createCell(3).setCellValue(map.get("errorMsg"));
                rowIndex++;
            }
            fileOut = new FileOutputStream(file);

            wb.write(fileOut);
            fileOut.close();
            wb.close();

            String platform = commonService.getSystemConfig("PLATFORM_SIGN");
            String aliUrl = BcxinFileUtils.aliOSSFileTempUpload(file, platform);
            file.delete();
            return aliUrl;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        } finally {
            if (fileOut != null) {
                try {
                    fileOut.close();
                } catch (IOException e1) {
                }
            }
        }
    }


    /***
     * 导出渤海薪资银行卡代发模板
     * @param dto
     * @author llc
     * @date 2020-12-04
     * @return
     * @throws V5BusinessException
     */
    @Override
    public Result exportOfflineBankGrantTemp(ComBhGrantInfoDTO dto, HttpServletResponse response) throws V5BusinessException {
        OutputStream os = null;
        HSSFWorkbook wb = new HSSFWorkbook();
        try {
            String fileName = "_" + "薪资代发(银行卡)模板" + DateUtils.getDate("yyyyMMddHHmmss") + ".xls";
            response.setContentType("application/vnd.ms-excel");
            response.setHeader("Content-Disposition",
                    "attachment;fileName=" + new String(fileName.getBytes("gb2312"), "ISO8859-1"));
            os = response.getOutputStream();
            HSSFSheet sheet1 = wb.createSheet("薪资代发(银行卡)模板");

            /**** excel标题栏 ****/
            String headerTitle = "薪资代发(银行卡)模板填写须知：\n"
                    + " <1>表单列目中带*号为必填字段；\n"
                    + " <2>系统根据导入人员的收款账号进行发放；\n"
                    + " <3>一个代发单里的人员（收款账户）只能一条，若需要给某个人员（收款账户）发放多笔资金，请另建代发单发放；\n"
                    + " <4>当需要更新已导入发放数据时，请先在系统页面删除相应记录后，再重新导入；";

            /*** 填充文本样式 ***/
            CellStyle stringStyle = ExcelUtil.getStringStyle(wb);

            /*** 填充必填样式 ***/
            CellStyle style1 = ExcelUtil.getNecessaryTitleStyle(wb);

            /**** 生成sheet1的内容 ***/
            HSSFRow rowFirst = sheet1.createRow(0); // 第一个sheet的第一行为标题
            HSSFFont titleFont = wb.createFont();
            HSSFCellStyle titleStyle = wb.createCellStyle();
            titleFont.setFontHeightInPoints((short) 11);
            titleFont.setColor(HSSFColor.BLUE.index);
            titleFont.setFontName("黑体");
            titleStyle.setFont(titleFont);
            titleStyle.setWrapText(true);
            rowFirst.setHeight((short) (3000));
            sheet1.addMergedRegion(new CellRangeAddress(0, (short) 0, 0, 3));
            HSSFCell titleCell = rowFirst.createCell(0);
            titleCell.setCellValue(headerTitle);
            titleCell.setCellStyle(titleStyle);

            /*** 填充第一行数据 ***/
            HSSFRow rowSecond = sheet1.createRow(1);
            String[] firstArr = {"*姓名", "*身份证","*银行卡号","*发放金额"};
            int column;
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowSecond.createCell(i); // 获取第二行的每个单元格
                if (i == 0 || i == 3) {
                    column = 4000;
                } else {
                    column = 6000;
                }
                sheet1.setColumnWidth(i, column); // 设置每列的列宽
                sheet1.setDefaultColumnStyle(i, stringStyle); // 设置单元格格式 --文本格式
            }

            /*** 填充第二行数据 ***/
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowSecond.createCell(i); // 获取第二行的每个单元格
                cell.setCellValue(firstArr[i]);
                cell.setCellStyle(style1);
            }
            wb.write(os);
            os.close();
            wb.close();
        } catch (Exception e) {
            throw new V5BusinessException(Result.ERROR, e.getMessage());
        } finally {
            try {
                os.close();
                wb.close();
            } catch (IOException e1) {
            }
        }
        return Result.success("导出薪资代发(银行卡)模板文件成功");
    }


    /**
     * 导入渤海薪资银行卡代发模板
     *
     * @param xfile
     * @param dto
     * @author llc
     * @date 2020-12-04
     */
    @Override
    public Result importOfflineBankGrantTemp(MultipartFile xfile, ComBhGrantInfoDTO dto)
            throws V5BusinessException {
        if (dto.getComBhGrantId() == null) {
            throw new V5BusinessException(Result.ERROR, "企业发放ID不能为空");
        }
        if (StringUtil.isEmpty(dto.getGrantYears())) {
            throw new V5BusinessException(Result.ERROR, "发放年月不能为空");
        }
        /*** 校验文件格式 ***/
        String fileType = xfile.getOriginalFilename().substring(xfile.getOriginalFilename().lastIndexOf(".") + 1)
                .toLowerCase();// 文件类型
        if (xfile != null && xfile.getSize() > 0) {
            if (!fileType.equals(ExcelEnum.EXCEL_TYPE_XLS.getValue())
                    && !fileType.equals(ExcelEnum.EXCEL_TYPE_XLSX.getValue())) {
                return Result.fail("上传文件格式不正确，不是Excel文件！");
            }
        }
        InputStream inputStream = null;
        Integer importCount = 0;
        List<Map<String, String>> list = new ArrayList<Map<String, String>>(); // 数据库校验的list
        List<Map<String, String>> faildList = new ArrayList<Map<String, String>>(); // 定义失败的list
        List<Map<String, String>> returnList = new ArrayList<Map<String, String>>(); // 定义返回的list
        try {
            inputStream = xfile.getInputStream();
            Workbook workBook = WorkbookFactory.create(inputStream);
            Sheet sheet = workBook.getSheetAt(0); // 读取第一个sheet

            boolean flag = true;
            int i = 2; // 设置初始行第3行
            while (flag) {
                Row row = sheet.getRow(i);
                if (row != null) {
                    Map<String, String> data = new HashMap<String, String>();
                    String errorMsg = ""; // 校验错误信息

                    /*** 姓名 ***/
                    String name = ExcelUtil.getCellValue(row.getCell(0));
                    name = name.trim().replaceAll(" ", "").replaceAll(" ", "").replaceAll(" ", "");
                    data.put("name", name);

                    /*** 证件号码 ***/
                    String idCardNo = ExcelUtil.getCellValue(row.getCell(1));
                    idCardNo = idCardNo.trim().toUpperCase().replaceAll(" ", "").replaceAll(" ", "").replaceAll(" ",
                            "");
                    /*** 因为客户之前的excel身份证号可能存在科学计数法 ***/
                    if (idCardNo.contains("E") && idCardNo.contains(".")) {
                        BigDecimal bd = new BigDecimal(idCardNo);
                        idCardNo = bd.toPlainString();
                    }
                    data.put("idCardNo", idCardNo);


                    /*** 银行卡号 ***/
                    String bankCardNo = ExcelUtil.getCellValue(row.getCell(2));
                    bankCardNo = bankCardNo.trim().toUpperCase().replaceAll(" ", "").replaceAll(" ", "").replaceAll(" ",
                            "");
                    /*** 因为客户之前的excel银行卡号可能存在科学计数法 ***/
                    if (bankCardNo.contains("E") && bankCardNo.contains(".")) {
                        BigDecimal bankCardNoBd = new BigDecimal(bankCardNo);
                        bankCardNo = bankCardNoBd.toPlainString();
                    }
                    data.put("bankCardNo", bankCardNo);

                    /*** 金额 ***/
                    String amount = ExcelUtil.getCellValue(row.getCell(3));
                    amount = amount.trim().replaceAll(" ", "").replaceAll(" ", "").replaceAll(" ", "");
                    data.put("amount", amount);

                    /***** 校验excel字段 ****/

                    /*** 姓名 ***/
                    if (StringUtil.isEmpty(name)) {
                        errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "姓名不能为空" : " || " + "姓名不能为空");
                    }

                    /*** 证件号码 ***/
                    if (StringUtil.isEmpty(idCardNo)) {
                        errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "证件号码不能为空" : " || " + "证件号码不能为空");
                    }

                    /*** 银行卡号 ***/
                    if (StringUtil.isEmpty(bankCardNo)) {
                        errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "银行卡号不能为空" : " || " + "银行卡号不能为空");
                    }else{
                       /*** 校验银行卡号是否合理 ****/
                       if(bankCardNo.length() <= 12 || bankCardNo.length() >= 20) {
                           errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "请确认银行卡号是否有效" : " || " + "请确认银行卡号是否有效");
                       }
                    }

                    /*** 金额 ***/
                    if (StringUtil.isEmpty(amount)) {
                        errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "发放金额不能为空" : " || " + "发放金额不能为空");
                    } else {
                        /*** 校验金额是否合理 ****/
                        if (!StringUtils.isBigDecimal(amount)) {
                            errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "发放金额有误" : " || " + "发放金额有误");
                        } else {
                            if (new BigDecimal(amount).compareTo(BigDecimal.ZERO) < 1) {
                                errorMsg = errorMsg + (StringUtil.isEmpty(errorMsg) ? "发放金额不能为0或负数" : " || " + "发放金额不能为0或负数");
                            }
                        }
                    }
                    if (StringUtil.isNotEmpty(errorMsg)) {
                        data.put("errorMsg", errorMsg);
                        faildList.add(data);
                        returnList.add(data);
                    } else {
                        list.add(data);
                    }
                    importCount++;
                } else {
                    flag = false;
                }
                i++;
            }

            /** 如果客户没上传内容 **/
            if (list.size() == 0 && faildList.size() == 0) {
                return Result.fail("导入模板没有信息");
            }

            for (Map<String, String> map : list) {
                map.put("comBhGrantDetailId", String.valueOf(idWorker.nextId()));
            }

            /**** 处理业务逻辑 ***/
            if (list != null && list.size() > 0) {
                List<Map<String, String>> tempInputPerList = saveOfflineBankGrantInfo(list, importCount, dto);
                List<Map<String, String>> checkFaildList = new ArrayList<>();
                for (Map<String, String> map : tempInputPerList) {
                    if (map.get("isSuccess").equals(CommonConst.N)) { // 失败人员
                        checkFaildList.add(map);
                    }
                    returnList.add(map);
                }

                /**** 如果校验的导入人员信息有误 ****/
                if (checkFaildList != null && checkFaildList.size() > 0) {
                    faildList.addAll(checkFaildList);
                }

            }

        } catch (Exception e) {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e1) {

                }
            }
            e.printStackTrace();
            throw new V5BusinessException(Result.ERROR, "表格数据有误");
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e1) {

                }
            }
        }

        /*** 如果有失败人员信息,返回所有导入信息 ***/
        String resultMsg = MessageFormat.format("本次导入条数【{0}】,其中校验失败条数:【{1}】", importCount,
                faildList.size());
        if (faildList.size() > 0) {
            String fileUrl = "";
            try {
                fileUrl = buildBankFaildExcel(returnList);
            } catch (Exception e) {
                e.printStackTrace();
                throw new V5BusinessException(Result.ERROR, "生成错误导入文件错误");
            }
            return Result.fail(resultMsg, fileUrl);
        } else {
            return Result.success(resultMsg);
        }
    }

    /**
     * 处理导入线下银行卡工资方法
     *
     * @param grantList
     * @param sumCount  导入的总记录
     * @param dto
     * @author llc
     * @date 2020-08-17
     */
    public List<Map<String, String>> saveOfflineBankGrantInfo(List<Map<String, String>> grantList, Integer sumCount, ComBhGrantInfoDTO dto)
            throws V5BusinessException {
        Long importBatchId = idWorker.nextId();
        /***** 批量插入线下代发导入临时表 ****/

        /**** 数据量太大导致java内存溢出 ,这里分批插入 ****/
        int insertLength = grantList.size();
        int i = 0;
        while (insertLength > 2000) {
            comBhGrantInfoMapper.insertTempOfflineBhGrant(grantList.subList(i, i + 2000), importBatchId);
            i = i + 2000;
            insertLength = insertLength - 2000;
        }
        if (insertLength > 0) {
            comBhGrantInfoMapper.insertTempOfflineBhGrant(grantList.subList(i, i + insertLength), importBatchId);
        }

        /***** 导入逻辑 ***/
        comBhGrantInfoMapper.checkIdCardNoRepeat(importBatchId);

        comBhGrantInfoMapper.checkExistsIdCard(importBatchId);

        comBhGrantInfoMapper.checkBankCardNoRepeat(importBatchId);

        comBhGrantInfoMapper.updateCheckPassPerId(importBatchId);

//        comBhGrantInfoMapper.checkIdCardNoInCom(importBatchId, dto.getComId());


        comBhGrantInfoMapper.checkExistsRecord(importBatchId, dto.getComBhGrantId());

        /***** 查询本次导入的线下代发记录信息 ****/
        List<Map<String, String>> tempOfflineGrantList = comBhGrantInfoMapper.getTempOfflineBhGrantList(importBatchId);

        /*** 校验通过的代发记录数 ***/
        Integer checkPassCount = 0;
        for (Map<String, String> map : tempOfflineGrantList) {
            if (map.get("isSuccess").equals(CommonConst.Y)) {
                checkPassCount++;
            }
        }

        /*** 代发列表全部校验通过才需要生成代发记录 ***/
        if (sumCount.equals(checkPassCount)) {
            comBhGrantInfoMapper.updateBhGrantInfo(importBatchId, dto.getUpdateBy(), dto.getComBhGrantId(), dto.getGrantYears(), DictConst.GRANTDATASOURCE_DRXXSJ);
            comBhGrantDetailMapper.insertBhBankGrantDetail(importBatchId, dto.getCreateBy(), dto.getComBhGrantId(), dto.getGrantYears());

            /*** 插入操作记录 ***/
            ComBhGrantOperRecord record = new ComBhGrantOperRecord();
            record.setCreateBy(dto.getCreateBy());
            record.setCreateTime(DateUtil.date());
            record.setComId(dto.getComId());
            record.setComBhGrantId(dto.getComBhGrantId());
            record.setComBhGrantOperRecordId(idWorker.nextId());
            record.setOperGrantFuncType(DictConst.OPERGRANTFUNCTYPE_DRXXSJ);
            record.setOperResult("导入发放数据" + sumCount + "条");
            comBhGrantOperRecordMapper.insertComBhGrantOperRecord(record);
        }

        /***** 删除本次导入的线下代发信息 ****/
        comBhGrantInfoMapper.deleteTempOfflineBhGrantImport(importBatchId);

        return tempOfflineGrantList;
    }

    /**
     * 生成银行卡导入错误列表的excel
     *
     * @param failList
     * @author llc
     * @date 2020-12-04
     */
    private String buildBankFaildExcel(List<Map<String, String>> failList) throws V5BusinessException {
        FileOutputStream fileOut = null;
        try {
            Date now = new Date();
            String upload_path = commonService.getSystemConfig("SYS_PIC_PATH");
            String dirDate = DateUtil.today(); // 当前日期
            String dirPaht = upload_path + "temp/" + dirDate;
            File dir = new File(dirPaht);
            if (!dir.exists()) {
                dir.mkdirs();
            }

            String fileName = now.getTime() + RandomUtil.randomNumbers(6) + ".xls";
            String filePath = dirPaht + "/" + fileName;
            File file = new File(filePath);
            HSSFWorkbook wb = new HSSFWorkbook();
            HSSFSheet sheet1 = wb.createSheet("薪资代发(银行卡)模板");

            /**** excel标题栏 ****/
            String headerTitle = "薪资代发(银行卡)模板填写须知：\n"
                    + " <1>表单列目中带*号为必填字段；\n"
                    + " <2>系统根据导入人员的收款账号进行发放；\n"
                    + " <3>一个代发单里的人员（收款账户）只能一条，若需要给某个人员（收款账户）发放多笔资金，请另建代发单发放；\n"
                    + " <4>当需要更新已导入发放数据时，请先在系统页面删除相应记录后，再重新导入；";

            /*** 填充文本样式 ***/
            CellStyle stringStyle = ExcelUtil.getStringStyle(wb);

            /*** 填充必填样式 ***/
            CellStyle style1 = ExcelUtil.getNecessaryTitleStyle(wb);

            /**** 生成sheet1的内容 ***/
            HSSFRow rowFirst = sheet1.createRow(0); // 第一个sheet的第一行为标题
            HSSFFont titleFont = wb.createFont();
            HSSFCellStyle titleStyle = wb.createCellStyle();
            titleFont.setFontHeightInPoints((short) 11);
            titleFont.setColor(HSSFColor.BLUE.index);
            titleFont.setFontName("黑体");
            titleStyle.setFont(titleFont);
            titleStyle.setWrapText(true);
            rowFirst.setHeight((short) (3000));
            sheet1.addMergedRegion(new CellRangeAddress(0, (short) 0, 0, 3));
            HSSFCell titleCell = rowFirst.createCell(0);
            titleCell.setCellValue(headerTitle);
            titleCell.setCellStyle(titleStyle);

            /*** 填充第一行数据 ***/
            HSSFRow rowSecond = sheet1.createRow(1);
            String[] firstArr = {"*姓名", "*身份证", "*银行卡号", "*发放金额", "错误信息"};
            int column;
            for (int i = 0; i < firstArr.length; i++) {
                if (i == 0 || i == 3) {
                    column = 4000;
                } else if (i == 1 || i == 2) {
                    column = 6000;
                } else {
                    column = 8000;
                }
                sheet1.setColumnWidth(i, column); // 设置每列的列宽
                sheet1.setDefaultColumnStyle(i, stringStyle); // 设置单元格格式 --文本格式
            }

            /*** 填充第二行数据 ***/
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowSecond.createCell(i); // 获取第二行的每个单元格
                cell.setCellValue(firstArr[i]);
                cell.setCellStyle(style1);
            }
            HSSFRow row = null;
            int rowIndex = 2;
            for (Map<String, String> map : failList) {
                row = sheet1.createRow(rowIndex);
                row.createCell(0).setCellValue(map.get("name"));
                row.createCell(1).setCellValue(map.get("idCardNo"));
                row.createCell(2).setCellValue(map.get("bankCardNo"));
                row.createCell(3).setCellValue(map.get("amount"));
                row.createCell(4).setCellValue(map.get("errorMsg"));
                rowIndex++;
            }
            fileOut = new FileOutputStream(file);

            wb.write(fileOut);
            fileOut.close();
            wb.close();

            String platform = commonService.getSystemConfig("PLATFORM_SIGN");
            String aliUrl = BcxinFileUtils.aliOSSFileTempUpload(file, platform);
            file.delete();
            return aliUrl;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        } finally {
            if (fileOut != null) {
                try {
                    fileOut.close();
                } catch (IOException e1) {
                }
            }
        }
    }


    /**
     * 删除企业渤海薪资代明细
     *
     * @param comBhGrantDetail
     * @author llc
     * @date 2020-08-17
     */
    @Override
    public Result deleteComBhGrantDetailById(ComBhGrantDetail comBhGrantDetail) throws V5BusinessException {
        if (comBhGrantDetail.getComBhGrantDetailId() == null) {
            throw new V5BusinessException(Result.ERROR, "请选中需要删除的代发记录");
        }
        comBhGrantDetail = comBhGrantDetailMapper.getComBhGrantDetailById(comBhGrantDetail.getComBhGrantDetailId());
        if (comBhGrantDetail == null) {
            throw new V5BusinessException(Result.ERROR, "代发记录ID有误");
        }
        /*** 更新渤海薪资代发单 ***/
        ComBhGrantInfo comBhGrantInfo = comBhGrantInfoMapper.getComBhGrantInfoById(comBhGrantDetail.getComBhGrantId());
        comBhGrantInfo.setGrantCount(comBhGrantInfo.getGrantCount() - 1);
        comBhGrantInfo.setSumAmount(comBhGrantInfo.getSumAmount().subtract(comBhGrantDetail.getAmount()));
        comBhGrantInfo.setUpdateBy(comBhGrantDetail.getUpdateBy());
        comBhGrantInfo.setUpdateTime(comBhGrantDetail.getUpdateTime());
        comBhGrantInfoMapper.updateComBhGrantInfo(comBhGrantInfo);

        /*** 删除代发明细 ***/
        comBhGrantDetailMapper.deleteById(comBhGrantDetail.getComBhGrantDetailId());

        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * 提交发放申请
     *
     * @param dto
     * @author llc
     * @date 2020-08-17
     */
    @Override
    public Result commitGrantApply(ComBhGrantInfoDTO dto) throws V5BusinessException {
        if (dto.getComBhGrantId() == null) {
            throw new V5BusinessException(Result.ERROR, "发放ID不能为空");
        }

        if (StringUtils.isEmpty(dto.getActiveCode())) {
            throw new V5BusinessException(Result.ERROR, "验证码不能为空");
        }
        if (StringUtils.isEmpty(dto.getRemark())) {
            throw new V5BusinessException(Result.ERROR, "备注不能为空");
        }
        ComBhGrantInfo comBhGrantInfo = comBhGrantInfoMapper.getComBhGrantInfoById(dto.getComBhGrantId());
        if (comBhGrantInfo == null) {
            throw new V5BusinessException(Result.ERROR, "发放ID有误");
        }

        /*** 组装发放文件txt ***/
        String txtTitle = comBhGrantInfo.getSumAmount() + CommonConst.COMMA + comBhGrantInfo.getGrantCount();

        List<Map<String, Object>> grantDetailList = comBhGrantDetailMapper.getComBhGrantDetailList(dto);
        if (grantDetailList == null) {
            throw new V5BusinessException(Result.ERROR, "没有发放明细");
        }

        /*** 生成txt文件并存入阿里云 ***/
        Map<String, String> map = buildGrantFile(txtTitle, grantDetailList, dto);

        /*** 如果不是生产环境，则直接返回文件手动提供给渤海人员进行手动上传 ***/
        String platform = commonService.getSystemConfig("PLATFORM_SIGN");
        if (!ObjectUtils.equals(platform, "SAAS-QY")) {
            return Result.success(Result.SUCCESS_MSG, map.get("fileUrl"));
        }

        /*** 调用支付平台上传发放文件到渤海sftp ***/
        Map<Object, Object> params = Maps.newHashMap();
        params.put("dfmxPath", map.get("fileUrl"));//代发明细文件地址
        String token = JwtUtil.createJWT(params);//加密生成token
        Map<String, Object> requestMap = Maps.newHashMap();
        requestMap.put("token", token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.UPLOAD_GRANT_FILE_TO_SFTO, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            String resultFile = result.getData().toString();

            /*** 调用支付平台上传发放文件 ***/
            Map<Object, Object> p = Maps.newHashMap();
            p.put("MerUserId", dto.getComId().toString());//平台会员号
            p.put("VerifyNo", dto.getActiveCode());//手机劢态验证码
            p.put("VirlAcctType", "01");//平台会员账户类型，用于校验交易密码01:基本户；02:保证金户； 04:只收丌付户；05:融资与用户
            p.put("FileName", resultFile);
            p.put("NoticeType", BHGRANTFILETYPE);//4：代发
            token = JwtUtil.createJWT(p);// 加密生成token
            requestMap.put("token", token);
            platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
            resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.UPLOAD_GRANT_FILE, requestMap);
            result = JSON.parseObject(resultStr, Result.class);
            /*** 如果返回成功 ***/
            if (Result.SUCCESS.equals(result.getRetType())) {
                /*** 更新渤海薪资代发单 ***/
                comBhGrantInfo.setTotalGrantCount(comBhGrantInfo.getGrantCount());
                comBhGrantInfo.setTotalSumAmount(comBhGrantInfo.getSumAmount());
                comBhGrantInfo.setUpdateBy(dto.getUpdateBy());
                comBhGrantInfo.setUpdateTime(dto.getUpdateTime());
                comBhGrantInfo.setCommitStatus(CommonConst.Y);
                comBhGrantInfo.setFileName(map.get("fileName"));
                comBhGrantInfoMapper.updateComBhGrantInfo(comBhGrantInfo);

                /*** 插入操作记录 ***/
                ComBhGrantOperRecord record = new ComBhGrantOperRecord();
                record.setCreateBy(dto.getCreateBy());
                record.setCreateTime(DateUtil.date());
                record.setComId(dto.getComId());
                record.setComBhGrantId(dto.getComBhGrantId());
                record.setComBhGrantOperRecordId(idWorker.nextId());
                record.setOperResult("提交发放申请");
                record.setOperGrantFuncType(DictConst.OPERGRANTFUNCTYPE_TJFFSQ);
                comBhGrantOperRecordMapper.insertComBhGrantOperRecord(record);
                return Result.success(Result.SUCCESS_MSG);
            } else {
                return Result.fail(result.getMsg());
            }
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
    }

    /**
     * 生成发放文件
     */

    private Map<String, String> buildGrantFile(String title, List<Map<String, Object>> grantDetailList, ComBhGrantInfoDTO dto) throws V5BusinessException {
        Map<String, String> resultMap = new HashMap<>();
        BufferedWriter out = null;
        OutputStreamWriter fw = null;
        try {
            String upload_path = commonService.getSystemConfig("SYS_PIC_PATH");
            String dirDate = DateUtil.today(); // 当前日期

            String dirPaht = upload_path + "temp/" + dirDate;
            File dir = new File(dirPaht);
            if (!dir.exists()) {
                dir.mkdirs();
            }

            /**
             * 文件名
             * YYYYMMDD：日期
             * CH：渠道代码
             * SSSSSS：文件批次号
             * NoticeType：文件类型
             * NoticeType_YYYYMMDD_CH_SSSSSS.rq
             * 1_20171212_C1_000001.rq
             */
            Date date = new Date();
            String day = DateUtils.formatDate(date, "yyyyMMdd");
            String fileName = BHGRANTFILETYPE + CommonConst.CONNECT_CHAR + day + CommonConst.CONNECT_CHAR + BHGRANTQDDM + CommonConst.CONNECT_CHAR + DateUtils.formatDate(date, "HHmmss") + ".rq";
            resultMap.put("fileName", fileName);
            String filePath = dirPaht + "/" + fileName;
            File file = new File(filePath);
            fw = new OutputStreamWriter(new FileOutputStream(file, false), "UTF-8");
            out = new BufferedWriter(fw);
            String line = "\r\n"; // 系统换行符
            out.write(title + line); // \r\n即为换行

            /*** 组装每行内容 ***/
            int i = 1;
            String today = DateUtils.formatDate(date);
            for (Map<String, Object> map : grantDetailList) {

                String info = String.valueOf(i) + CommonConst.COMMA + fileName + CommonConst.COMMA + map.get("bankCardNo") + CommonConst.COMMA + map.get("name") + CommonConst.COMMA + map.get("amount")
                        + CommonConst.COMMA + "01" + CommonConst.COMMA + today + CommonConst.COMMA + map.get("comBhGrantDetailId") + CommonConst.COMMA + dto.getRemark();
                if (i < grantDetailList.size()) {
                    out.write(info + line);
                } else {
                    out.write(info);
                }
                i++;
            }
            out.flush(); // 把缓存区内容压入文件
            out.close(); // 最后记得关闭文件
            String platform = commonService.getSystemConfig("PLATFORM_SIGN");
            String aliUrl = BcxinFileUtils.aliOSSFixFileTempUpload(file, platform);
            file.delete();
            resultMap.put("fileUrl", aliUrl);
            return resultMap;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        } finally {
            if (out != null) {
                try {
                    out.close();
                } catch (IOException e1) {
                }
            }
            if (fw != null) {
                try {
                    fw.close();
                } catch (IOException e1) {
                }
            }
        }
    }


    /**
     * 提交发放申请(测试用)
     *
     * @param dto
     * @author llc
     * @date 2020-09-02
     */
    @Override
    public Result commitGrantApplyCs(ComBhGrantInfoDTO dto) throws V5BusinessException {
        if (dto.getComBhGrantId() == null) {
            throw new V5BusinessException(Result.ERROR, "发放ID不能为空");
        }
        if (StringUtils.isEmpty(dto.getActiveCode())) {
            throw new V5BusinessException(Result.ERROR, "验证码不能为空");
        }
        ComBhGrantInfo comBhGrantInfo = comBhGrantInfoMapper.getComBhGrantInfoById(dto.getComBhGrantId());
        if (comBhGrantInfo == null) {
            throw new V5BusinessException(Result.ERROR, "发放ID有误");
        }

        Map<String, Object> requestMap = Maps.newHashMap();
        /*** 调用支付平台上传发放文件 ***/
        Map<Object, Object> p = Maps.newHashMap();
        p.put("MerUserId", dto.getComId().toString());//平台会员号
        p.put("VerifyNo", dto.getActiveCode());//手机劢态验证码
        p.put("VirlAcctType", "01");//平台会员账户类型，用于校验交易密码01:基本户；02:保证金户； 04:只收丌付户；05:融资与用户
        p.put("FileName", dto.getKeyword());
        p.put("NoticeType", BHGRANTFILETYPE);//4：代发
        p.put("ReMark", dto.getRemark());//4：代发
        String token = JwtUtil.createJWT(p);// 加密生成token
        requestMap.put("token", token);
        String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
        String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.UPLOAD_GRANT_FILE, requestMap);
        Result result = JSON.parseObject(resultStr, Result.class);
        /*** 如果返回成功 ***/
        if (Result.SUCCESS.equals(result.getRetType())) {
            /*** 更新渤海薪资代发单 ***/
            comBhGrantInfo.setTotalGrantCount(comBhGrantInfo.getGrantCount());
            comBhGrantInfo.setTotalSumAmount(comBhGrantInfo.getSumAmount());
            comBhGrantInfo.setUpdateBy(dto.getUpdateBy());
            comBhGrantInfo.setUpdateTime(dto.getUpdateTime());
            comBhGrantInfo.setCommitStatus(CommonConst.Y);
            comBhGrantInfoMapper.updateComBhGrantInfo(comBhGrantInfo);
            return Result.success(Result.SUCCESS_MSG);
        } else {
            /*** 返回失败则抛错 ***/
            return Result.fail(result.getMsg());
        }
    }

    /**
     * 获取企业渤海薪资代发记录
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-08-14
     */
    @Override
    public Result getBhGrantOperRecordList(ComBhGrantInfoDTO dto) throws V5BusinessException {
        if (dto.getComBhGrantId() == null) {
            return Result.fail("发放ID不能为空");
        }
        List<Map<String, Object>> list = comBhGrantOperRecordMapper.getBhGrantOperRecordList(dto.getComBhGrantId());
        return Result.success(Result.SUCCESS_QUERY_MSG, list);
    }


    /**
     * 获取渤海银行回单文件URL
     *
     * @param dto
     * @author llc
     * @date 2020-09-16
     */
    @Override
    public Result getBhBankReceiptFileUrl(ComBhGrantInfoDTO dto) throws V5BusinessException {
        if (dto.getComBhGrantId() == null) {
            return Result.fail("发放ID不能为空");
        }
        ComBhGrantInfo comBhGrantInfo = comBhGrantInfoMapper.getComBhGrantInfoById(dto.getComBhGrantId());
        if (comBhGrantInfo == null) {
            return Result.fail("发放ID有误");
        }
        /*** 如果已经生成了回执文件，则直接返回 ***/
        if (StringUtil.isNotEmpty(comBhGrantInfo.getBankReceiptFileUrl())) {
            return Result.success(comBhGrantInfo.getBankReceiptFileUrl());
        } else {
            /*** 调用支付平台获取回执文件名称 ***/
            Map<String, Object> requestMap = Maps.newHashMap();
            Map<Object, Object> p = Maps.newHashMap();
            p.put("MerUserId", dto.getComId().toString());//平台会员号
            p.put("VirlAcctType", "01");//平台会员账户类型，用于校验交易密码01:基本户；02:保证金户； 04:只收丌付户；05:融资与用户
            p.put("FileName", comBhGrantInfo.getFileName());
            String token = JwtUtil.createJWT(p);// 加密生成token
            requestMap.put("token", token);
            String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
            String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.GET_RECEIPT_FILE, requestMap);
            Result result = JSON.parseObject(resultStr, Result.class);
            /*** 如果返回成功 ***/
            if (Result.SUCCESS.equals(result.getRetType())) {
                if(result.getData() == null){
                    return Result.fail("获取电子回单文件名称失败！");
                }
                String data = String.valueOf(result.getData());
                String FileName = data.substring(2,data.length()-2);
                /*** 调用支付平台获取代发回执文件 ***/
                Map<Object, Object> params = Maps.newHashMap();
                params.put("FileName", FileName);
                token = JwtUtil.createJWT(params);// 加密生成token
                requestMap.put("token", token);
                platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
                resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.DOWNLOAD_FILE, requestMap);
                result = JSON.parseObject(resultStr, Result.class);
                /*** 如果返回成功 ***/
                if (Result.SUCCESS.equals(result.getRetType())) {
                    String fileStr = result.getData().toString(); // 返回的文件base64数据

                    try {
                        String upload_path = commonService.getSystemConfig("SYS_PIC_PATH");
                        String dirDate = DateUtil.today(); // 当前日期
                        String dirPaht = upload_path + "temp/" + dirDate;
                        File dir = new File(dirPaht);
                        if (!dir.exists()) {
                            dir.mkdirs();
                        }
                        String fileName = idWorker.nextId() + ".pdf";
                        String filePath = dirPaht + "/" + fileName;
                        Base64Utils.decodeToFile(filePath, fileStr);

                        /*** 文件上传阿里云 ***/
                        File file = new File(filePath);
                        String platform = commonService.getSystemConfig("PLATFORM_SIGN");
                        String aliUrl = BcxinFileUtils.aliOSSFixFileTempUpload(file, platform);
                        file.delete();

                        /*** 更新渤海发放信息 ***/
                        comBhGrantInfo.setBankReceiptFileUrl(aliUrl);
                        comBhGrantInfo.setUpdateTime(new Date());
                        comBhGrantInfo.setUpdateBy(dto.getUpdateBy());
                        comBhGrantInfoMapper.updateComBhGrantInfo(comBhGrantInfo);
                        return Result.success(aliUrl);
                    } catch (Exception e) {
                        e.printStackTrace();
                        return Result.success("获取回单文件出错");
                    } finally {

                    }

                } else {
                    /*** 返回失败则抛错 ***/
                    return Result.fail(result.getMsg());
                }
            } else {
                /*** 返回失败则抛错 ***/
                return Result.fail(result.getMsg());
            }
        }
    }


    /**
     * 同步渤海发放结果
     *
     * @param dto
     * @author llc
     * @date 2020-09-17
     */
    @Override
    public Result syncBhGrantResult(ComBhGrantInfoDTO dto) throws V5BusinessException {
        if (dto.getComBhGrantId() == null) {
            return Result.fail("发放ID不能为空");
        }
        ComBhGrantInfo comBhGrantInfo = comBhGrantInfoMapper.getComBhGrantInfoById(dto.getComBhGrantId());
        if (comBhGrantInfo == null) {
            return Result.fail("代发单ID有误");
        }

        if (ObjectUtils.equals(comBhGrantInfo.getIsSyncBankResult(), CommonConst.Y)) {
            return Result.fail("代发结果已同步，无需再次同步");
        }
        /*** 如果已提交到渤海银行且未接收到结果信息 ***/
        if (ObjectUtils.equals(comBhGrantInfo.getCommitStatus(), CommonConst.Y) && ObjectUtils.equals(comBhGrantInfo.getIsSyncBankResult(), CommonConst.N)) {
            /*** 调用渤海银行接口获取代发结果 ***/

            Map<String, Object> requestMap = Maps.newHashMap();
            Map<Object, Object> p = Maps.newHashMap();
            p.put("MerUserId", dto.getComId().toString());//平台会员号
            p.put("VirlAcctType", "01");//平台会员账户类型，用于校验交易密码01:基本户；02:保证金户； 04:只收丌付户；05:融资与用户
            p.put("FileName", comBhGrantInfo.getFileName());
         //   p.put("TransStat", "F");//主机状态 执行结果：S-成功
            String token = JwtUtil.createJWT(p);// 加密生成token
            requestMap.put("token", token);
            String platUrl = commonService.getSystemConfig(CommonConst.PAYMENT_PLATFORM_URL);
            String resultStr = HttpUtil.post(platUrl + BhPaymentServiceConst.GET_GRANT_RESULT, requestMap);
            Result result = JSON.parseObject(resultStr, Result.class);
            /*** 如果返回成功 ***/
            if (Result.SUCCESS.equals(result.getRetType())) {
                Map<String, Object> resultMap = (Map<String, Object>) result.getData();

                if (resultMap != null && resultMap.get("TransInfoList") != null) {
                    List<Map> transInfoList = JSON.parseArray(resultMap.get("TransInfoList").toString(), Map.class);
                    logger.info(JSON.toJSONString(transInfoList));
                    List<Map> successTransInfoList = new ArrayList<>(); // 成功交易列表
                    List<Map> failTransInfoList = new ArrayList<>(); // 失败交易列表
                    BigDecimal totalSumAmount = new BigDecimal(0) ;  // 实际发放总额
                    BigDecimal failSumAmount = new BigDecimal(0); // 失败发放总额
                    for (Map map : transInfoList) {
                        if (ObjectUtils.equals(map.get("HrspCode").toString(), BhPaymentServiceConst.TRANS_SUCCESS_CODE)) {
                            successTransInfoList.add(map);
                            totalSumAmount = totalSumAmount.add(new BigDecimal(map.get("Amt").toString()));
                        } else {
                            failSumAmount = failSumAmount.add(new BigDecimal(map.get("Amt").toString()));
                            failTransInfoList.add(map);
                        }
                    }

                    /*** 更新成功交易列表 ***/
                    if (successTransInfoList != null && successTransInfoList.size() > 0) {
                        comBhGrantDetailMapper.updateBhGrantSuccessDetail(dto.getUpdateBy(), dto.getComBhGrantId(), successTransInfoList);
                    }

                    /*** 更新失败交易列表 ***/
                    if (failTransInfoList != null && failTransInfoList.size() > 0) {
                        comBhGrantDetailMapper.updateBhGrantFailDetail(dto.getUpdateBy(), dto.getComBhGrantId(), failTransInfoList);
                    }

                    /*** 更新代发单成功笔数和失败笔数、代发结果同步标志 ***/
                    comBhGrantInfo.setTotalSumAmount(totalSumAmount);
                    comBhGrantInfo.setTotalGrantCount(successTransInfoList.size());
                    comBhGrantInfo.setFailSumAmount(failSumAmount);
                    comBhGrantInfo.setFailGrantCount(failTransInfoList.size());
                    comBhGrantInfo.setUpdateTime(new Date());
                    comBhGrantInfo.setUpdateBy(dto.getUpdateBy());
                    comBhGrantInfo.setIsSyncBankResult(CommonConst.Y);
                    comBhGrantInfoMapper.updateComBhGrantInfo(comBhGrantInfo);
                    return Result.success(Result.SUCCESS_QUERY_MSG);
                }
            }else {
                /*** 返回失败则抛错 ***/
                return Result.fail(result.getMsg());
            }

        }
        return Result.success(Result.SUCCESS_QUERY_MSG);
    }
}