package com.bcxin.platform.service.cache;

import com.alibaba.fastjson.JSON;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.common.utils.StringUtils;
import com.bcxin.platform.service.oauth.RedisUtil;
import com.bcxin.platform.util.AESUtil;
import com.bcxin.platform.util.HttpUtils;
import com.bcxin.platform.util.ResultJson;
import com.bcxin.platform.util.constants.BBDServiceConst;
import com.bcxin.platform.util.constants.HuaweiMeetConst;
import com.bcxin.platform.util.constants.PublicConst;
import com.bcxin.platform.util.huawei.HttpBuildUtil;
import com.bcxin.platform.util.huawei.RestRequest;
import com.bcxin.platform.util.huawei.RestResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @Author: 陈堂君
 * @Date: 2018/1/2
 * @Description:
 * @Modified By:
 */
@Service("taskCacheService")
public class TaskCacheServiceImpl implements TaskCacheService {

    /**
     * 日志对象
     */
    protected Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    @Resource
    private RedisUtil redisUtil;

    public static Result bbdLogin(String comIdEncode, String configURL) {
        Map<String, Object> paramMap = new HashMap<String, Object>();
        paramMap.put("comId", comIdEncode);
        String url = configURL + BBDServiceConst.BBD_LOGIN;
        ResultJson resultJson = HttpUtils.sendHttpMap(url, paramMap);
        if (resultJson != null && resultJson.isSuccessful()) {
            return Result.success(Result.SUCCESS_QUERY_MSG, resultJson.getData());
        } else {
            return Result.fail(resultJson.getMsg());
        }
    }

    @Override
    public String getBbdAccessToken(Long comId, String platUrl) {
        String redisKey = redisUtil.REDIS_PREFIX_KEY + ":bbdToken" + comId;
        String accessToken = "";
        if (!stringRedisTemplate.hasKey(redisKey)) {
            String comIdEncode = "";
            try {
                comIdEncode = AESUtil.encode(comId.toString());
            } catch (Exception e) {
                logger.error(e.getMessage());
            }
            Result result = bbdLogin(comIdEncode, platUrl);
            if (result.getRetType().equals(Result.SUCCESS)) {
                Map<String, Object> resultMap = JSON.parseObject(result.getData().toString(), Map.class);
                if (resultMap != null) {
                    accessToken = resultMap.get("sid").toString();
                    /*** 百保盾token失效时间- 1小时 ***/
                    Long expiresTime = PublicConst.ONE_MONTH_TIMEOUT_SECONDS - 3600L;
                    stringRedisTemplate.opsForValue().set(redisKey, accessToken, expiresTime, TimeUnit.SECONDS);
                }
            } else {
                throw new V5BusinessException(Result.BUSINESS_ERROR, "登录公安子站失败");
            }
        } else {
            accessToken = stringRedisTemplate.opsForValue().get(redisKey);
        }
        return accessToken;
    }

    @Override
    public void setActiveCodeSerial(String key, String value) {
        // 使用冒号分隔，客户端工具可以形成命名空间折叠方便阅读。
        stringRedisTemplate.opsForValue().set(redisUtil.REDIS_PREFIX_KEY + ":ActiveCodeSerial:" + key, value,
                PublicConst.PHONE_AUTH_CODE_TIMEOUT, TimeUnit.MILLISECONDS);
    }

    @Override
    public String getActiveCodeSerial(String key) {
        return stringRedisTemplate.opsForValue().get(redisUtil.REDIS_PREFIX_KEY + ":ActiveCodeSerial:" + key);
    }

    /**
     * 记录手机号发送验证码次数
     */
    @Override
    public void setSMSPhoneCount(String key, int count) {
        if (count == 0) {
            stringRedisTemplate.opsForValue().set(redisUtil.REDIS_PREFIX_KEY + ":PhoneCount:" + key, (count + 1) + "",
                    PublicConst.ONE_DAY_TIMEOUT_MILLISECONDS, TimeUnit.MILLISECONDS);
        } else {
            stringRedisTemplate.opsForValue().set(redisUtil.REDIS_PREFIX_KEY + ":PhoneCount:" + key, (count + 1) + "", 0);
        }
    }

    /**
     * 记录手机号发送验证码次数
     */
    @Override
    public int getSMSPhoneCount(String key) {
        String count = stringRedisTemplate.opsForValue().get(redisUtil.REDIS_PREFIX_KEY + ":PhoneCount:" + key);
        if (StringUtils.isEmpty(count)) {
            return 0;
        }
        return Integer.parseInt(count);
    }

    /**
     * 设置手机验证码缓存
     *
     * @param key
     * @param value
     * @apiNote 作者 陈堂君
     * @since 创建于 2018/1/2 16:16
     */
    @Override
    public void setPhoneAuthCode(String key, String value) {
        // 使用冒号分隔，客户端工具可以形成命名空间折叠方便阅读。
        stringRedisTemplate.opsForValue().set(redisUtil.REDIS_PREFIX_KEY + ":PhoneAuthCode:" + key + value, value,
                PublicConst.PHONE_AUTH_CODE_TIMEOUT, TimeUnit.MILLISECONDS);
    }

    /**
     * 获取手机验证码
     *
     * @param key
     * @return
     * @apiNote 作者 陈堂君
     * @since 创建于 2018/1/2 16:17
     */
    @Override
    public String getPhoneAuthCode(String key, String value) {

        /**** 为了测试方便，验证码固定为888888则跳过校验 ****/
        if (value.equalsIgnoreCase("888888")) {
            return value;
        }
        String result = stringRedisTemplate.opsForValue().get(redisUtil.REDIS_PREFIX_KEY + ":PhoneAuthCode:" + key + value);
        return result;
    }


    /**
     * 获取华为云会议acessToken
     *
     * @param userName
     * @param passWord
     * @param perId
     * @return accessToken
     * @auth llc
     * @date 2019-12-12
     */
    @Override
    public String getHuaweiMeetAccessToken(String userName, String passWord, Long perId) {
        String redisKey = redisUtil.REDIS_PREFIX_KEY + ":huaweiMeet" + perId;
        String accessToken = "";
        if (!stringRedisTemplate.hasKey(redisKey)) {
            String authorization = Base64.getEncoder().encodeToString((userName + ":" + passWord).getBytes());
            Map<String, Object> map = new HashMap<>();
            map.put("account", userName);
            map.put("clientType", 0);
            map.put("createTokenType", 0);
            String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_ACCESSTOKEN;
            Map<String, String> headMap = new HashMap<>();
            headMap.put("Authorization", "Basic " + authorization);
            RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, JSON.toJSONString(map), null);
            RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
            if (response.getHttpCode() == HuaweiMeetConst.HUAWEI_SUCCESS_CODE) {
                Map<String, Object> resultMap = JSON.parseObject(response.getEntity().toString(), Map.class);
                if (resultMap != null) {
                    accessToken = resultMap.get("accessToken").toString();
                    String expires_in = resultMap.get("validPeriod").toString();
                    /*** 华为云token失效时间- 一小时 ***/
                    Long expiresTime = Long.parseLong(expires_in) - 3600L;
                    if (expiresTime > 0) {
                        stringRedisTemplate.opsForValue().set(redisKey, accessToken, expiresTime, TimeUnit.SECONDS);
                    }
                }
            }
        } else {
            accessToken = stringRedisTemplate.opsForValue().get(redisKey);
            logger.info("从缓存中拿accessToken:" + accessToken);
        }
        return accessToken;
    }


    /**
     * 获取企业管理员华为云会议acessToken
     *
     * @param userName
     * @param passWord
     * @param comId
     * @return accessToken
     * @auth llc
     * @date 2019-12-18
     */
    @Override
    public String getComManageHuaweiMeetAccessToken(String userName, String passWord, Long comId) {
        String redisKey = redisUtil.REDIS_PREFIX_KEY + ":huaweiMeetCom" + comId;
        String accessToken = "";
        if (!stringRedisTemplate.hasKey(redisKey)) {
            String authorization = Base64.getEncoder().encodeToString((userName + ":" + passWord).getBytes());
            Map<String, Object> map = new HashMap<>();
            map.put("account", userName);
            map.put("clientType", 0);
            map.put("createTokenType", 0);
            String url = HuaweiMeetConst.GATEWAY_URL + HuaweiMeetConst.GET_ACCESSTOKEN;
            Map<String, String> headMap = new HashMap<>();
            headMap.put("Authorization", "Basic " + authorization);
            RestRequest restRequest = HttpBuildUtil.buildRestRequest("POST", headMap, JSON.toJSONString(map), null);
            RestResponse response = HttpBuildUtil.sendMsg(url, restRequest);
            System.out.println("getComManageHuaweiMeetAccessToken => response:" + response.toString());
            if (response.getHttpCode() == HuaweiMeetConst.HUAWEI_SUCCESS_CODE) {
                Map<String, Object> resultMap = JSON.parseObject(response.getEntity().toString(), Map.class);
                if (resultMap != null) {
                    accessToken = resultMap.get("accessToken").toString();
                    String expires_in = resultMap.get("validPeriod").toString();
                    /*** 华为云token失效时间- 一小时 ***/
                    Long expiresTime = Long.parseLong(expires_in) - 3600L;
                    if (expiresTime > 0) {
                        stringRedisTemplate.opsForValue().set(redisKey, accessToken, expiresTime, TimeUnit.SECONDS);
                    }
                }
            }
        } else {
            accessToken = stringRedisTemplate.opsForValue().get(redisKey);
            logger.info("从缓存中拿accessToken:" + accessToken);
        }
        return accessToken;
    }

    /**
     * <b> 设置房间人数 </b>
     *
     * @param room  房间号
     * @param count 人数
     * @author ZXF
     * @create 2020/07/10 0010 13:37
     * @version
     * @注意事项 </b>
     */
    @Override
    public boolean setRoomPerNum(String room, String count) {
        if (StringUtils.isNotEmpty(count) && StringUtils.isNotEmpty(room)) {
            String redisKey = redisUtil.REDIS_PREFIX_KEY + ":room_" + room;
            logger.info("设置房间人数（" + redisKey + "）:" + count);
            stringRedisTemplate.opsForValue().set(redisKey, count, PublicConst.ONE_DAY_TIMEOUT, TimeUnit.MILLISECONDS);
            return true;
        }
        return false;
    }

    /**
     * <b> 获取房间人数 </b>
     *
     * @return 人数#是否解散房间（字符串直接拼接）
     * @author ZXF
     * @create 2020/07/10 0010 13:38
     * @version
     * @注意事项 </b>
     */
    @Override
    public String getRoomPerNum(String room) {
        String redisKey = redisUtil.REDIS_PREFIX_KEY + ":room_" + room;
        String count = stringRedisTemplate.opsForValue().get(redisKey);
        logger.info("获取房间人数（" + redisKey + "）:" + count);
        return stringRedisTemplate.opsForValue().get(redisUtil.REDIS_PREFIX_KEY + ":room_" + room);
    }

    /**
     * <b> 解散房间时删除房间redis记录 </b>
     *
     * @author ZXF
     * @create 2020/08/20 0020 10:27
     * @version
     * @注意事项 </b>
     */
    @Override
    public void delRoom(String room) {
        String redisKey = redisUtil.REDIS_PREFIX_KEY + ":room_" + room;
        logger.info("解散房间（" + redisKey + "）--------------");
        getRoomPerNum(room);
        stringRedisTemplate.delete(redisKey);
    }
}
