Object.defineProperty(exports, "__esModule", { value: true });
exports.createLanguageContext = exports.getPlugins = void 0;
const path_1 = require("path");
const localTypes = require("./utils/localTypes");
const sourceFile_1 = require("./sourceFile");
const documentRegistry_1 = require("./documentRegistry");
const useHtmlFilePlugin = require("./plugins/file-html");
const useMdFilePlugin = require("./plugins/file-md");
const useVueFilePlugin = require("./plugins/file-vue");
const useVueSfcCustomBlocks = require("./plugins/vue-sfc-customblocks");
const useVueSfcScriptsFormat = require("./plugins/vue-sfc-scripts");
const useVueSfcStyles = require("./plugins/vue-sfc-styles");
const useVueSfcTemplate = require("./plugins/vue-sfc-template");
const useHtmlPlugin = require("./plugins/vue-template-html");
const usePugPlugin = require("./plugins/vue-template-pug");
const vue_tsx_1 = require("./plugins/vue-tsx");
const ts_1 = require("./utils/ts");
function getPlugins(ts, rootDir, compilerOptions, _vueCompilerOptions, extraPlugins = []) {
    const _plugins = [
        useVueFilePlugin,
        useMdFilePlugin,
        useHtmlFilePlugin,
        useHtmlPlugin,
        usePugPlugin,
        useVueSfcStyles,
        useVueSfcCustomBlocks,
        useVueSfcScriptsFormat,
        useVueSfcTemplate,
        vue_tsx_1.default,
        ...extraPlugins,
    ];
    const vueCompilerOptions = (0, ts_1.getVueCompilerOptions)(_vueCompilerOptions);
    if (typeof (require === null || require === void 0 ? void 0 : require.resolve) === 'function') {
        for (const pluginPath of vueCompilerOptions.plugins) {
            try {
                const importPath = require.resolve(pluginPath, { paths: [rootDir] });
                const plugin = require(importPath);
                _plugins.push(plugin);
            }
            catch (error) {
                console.log('Load plugin failed', pluginPath, error);
            }
        }
    }
    else {
        console.log('vueCompilerOptions.plugins is not available in Web.');
    }
    const pluginCtx = {
        modules: {
            typescript: ts,
        },
        compilerOptions,
        vueCompilerOptions: vueCompilerOptions,
    };
    const plugins = _plugins.map(plugin => plugin(pluginCtx)).sort((a, b) => {
        var _a, _b;
        const aOrder = (_a = a.order) !== null && _a !== void 0 ? _a : 0;
        const bOrder = (_b = b.order) !== null && _b !== void 0 ? _b : 0;
        return aOrder - bOrder;
    });
    return plugins;
}
exports.getPlugins = getPlugins;
function createLanguageContext(host, extraPlugins = [], exts = ['.vue', '.md', '.html']) {
    let lastProjectVersion;
    let tsProjectVersion = 0;
    const ts = host.getTypeScriptModule();
    // from https://github.com/johnsoncodehk/volar/pull/1543
    if (!(ts.__VLS_pitched_resolveModuleNames)) {
        ts.__VLS_pitched_resolveModuleNames = true;
        const resolveModuleNames = ts.resolveModuleName;
        ts.resolveModuleName = (...args) => {
            if (args[0].endsWith('.vue') && args[6] === ts.ModuleKind.ESNext) {
                args[6] = ts.ModuleKind.CommonJS;
            }
            return resolveModuleNames(...args);
        };
    }
    const documentRegistry = (0, documentRegistry_1.createDocumentRegistry)();
    const compilerOptions = host.getCompilationSettings();
    const vueCompilerOptions = (0, ts_1.getVueCompilerOptions)(host.getVueCompilationSettings());
    const tsFileVersions = new Map();
    const sharedTypesScript = ts.ScriptSnapshot.fromString(localTypes.getTypesCode(vueCompilerOptions.target));
    const scriptSnapshots = new Map();
    const fileVersions = new WeakMap();
    const vueFileVersions = new WeakMap();
    const _tsHost = {
        fileExists: host.fileExists
            ? fileName => {
                var _a, _b;
                const basename = path_1.posix.basename(fileName);
                if (basename === localTypes.typesFileName) {
                    return true;
                }
                // .vue.js -> .vue
                // .vue.ts -> .vue
                // .vue.d.ts (never)
                const vueFileName = fileName.substring(0, fileName.lastIndexOf('.'));
                if (exts.some(ext => vueFileName.endsWith(ext))) {
                    const vueFile = documentRegistry.get(vueFileName);
                    if (!vueFile) {
                        const fileExists = !!((_a = host.fileExists) === null || _a === void 0 ? void 0 : _a.call(host, vueFileName));
                        if (fileExists) {
                            // create virtual files
                            const scriptSnapshot = host.getScriptSnapshot(vueFileName);
                            if (scriptSnapshot) {
                                documentRegistry.set(vueFileName, (0, sourceFile_1.createSourceFile)(vueFileName, scriptSnapshot, ts, plugins));
                            }
                        }
                    }
                }
                if (!!documentRegistry.fromEmbeddedFileName(fileName)) {
                    return true;
                }
                return !!((_b = host.fileExists) === null || _b === void 0 ? void 0 : _b.call(host, fileName));
            }
            : undefined,
        getProjectVersion: () => {
            return tsProjectVersion.toString();
        },
        getScriptFileNames,
        getScriptVersion,
        getScriptSnapshot,
        readDirectory: (_path, extensions, exclude, include, depth) => {
            var _a, _b;
            const result = (_b = (_a = host.readDirectory) === null || _a === void 0 ? void 0 : _a.call(host, _path, extensions, exclude, include, depth)) !== null && _b !== void 0 ? _b : [];
            for (const vuePath of documentRegistry.getFileNames()) {
                const vuePath2 = path_1.posix.join(_path, path_1.posix.basename(vuePath));
                if (path_1.posix.relative(_path.toLowerCase(), vuePath.toLowerCase()).startsWith('..')) {
                    continue;
                }
                if (!depth && vuePath.toLowerCase() === vuePath2.toLowerCase()) {
                    result.push(vuePath2);
                }
                else if (depth) {
                    result.push(vuePath2); // TODO: depth num
                }
            }
            return result;
        },
        getScriptKind(fileName) {
            if (exts.some(ext => fileName.endsWith(ext)))
                return ts.ScriptKind.TSX; // can't use External, Unknown
            switch (path_1.posix.extname(fileName)) {
                case '.js': return ts.ScriptKind.JS;
                case '.jsx': return ts.ScriptKind.JSX;
                case '.ts': return ts.ScriptKind.TS;
                case '.tsx': return ts.ScriptKind.TSX;
                case '.json': return ts.ScriptKind.JSON;
                default: return ts.ScriptKind.Unknown;
            }
        },
    };
    const plugins = getPlugins(ts, host.getCurrentDirectory(), compilerOptions, vueCompilerOptions, extraPlugins);
    return {
        typescriptLanguageServiceHost: new Proxy(_tsHost, {
            get: (target, property) => {
                update();
                return target[property] || host[property];
            },
        }),
        mapper: new Proxy(documentRegistry, {
            get: (target, property) => {
                update();
                return target[property];
            },
        }),
        plugins,
    };
    function update() {
        var _a;
        const newProjectVersion = (_a = host.getProjectVersion) === null || _a === void 0 ? void 0 : _a.call(host);
        const sholdUpdate = newProjectVersion === undefined || newProjectVersion !== lastProjectVersion;
        if (!sholdUpdate)
            return;
        lastProjectVersion = newProjectVersion;
        const fileNames = host.getScriptFileNames();
        const vueFileNames = new Set(fileNames.filter(file => exts.some(ext => file.endsWith(ext))));
        const tsFileNames = new Set(fileNames.filter(file => !exts.some(ext => file.endsWith(ext))));
        const fileNamesToRemove = [];
        const fileNamesToCreate = [];
        const fileNamesToUpdate = [];
        let tsFileUpdated = false;
        // .vue
        for (const vueFile of documentRegistry.getAll()) {
            const newVersion = host.getScriptVersion(vueFile.fileName);
            if (vueFileVersions.get(vueFile) !== newVersion) {
                vueFileVersions.set(vueFile, newVersion);
                if (host.getScriptSnapshot(vueFile.fileName)) {
                    // update
                    fileNamesToUpdate.push(vueFile.fileName);
                }
                else {
                    // delete
                    fileNamesToRemove.push(vueFile.fileName);
                }
            }
        }
        for (const nowFileName of vueFileNames) {
            if (!documentRegistry.get(nowFileName)) {
                // add
                fileNamesToCreate.push(nowFileName);
            }
        }
        // .ts / .js / .d.ts / .json ...
        for (const [oldTsFileName, oldTsFileVersion] of [...tsFileVersions]) {
            const newVersion = host.getScriptVersion(oldTsFileName);
            if (oldTsFileVersion !== newVersion) {
                if (!tsFileNames.has(oldTsFileName) && !host.getScriptSnapshot(oldTsFileName)) {
                    // delete
                    tsFileVersions.delete(oldTsFileName);
                }
                else {
                    // update
                    tsFileVersions.set(oldTsFileName, newVersion);
                }
                tsFileUpdated = true;
            }
        }
        for (const nowFileName of tsFileNames) {
            if (!tsFileVersions.has(nowFileName)) {
                // add
                const newVersion = host.getScriptVersion(nowFileName);
                tsFileVersions.set(nowFileName, newVersion);
                tsFileUpdated = true;
            }
        }
        for (const uri of fileNamesToRemove) {
            if (documentRegistry.delete(uri)) {
                tsFileUpdated = true;
            }
        }
        for (const fileName of [
            ...fileNamesToCreate,
            ...fileNamesToUpdate,
        ]) {
            const scriptSnapshot = host.getScriptSnapshot(fileName);
            if (!scriptSnapshot) {
                continue;
            }
            const sourceFile = documentRegistry.get(fileName);
            if (!sourceFile) {
                documentRegistry.set(fileName, (0, sourceFile_1.createSourceFile)(fileName, scriptSnapshot, ts, plugins));
                tsFileUpdated = true;
            }
            else {
                const oldScripts = {};
                const newScripts = {};
                if (!tsFileUpdated) {
                    for (const embedded of sourceFile.allEmbeddeds) {
                        if (embedded.file.isTsHostFile) {
                            oldScripts[embedded.file.fileName] = embedded.file.codeGen.getText();
                        }
                    }
                }
                sourceFile.update(scriptSnapshot);
                if (!tsFileUpdated) {
                    for (const embedded of sourceFile.allEmbeddeds) {
                        if (embedded.file.isTsHostFile) {
                            newScripts[embedded.file.fileName] = embedded.file.codeGen.getText();
                        }
                    }
                }
                if (Object.keys(oldScripts).length !== Object.keys(newScripts).length
                    || Object.keys(oldScripts).some(fileName => oldScripts[fileName] !== newScripts[fileName])) {
                    tsFileUpdated = true;
                }
            }
        }
        if (tsFileUpdated) {
            tsProjectVersion++;
        }
    }
    function getScriptFileNames() {
        const tsFileNames = documentRegistry.getDirs().map(dir => path_1.posix.join(dir, localTypes.typesFileName));
        for (const mapped of documentRegistry.getAllEmbeddeds()) {
            if (mapped.embedded.file.isTsHostFile) {
                tsFileNames.push(mapped.embedded.file.fileName); // virtual .ts
            }
        }
        for (const fileName of host.getScriptFileNames()) {
            if (host.isTsPlugin) {
                tsFileNames.push(fileName); // .vue + .ts
            }
            else if (!exts.some(ext => fileName.endsWith(ext))) {
                tsFileNames.push(fileName); // .ts
            }
        }
        return tsFileNames;
    }
    function getScriptVersion(fileName) {
        var _a, _b, _c;
        const basename = path_1.posix.basename(fileName);
        if (basename === localTypes.typesFileName) {
            return '';
        }
        let mapped = documentRegistry.fromEmbeddedFileName(fileName);
        if (mapped) {
            if (fileVersions.has(mapped.embedded.file)) {
                return fileVersions.get(mapped.embedded.file);
            }
            else {
                let version = (_c = (_b = (_a = ts.sys) === null || _a === void 0 ? void 0 : _a.createHash) === null || _b === void 0 ? void 0 : _b.call(_a, mapped.embedded.file.codeGen.getText())) !== null && _c !== void 0 ? _c : mapped.embedded.file.codeGen.getText();
                if (host.isTsc) {
                    // fix https://github.com/johnsoncodehk/volar/issues/1082
                    version = host.getScriptVersion(mapped.vueFile.fileName) + ':' + version;
                }
                fileVersions.set(mapped.embedded.file, version);
                return version;
            }
        }
        return host.getScriptVersion(fileName);
    }
    function getScriptSnapshot(fileName) {
        var _a;
        const basename = path_1.posix.basename(fileName);
        if (basename === localTypes.typesFileName) {
            return sharedTypesScript;
        }
        const version = getScriptVersion(fileName);
        const cache = scriptSnapshots.get(fileName.toLowerCase());
        if (cache && cache[0] === version) {
            return cache[1];
        }
        const mapped = documentRegistry.fromEmbeddedFileName(fileName);
        if (mapped) {
            const text = mapped.embedded.file.codeGen.getText();
            const snapshot = ts.ScriptSnapshot.fromString(text);
            scriptSnapshots.set(fileName.toLowerCase(), [version, snapshot]);
            return snapshot;
        }
        let tsScript = host.getScriptSnapshot(fileName);
        if (tsScript) {
            if (!((_a = vueCompilerOptions.strictTemplates) !== null && _a !== void 0 ? _a : false) && (
            // for vue 2.6 and vue 3
            basename === 'runtime-dom.d.ts' ||
                // for vue 2.7
                basename === 'jsx.d.ts')) {
                // allow arbitrary attributes
                let tsScriptText = tsScript.getText(0, tsScript.getLength());
                tsScriptText = tsScriptText.replace('type ReservedProps = {', 'type ReservedProps = { [name: string]: any');
                tsScript = ts.ScriptSnapshot.fromString(tsScriptText);
            }
            scriptSnapshots.set(fileName.toLowerCase(), [version, tsScript]);
            return tsScript;
        }
    }
}
exports.createLanguageContext = createLanguageContext;
//# sourceMappingURL=lsContext.js.map