import addDays from '../addDays/index.js';
import addMonths from '../addMonths/index.js';
import toDate from '../toDate/index.js';
import requiredArgs from '../_lib/requiredArgs/index.js';
import toInteger from '../_lib/toInteger/index.js';
/**
 * @name add
 * @category Common Helpers
 * @summary Add the specified years, months, days, hours, minutes and seconds to the given date.
 *
 * @description
 * Add the specified years, months, days, hours, minutes and seconds to the given date.
 *
 * @param {Date|Number} date - the date to be changed
 * @param {Duration} duration - the object with years, months, days, hours, minutes and seconds to be added
 *
 * | Key            | Description                        |
 * |----------------|------------------------------------|
 * | years          | Amount of years to be added        |
 * | months         | Amount of months to be added       |
 * | days           | Amount of days to be added         |
 * | hours          | Amount of hours to be added        |
 * | minutes        | Amount of minutes to be added      |
 * | seconds        | Amount of seconds to be added      |
 *
 * All values default to 0
 *
 * @returns {Date} the new date with the seconds added
 * @throws {TypeError} 2 arguments required
 *
 * @example
 * // Add the following duration to 1 September 2014, 10:19:50
 * var result = add(new Date(2014, 8, 1, 10, 19, 50), {
 *   years: 2,
 *   months: 24,
 *   days: 7,
 *   hours: 5,
 *   minutes: 9,
 *   seconds: 30,
 * })
 * //=> Sat Sep 8 2018 15:29:20
 */

export default function add(dirtyDate, duration) {
  requiredArgs(2, arguments);
  if (!duration || typeof duration !== 'object') return new Date(NaN);
  var years = 'years' in duration ? toInteger(duration.years) : 0;
  var months = 'months' in duration ? toInteger(duration.months) : 0;
  var days = 'days' in duration ? toInteger(duration.days) : 0;
  var hours = 'hours' in duration ? toInteger(duration.hours) : 0;
  var minutes = 'minutes' in duration ? toInteger(duration.minutes) : 0;
  var seconds = 'seconds' in duration ? toInteger(duration.seconds) : 0; // Add years and months

  var dateWithMonths = addMonths(toDate(dirtyDate), months + years * 12); // Add days

  var dateWithDays = addDays(dateWithMonths, days); // Add days, hours, minutes and seconds

  var minutesToAdd = minutes + hours * 60;
  var secondsToAdd = seconds + minutesToAdd * 60;
  var msToAdd = secondsToAdd * 1000;
  var finalDate = new Date(dateWithDays.getTime() + msToAdd);
  return finalDate;
}