package com.bcxin.tenant.domain.v5.services.impls;

import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.api.interfaces.rbacs.RbacCategoryRpcProvider;
import com.bcxin.api.interfaces.rbacs.responses.CategoryDetailGetResponse;
import com.bcxin.tenant.domain.v5.business.dtos.AppPackageReadonlyDto;
import com.bcxin.tenant.domain.v5.business.repositories.SettingRepository;
import com.bcxin.tenant.domain.v5.entities.TDomainEntity;
import com.bcxin.tenant.domain.v5.entities.TUserEntity;
import com.bcxin.tenant.domain.v5.repositories.TDomainRepository;
import com.bcxin.tenant.domain.v5.repositories.TUserRepository;
import com.bcxin.tenant.domain.v5.services.TDomainService;
import com.bcxin.tenant.domain.v5.services.TUserService;
import com.bcxin.tenant.domain.v5.services.commands.ApprovedDomainCommand;
import com.bcxin.tenant.domain.v5.services.commands.SyncDomainCommand;
import com.bcxin.tenant.domain.v5.services.commands.SyncUserRoleCommand;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

@Service
public class TDomainServiceImpl implements TDomainService {
    private final TDomainRepository domainRepository;
    private final UnitWork unitWork;
    private final JsonProvider jsonProvider;
    private final SettingRepository settingRepository;
    private final RbacCategoryRpcProvider categoryRpcProvider;
    private final TUserService userService;
    private final TUserRepository userRepository;

    public TDomainServiceImpl(TDomainRepository domainRepository,
                              UnitWork unitWork,
                              JsonProvider jsonProvider,
                              SettingRepository settingRepository,
                              RbacCategoryRpcProvider categoryRpcProvider,
                              TUserRepository userRepository,
                              TUserService userService) {
        this.domainRepository = domainRepository;
        this.unitWork = unitWork;
        this.jsonProvider = jsonProvider;
        this.settingRepository = settingRepository;
        this.categoryRpcProvider = categoryRpcProvider;
        this.userService = userService;
        this.userRepository = userRepository;
    }

    @Override
    public void create(SyncDomainCommand command) {
        update(command);
    }

    @Override
    public void update(SyncDomainCommand command) {
        Optional<TDomainEntity> tDomainOptional = this.domainRepository.findById(command.getId());
        if (!tDomainOptional.isPresent()) {
            TDomainEntity tDomain = TDomainEntity.create(command.getId(), command.getName());
            if(!CollectionUtils.isEmpty(command.getDynamic())) {
                tDomain.assignDynamic(this.jsonProvider.getJson(command.getDynamic()));
            }
            this.domainRepository.save(tDomain);
        } else {
            TDomainEntity tDomain = tDomainOptional.get();
            tDomain.change(command.getName());
            this.domainRepository.save(tDomain);
        }
    }


    @Override
    public void dispatch(ApprovedDomainCommand command) {
        AppPackageReadonlyDto appPackageReadonlyDto =
                    this.settingRepository.getBy(
                            command.getRegisterProvince(),
                            command.getRegisterCity(),
                            command.getSuperviseRegionCode(),
                            command.getInstitutionalCode()
                    );
        Optional<TDomainEntity> domainOptional = this.domainRepository.findById(command.getId());
        if (!domainOptional.isPresent()) {
            throw new BadTenantException("企业信息无效");
        }

        TDomainEntity domainEntity = domainOptional.get();
        Collection<String> apps = null;
        if (StringUtils.hasLength(appPackageReadonlyDto.getOwnApps())) {
            apps = Arrays.stream(appPackageReadonlyDto.getOwnApps().split(";")).collect(Collectors.toList());
        }

        String appJsons = this.jsonProvider.getJson(apps);
        domainEntity.changeBindApplication(appJsons);
        /**
         * 理论上, 应该只授权系统管理员, 总经理, 副总经理所有角色信息
         */
        Collection<TUserEntity> userEntities =
                this.userRepository.getByDomainId(domainEntity.getId());
        Collection<String> roleIds = this.getRoleIds(apps);
        this.unitWork.executeTran(() -> {
            Collection<String> userIds =
                    userEntities.stream().map(ii -> ii.getId()).collect(Collectors.toList());

            this.userService.dispatch(SyncUserRoleCommand.create(true,
                    domainEntity.getId(), userIds, roleIds
            ));

            if (!StringUtils.isEmpty(domainEntity.getDynamic())) {
                Map dynamic = jsonProvider.toObject(Map.class, domainEntity.getDynamic());
                if (dynamic != null && dynamic.containsKey("dtl") &&
                        !StringUtils.isEmpty(dynamic.get("dtl"))) {
                    String dtl = String.valueOf(dynamic.get("dtl"));
                    this.domainRepository.bindSupervise(dtl, domainEntity.getId());
                }
            }
            this.domainRepository.save(domainEntity);
        });
    }

    private Collection<String> getRoleIds(Collection<String> appIds) {
        Collection<CategoryDetailGetResponse> categoryDetailGetResponses = this.categoryRpcProvider.getByIds(appIds);

        return categoryDetailGetResponses.stream().flatMap(ii -> ii.getResponses().stream().map(ix -> ix.getId()))
                .collect(Collectors.toList());
    }
}
