package com.zbkj.admin.controller.merchant;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.zbkj.common.exception.CrmebException;
import com.zbkj.common.model.admin.SystemAdmin;
import com.zbkj.common.page.CommonPage;
import com.zbkj.common.request.PageParamRequest;
import com.zbkj.common.request.bcx.BcxOrderFapiaoQueryRequest;
import com.zbkj.common.request.bcx.BcxWriteFapiaoRequest;
import com.zbkj.common.response.bcx.BcxOrderFapiaoResponse;
import com.zbkj.common.response.bcx.BcxWriteFapiaoImportResponse;
import com.zbkj.common.response.excel.BcxOrderFapiaoExcelDto;
import com.zbkj.common.result.CommonResult;
import com.zbkj.common.utils.ExportUtil;
import com.zbkj.common.utils.SecurityUtil;
import com.zbkj.common.vo.LoginUserVo;
import com.zbkj.service.service.bcx.BcxOrderFapiaoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 发票管理 控制层
 */
@Slf4j
@Api(tags = "商户-发票管理-控制器")
@RestController
@RequestMapping("/api/admin/merchant/fapiao")
public class BcxMerchantOrderFapiaoController {

    @Autowired
    private BcxOrderFapiaoService bcxOrderFapiaoService;

    @ApiOperation("查询分页-发票管理列表")
//    @PreAuthorize("hasAuthority('merchant:attachment:list')")
    @RequestMapping(value = "/list", method = RequestMethod.GET)
    public CommonResult<CommonPage<BcxOrderFapiaoResponse>> queryPagedList(@Validated BcxOrderFapiaoQueryRequest request,
                                                                           @Validated PageParamRequest pageParamRequest) {
        LoginUserVo loginUserVo = SecurityUtil.getLoginUserVo();
        SystemAdmin user = loginUserVo.getUser();
        return CommonResult.success(CommonPage.restPage(bcxOrderFapiaoService.queryMerchantFapiaoPagedList(user.getMerId(), request, pageParamRequest)));
    }

    @ApiOperation(value = "发票查询-导出excel")
    //@PreAuthorize("hasAuthority('merchant:attachment:list')")
    @RequestMapping(value = "/list/excel", method = {RequestMethod.POST})
    public void queryFapiaoListExcel(@RequestBody @Validated BcxOrderFapiaoQueryRequest searchRequest, HttpServletResponse response) {
        //获取导出excel表格的数据列表
        List<BcxOrderFapiaoExcelDto> resultList = Collections.singletonList(new BcxOrderFapiaoExcelDto());
        LoginUserVo loginUserVo = SecurityUtil.getLoginUserVo();
        SystemAdmin user = loginUserVo.getUser();
        PageParamRequest pageParamRequest = new PageParamRequest();
        pageParamRequest.setPage(1);
        pageParamRequest.setLimit(65500);
        List<BcxOrderFapiaoResponse> list = bcxOrderFapiaoService.queryMerchantFapiaoPagedList(user.getMerId(), searchRequest, pageParamRequest).getList();
        if (CollectionUtil.isNotEmpty(list)) {
            //转导出对象
            resultList = list.stream().map(reportResponse -> {
                BcxOrderFapiaoExcelDto excelVo = new BcxOrderFapiaoExcelDto();
                BeanUtils.copyProperties(reportResponse, excelVo);
                return excelVo;
            }).collect(Collectors.toList());
        }
        ExportUtil<BcxOrderFapiaoExcelDto> exportUtil = new ExportUtil<>();
        exportUtil.export("发票申请单列表", resultList, response);
    }

    //@PreAuthorize("hasAuthority('merchant:upload:file')")
    @ApiOperation(value = "批量开票-上传excel")
    @RequestMapping(value = "/upload", method = RequestMethod.POST)
    public CommonResult<List<BcxWriteFapiaoImportResponse>> file(@RequestParam(value = "file") MultipartFile file) {
        SystemAdmin systemAdmin = SecurityUtil.getLoginUserVo().getUser();
        List<BcxWriteFapiaoImportResponse> resultList = new ArrayList<>();
        AnalysisEventListener<BcxWriteFapiaoRequest> listener = new AnalysisEventListener<BcxWriteFapiaoRequest>() {
            //private static final int BATCH_COUNT = 2000;
            //List<BcxWriteFapiaoRequest> list = new ArrayList<BcxWriteFapiaoRequest>();
            @Override
            public void invoke(BcxWriteFapiaoRequest record, AnalysisContext analysisContext) {
                Integer rowIndex = analysisContext.readRowHolder().getRowIndex();
                log.debug("解析第" + rowIndex + "行数据：{}", JSON.toJSONString(record));
                BcxWriteFapiaoImportResponse response = new BcxWriteFapiaoImportResponse();
                try {
                    response.setSuccess(true);
                    response.setApplyNo(record.getApplyNo());
                    response.setRowIndex(rowIndex);
                    if (!bcxOrderFapiaoService.writeFapiao(record, systemAdmin.getMerId())) {
                        response.setSuccess(false);
                        response.setMsg("处理失败，请检查数据是否正确，或联系管理员！");
                    } else {
                        response.setMsg("导入成功！");
                    }
                } catch (Exception e) {
                    response.setSuccess(false);
                    response.setMsg(e.getMessage());
                    log.error(record.getApplyNo() + "单条处理失败" + JSONObject.toJSONString(response));
                } finally {
                    resultList.add(response);
                }
            }

            @Override
            public void doAfterAllAnalysed(AnalysisContext analysisContext) {
                log.debug("全部解析完成");
            }
        };

        readExcel(file, listener);

        return CommonResult.success(resultList);
    }

    public void readExcel(MultipartFile file, AnalysisEventListener<BcxWriteFapiaoRequest> listener) {
        try {
            log.debug("发票excel解析开始");
            EasyExcel.read(file.getInputStream())
                    .autoCloseStream(false)
                    .autoTrim(true)
                    .sheet(0)
                    .head(BcxWriteFapiaoRequest.class)
                    .headRowNumber(1)
                    .registerReadListener(listener)
                    .doReadSync();
        } catch (IOException e) {
            log.error("读取excel失败", e);
            throw new CrmebException("上传失败，请检查excel格式");
        }
    }

    @ApiOperation("合并开发票")
//    @PreAuthorize("hasAuthority('merchant:attachment:list')")
    @RequestMapping(value = "/write", method = RequestMethod.POST)
    public CommonResult<Boolean> create(@RequestBody @Validated BcxWriteFapiaoRequest request) {
        SystemAdmin systemAdmin = SecurityUtil.getLoginUserVo().getUser();
        if (bcxOrderFapiaoService.writeFapiao(request, systemAdmin.getMerId())) {
            return CommonResult.success();
        }
        return CommonResult.failed();
    }
}
