package com.bcxin.identify.util.common;

import org.springframework.util.StringUtils;

import java.beans.PropertyEditorSupport;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Date;

/**
 */
public class TimestampPropertyEditor extends PropertyEditorSupport {

	private final DateFormat dateFormat;

	private final boolean allowEmpty;

	private final boolean useTimestamp;

	private final int exactDateLength;

	/**
	 * Create a new CustomDateEditor instance, using the given DateFormat for
	 * parsing and rendering.
	 * <p>
	 * The "allowEmpty" parameter states if an empty String should be allowed
	 * for parsing, i.e. get interpreted as null value. Otherwise, an
	 * IllegalArgumentException gets thrown in that case.
	 * 
	 * @param dateFormat
	 *            DateFormat to use for parsing and rendering
	 * @param allowEmpty
	 *            if empty strings should be allowed
	 */
	public TimestampPropertyEditor(DateFormat dateFormat, boolean allowEmpty, boolean useTimestamp) {
		this.dateFormat = dateFormat;
		this.allowEmpty = allowEmpty;
		this.useTimestamp = useTimestamp;
		this.exactDateLength = -1;
	}

	/**
	 * Create a new CustomDateEditor instance, using the given DateFormat for
	 * parsing and rendering.
	 * <p>
	 * The "allowEmpty" parameter states if an empty String should be allowed
	 * for parsing, i.e. get interpreted as null value. Otherwise, an
	 * IllegalArgumentException gets thrown in that case.
	 * <p>
	 * The "exactDateLength" parameter states that IllegalArgumentException gets
	 * thrown if the String does not exactly match the length specified. This is
	 * useful because SimpleDateFormat does not enforce strict parsing of the
	 * year part, not even with <code>setLenient(false)</code>. Without an
	 * "exactDateLength" specified, the "01/01/05" would get parsed to
	 * "01/01/0005".
	 * 
	 * @param dateFormat
	 *            DateFormat to use for parsing and rendering
	 * @param allowEmpty
	 *            if empty strings should be allowed
	 * @param exactDateLength
	 *            the exact expected length of the date String
	 */
	public TimestampPropertyEditor(DateFormat dateFormat, boolean allowEmpty, int exactDateLength) {
		this.dateFormat = dateFormat;
		this.allowEmpty = allowEmpty;
		this.useTimestamp = false;
		this.exactDateLength = exactDateLength;
	}

	/**
	 * Parse the Date from the given text, using the specified DateFormat.
	 */
	public void setAsText(String text) throws IllegalArgumentException {
		if (this.allowEmpty && !StringUtils.hasText(text)) {
			// Treat empty String as null value.
			setValue(null);
		} else if (text != null && this.exactDateLength >= 0 && text.length() != this.exactDateLength) {
			throw new IllegalArgumentException("Could not parse date: it is not exactly" + this.exactDateLength + "characters long");
		} else {
			if (text.indexOf(":") < 0) {
				text = text.trim() + " 00:00:00";
			}

			try {
				Object value = this.dateFormat.parse(text);
				if (useTimestamp) {
					value = new Timestamp(((Date) value).getTime());
				}
				setValue(value);
			} catch (ParseException ex) {
				throw new IllegalArgumentException("Could not parse date: " + ex.getMessage());
			}
		}
	}

	/**
	 * Format the Date as String, using the specified DateFormat.
	 */
	public String getAsText() {
		Date value = null;
		if (useTimestamp) {
			value = (Date) getValue();
		} else
			value = (Date) getValue();
		return (value != null ? this.dateFormat.format(value) : "");
	}

}
