package com.bcxin.fdd.service;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.fdd.common.emus.TaskStatusEnum;
import com.bcxin.fdd.entity.BfOrganizationEntity;
import com.bcxin.fdd.entity.BfSignTaskEntity;
import com.fasc.open.api.bean.base.BaseRes;
import com.fasc.open.api.v5_1.client.OrgClient;
import com.fasc.open.api.v5_1.req.org.GetEntityListReq;
import com.fasc.open.api.v5_1.res.org.GetEntityListRes;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;
import java.util.Optional;

/**
 * description：法大大事件回调 服务实现类
 * author：linchunpeng
 * date：2024/7/11
 */
@Slf4j
@Service
public class CallbackService {

    @Autowired
    private BfOrganizationService bfOrganizationService;
    @Autowired
    private BfSignTaskService bfSignTaskService;
    @Autowired
    private FddApiUtilService fddApiUtilService;

    /**
     * description：法大大事件回调
     * author：linchunpeng
     * date：2024/8/15
     */
    @Async
    @Transactional
    public void callback(String event, String bizContent) {
        try {
            JSONObject resultObj = JSONObject.parseObject(bizContent);
            String signTaskId = resultObj.getString("signTaskId");
            BfSignTaskEntity signTask = null;
            switch (event) {
                case "corp-authorize":
                    log.info("企业用户授权事件");
                    String clientCorpId = resultObj.getString("clientCorpId");
                    String clientUserId = resultObj.getJSONArray("clientUserIds").getString(0);
                    String openCorpId = resultObj.getString("openCorpId");
                    String memberId = resultObj.getString("memberId");
                    String entityId = "";

                    BfOrganizationEntity organizationEntity = bfOrganizationService.getByBcxOrganizationId(clientCorpId);
                    if (organizationEntity.getAuthStatus() == 0) {
                        OrgClient orgClient = new OrgClient(fddApiUtilService.getOpenApiClient());
                        GetEntityListReq getEntityListReq = new GetEntityListReq();
                        getEntityListReq.setOpenCorpId(openCorpId);
                        BaseRes<List<GetEntityListRes>> corpEntityList = orgClient.getCorpEntityList(getEntityListReq);
                        if (CollectionUtil.isNotEmpty(corpEntityList.getData())) {
                            Optional<GetEntityListRes> primary = corpEntityList.getData().stream().filter(getEntityListRes -> getEntityListRes.getEntityType().equals("primary")).findFirst();
                            if (primary.isPresent()) {
                                entityId = primary.get().getEntityId();
                            }
                        }

                        organizationEntity.setFddClientCorpId(clientCorpId);
                        organizationEntity.setFddOpenCorpId(openCorpId);
                        organizationEntity.setFddEntityId(entityId);
                        organizationEntity.setFddAdminUserId(clientUserId);
                        organizationEntity.setFddAdminMemberId(memberId);
                        organizationEntity.setAuthStatus(1);
                        organizationEntity.setUpdateTime(new Date());
                        bfOrganizationService.updateById(organizationEntity);

                        //短信通知
                        bfOrganizationService.insertOrgAuthNotice(organizationEntity);
                    } else {
                        log.info("该企业已经授权过了");
                    }
                    break;
                case "sign-task-start":
                    log.info("签署任务提交事件");
                    signTask = this.getBfSignTaskEntity(signTaskId);
                    if (signTask != null && signTask.getTaskStatus() == TaskStatusEnum.START.getCode()) {
                        //必须是发起状态的
                        signTask.setTaskStatus(TaskStatusEnum.SUBMIT.getCode());
                        signTask.setUpdateTime(new Date());
                        bfSignTaskService.updateById(signTask);
                    }
                    break;
                case "sign-task-finished":
                    log.info("签署任务完成事件");
                    signTask = this.getBfSignTaskEntity(signTaskId);
                    if (signTask != null && signTask.getTaskStatus() == TaskStatusEnum.SIGNING.getCode()) {
                        //必须是签署中状态的
                        signTask.setTaskStatus(TaskStatusEnum.COMPLETE.getCode());
                        signTask.setTaskCompleteTime(new Date());
                        signTask.setUpdateTime(new Date());
                        bfSignTaskService.updateById(signTask);

                        //下载合同附件
                        bfSignTaskService.downloadTaskFile(signTask.getId());
                    }
                    break;
                case "sign-task-sign-rejected":
                    log.info("签署任务签署方拒签事件");
                    signTask = this.getBfSignTaskEntity(signTaskId);
                    String actorId = resultObj.getString("actorId");
                    String signRejectReason = resultObj.getString("signRejectReason");
                    if (signTask != null && signTask.getTaskStatus() == TaskStatusEnum.SIGNING.getCode()) {
                        //必须是签署中状态的
                        signTask.setTaskStatus(TaskStatusEnum.REJECTED.getCode());
                        signTask.setRemark(StringUtils.isNotBlank(actorId)?actorId:"".concat("拒签，原因：").concat(StringUtils.isNotBlank(signRejectReason)?signRejectReason:""));
                        signTask.setUpdateTime(new Date());
                        bfSignTaskService.updateById(signTask);
                    }
                    break;
                case "sign-task-canceled":
                    log.info("签署任务撤销事件");
                    signTask = this.getBfSignTaskEntity(signTaskId);
                    if (signTask != null && signTask.getTaskStatus() != TaskStatusEnum.CANCEL.getCode()) {
                        //必须是不等于撤销状态的
                        signTask.setTaskStatus(TaskStatusEnum.CANCEL.getCode());
                        signTask.setUpdateTime(new Date());
                        bfSignTaskService.updateById(signTask);
                    }
                    break;
                case "sign-task-expire":
                    log.info("签署任务过期事件");
                    signTask = this.getBfSignTaskEntity(signTaskId);
                    if (signTask != null && signTask.getTaskStatus() != TaskStatusEnum.EXPIRED.getCode()) {
                        //必须是不等于过期状态的
                        signTask.setTaskStatus(TaskStatusEnum.EXPIRED.getCode());
                        signTask.setUpdateTime(new Date());
                        bfSignTaskService.updateById(signTask);
                    }
                    break;
                case "sign-task-abolish":
                    log.info("签署任务作废事件");
                    signTask = this.getBfSignTaskEntity(signTaskId);
                    if (signTask != null && signTask.getTaskStatus() != TaskStatusEnum.ABANDON.getCode()) {
                        //必须是不等于作废状态的
                        signTask.setTaskStatus(TaskStatusEnum.ABANDON.getCode());
                        signTask.setUpdateTime(new Date());
                        bfSignTaskService.updateById(signTask);
                    }
                    break;
                case "sign-task-signed":
                    log.info("签署任务参与方签署事件");
                    signTask = this.getBfSignTaskEntity(signTaskId);
                    if (signTask != null && signTask.getTaskStatus() == TaskStatusEnum.SUBMIT.getCode()) {
                        //必须是确认提交状态的
                        signTask.setTaskStatus(TaskStatusEnum.SIGNING.getCode());
                        signTask.setUpdateTime(new Date());
                        bfSignTaskService.updateById(signTask);
                    }
                    break;
                default:
                    log.info("其他事件");

                    break;
            }
        } catch (Exception e) {
            log.info("法大大事件回调异常，{}", e.getMessage(), e);
        }
    }

    private BfSignTaskEntity getBfSignTaskEntity(String signTaskId) {
        if (signTaskId == null) {
            log.info("签署任务回调事件signTaskId为空");
            return null;
        } else {
            BfSignTaskEntity signTask = bfSignTaskService.findBySignTaskId(signTaskId);
            if (signTask == null) {
                log.info("签署任务回调事件signTaskId对应的签署记录为空");
            }
            return signTask;
        }
    }
}