package com.bcxin.tenant.apis.impls;

import cn.hutool.core.util.ReUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.Infrastructures.utils.RedisUtil;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.commons.SuperviseDepartmentRpcProvider;
import com.bcxin.api.interfaces.commons.responses.SuperviseDepartmentResponse;
import com.bcxin.tenant.domain.entities.SuperviseDepartmentEntity;
import com.bcxin.tenant.domain.repositories.SuperviseDepartmentRepository;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboService;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * description: 监管机构
 * author: linchunpeng
 * date:  2023-05-10 9:46
 */
@Slf4j
@DubboService(version = ApiConstant.VERSION,validation = "true",timeout = 120 *1000,retries = 0)
public class SuperviseDepartmentRpcProviderImpl implements SuperviseDepartmentRpcProvider {


    private final SuperviseDepartmentRepository superviseDepartmentRepository;

    private final RedisUtil redisUtil;

    private final static String SUPERVISE_DEPARTMENT_REDIS_KEY = "bcxin:tenant:platform:supervise:%s";

    public SuperviseDepartmentRpcProviderImpl(SuperviseDepartmentRepository superviseDepartmentRepository, RedisUtil redisUtil){
        this.superviseDepartmentRepository = superviseDepartmentRepository;
        this.redisUtil = redisUtil;
    }

    /**
     * description: 根据区域id获取监管机构树形结构
     * author: linchunpeng
     * date:  2023-05-10 9:37
     */
    @Override
    public SuperviseDepartmentResponse findByRegionId(String regionId, String industryCode) {
        if (StringUtils.isBlank(regionId) || StringUtils.isBlank(industryCode)) {
            return null;
        }
        if (industryCode.equals("06")) {
            //群防群治行业
            if (regionId.startsWith("11")) {
                //北京，默认读取自己的监管机构
                log.error("群防群治行业-北京，默认读取自己的监管机构：010000");
                regionId = "010000";
            }
        } else if (industryCode.equals("04")){
            //内保行业
            if (regionId.startsWith("11")) {
                //北京，默认读取自己的监管机构
                log.error("内保行业-北京，默认读取自己的监管机构：020000");
                regionId = "020000";
            }
        }

        Object redisCache = redisUtil.get(String.format(SUPERVISE_DEPARTMENT_REDIS_KEY, regionId));
        if (redisCache != null) {
            return JSONObject.parseObject(redisCache.toString(), SuperviseDepartmentResponse.class);
        }

        //根据区域查询监管机构
        String regionStartWith = ReUtil.replaceAll(regionId, "0+$", "");;
        List<SuperviseDepartmentEntity> departmentList = superviseDepartmentRepository.findByRegionIdStartWith(regionStartWith);

        if (CollectionUtils.isNotEmpty(departmentList)) {
            //是否只返回两个层级
            boolean isReturnSecond  = regionId.startsWith("11");
            String root = "";
            //监管机构集合
            Map<String, SuperviseDepartmentResponse> departmentMap = new HashMap<>();
            for (SuperviseDepartmentEntity entity : departmentList) {
                departmentMap.put(entity.getId(), new SuperviseDepartmentResponse(entity.getId(), entity.getName(), entity.getSuperior(), entity.getIndexCode(), null));
                if (StringUtils.isBlank(entity.getSuperior())) {
                    //根节点id
                    root = entity.getId();
                }
            }
            //循环组装子节点
            for (SuperviseDepartmentEntity entity : departmentList) {
                SuperviseDepartmentResponse node = departmentMap.get(entity.getId());
                //判断是否只返回两个层级
                if (isReturnSecond) {
                    //取出indexCode判断层级
                    String indexCode = node.getIndexCode();
                    if (indexCode.split("___").length > 2) {
                        continue;
                    }
                }

                SuperviseDepartmentResponse superior = departmentMap.get(entity.getSuperior());
                if (superior != null) {
                    if (CollectionUtils.isEmpty(superior.getChildren())) {
                        superior.setChildren(new ArrayList<>());
                    }
                    superior.getChildren().add(node);
                }
            }
            SuperviseDepartmentResponse result = departmentMap.get(root);
            //放入缓存
            redisUtil.set(String.format(SUPERVISE_DEPARTMENT_REDIS_KEY, regionId), JSONObject.toJSONString(result), 3600);
            return result;
        }
        return null;
    }

    /**
     * description: 清除redis缓存
     * author: linchunpeng
     * date:  2023-05-10 18:19
     */
    @Override
    public void clearRedis(String regionId) {
        redisUtil.del(String.format(SUPERVISE_DEPARTMENT_REDIS_KEY, regionId));
    }
}
