package com.bcxin.tenant.apis.impls;

import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.Infrastructures.utils.AuthUtil;
import com.bcxin.Infrastructures.utils.DateUtil;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.tenants.InviteDepartRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.inviteDepart.InviteDepartRequest;
import com.bcxin.api.interfaces.tenants.responses.InviteDepartResponse;
import com.bcxin.tenant.domain.entities.DepartmentAdminEntity;
import com.bcxin.tenant.domain.entities.DepartmentEntity;
import com.bcxin.tenant.domain.entities.InviteDepartEntity;
import com.bcxin.tenant.domain.entities.OrganizationEntity;
import com.bcxin.tenant.domain.repositories.DepartmentAdminRepository;
import com.bcxin.tenant.domain.repositories.DepartmentRepository;
import com.bcxin.tenant.domain.repositories.InviteDepartRepository;
import com.bcxin.tenant.domain.repositories.OrganizationRepository;
import org.apache.dubbo.config.annotation.DubboService;
import org.apache.dubbo.config.annotation.Method;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author linqinglin
 * @date 2021/12/08 0008 16:49
 */

@DubboService(version = ApiConstant.VERSION,validation = "true",retries = 0,timeout = 120 *1000, methods = {
        @Method(name = "create",retries = 0,timeout = 10 *1000)
})
public class InviteDepartRpcProviderImpl implements InviteDepartRpcProvider {


    private final InviteDepartRepository inviteDepartRepository;

    private final OrganizationRepository organizationRepository;

    private final DepartmentRepository departmentRepository;

    private final DepartmentAdminRepository departmentAdminRepository;

    public InviteDepartRpcProviderImpl(InviteDepartRepository inviteDepartRepository,
                                       OrganizationRepository organizationRepository,
                                       DepartmentRepository departmentRepository,
                                       DepartmentAdminRepository departmentAdminRepository){
        this.inviteDepartRepository = inviteDepartRepository;
        this.organizationRepository = organizationRepository;
        this.departmentRepository = departmentRepository;
        this.departmentAdminRepository = departmentAdminRepository;
    }

    @Override
    public InviteDepartResponse getUnOvertime(String id) {
        Optional<InviteDepartEntity> inviteDepartOptional = inviteDepartRepository.findById(id);
        if(!inviteDepartOptional.isPresent()){
            return null;
        }

        InviteDepartEntity inviteDepart = inviteDepartOptional.get();

        if(inviteDepart.getEndTime().before(DateUtil.fromString(DateUtil.format2ShortDate(new Date())))){
            return null;
        }

        InviteDepartResponse response = new InviteDepartResponse();
        response.setId(inviteDepart.getId());
        response.setDepartId(inviteDepart.getDepartment().getId());
        response.setDepartName(inviteDepart.getDepartment().getName());
        response.setOrganizationId(inviteDepart.getOrganization().getId());
        response.setOrganizationName(inviteDepart.getOrganization().getName());
        response.setOccupationType(inviteDepart.getOccupationType());
        response.setEndTime(inviteDepart.getEndTime());
        response.setAreaCode(inviteDepart.getAreaCode());
        response.setCreator(inviteDepart.getCreator());
        return response;
    }

    @Override
    public String create(String organizationId, InviteDepartRequest request) {
        Optional<OrganizationEntity> organization = organizationRepository.findById(organizationId);
        Optional<DepartmentEntity> department = departmentRepository.findById(request.getDepartId());
        if(!department.isPresent() || department.get().isDeleted()) {
            throw new NotFoundTenantException("该部门信息无效");
        }

        InviteDepartEntity inviteDepartEntity =
                InviteDepartEntity.create(organization.get(),
                department.get(),
                request.getOccupationType(),
                request.getEndTime(),
                AuthUtil.getAreaCode(),
                AuthUtil.getCurrentOperator());
        inviteDepartRepository.save(inviteDepartEntity);

        return inviteDepartEntity.getId();
    }

    @Override
    public List<String> getDeparts(String organizationId) {
        Collection<DepartmentEntity> departList = departmentRepository.getAllByOrganId(organizationId);

        TenantUserContext.UserModel currentUser = TenantContext.getInstance().getUserContext().get();
        if (currentUser == null) {
            throw new UnAuthorizedTenantException("当前用户无效");
        }
        boolean isMaster = currentUser.isMaster();
        boolean isDepartAdmin = currentUser.isDepartAdmin();
        if (!isMaster && !isDepartAdmin) {
            throw new UnAuthorizedTenantException("只有组织管理员或部门管理员有权限操作");
        }
        List<String> deptAdminIds = new ArrayList<>();
        if (!isMaster) {
            Set<String> deptAdminTopTreeCodes = getDeptAdminTopTreeCodes(organizationId);
            deptAdminIds = departList.stream().filter(department -> deptAdminTopTreeCodes.stream().anyMatch(code -> department.getIndexTree().startsWith(code)))
                    .map(DepartmentEntity::getId).collect(Collectors.toList());
        }

        for (DepartmentEntity department : departList) {
            department.resetIndexTree("/"+department.getIndexTree().replaceAll("--","#").replaceAll("-","/")+"/");
        }

        for (DepartmentEntity departmentDto : departList) {
            for (DepartmentEntity department : departList) {
                department.resetIndexTree(department.getIndexTree().replace("/"+departmentDto.getId().replaceAll("--","#")+"/","/"+departmentDto.getName()+"/"));
            }
        }
        for (DepartmentEntity department : departList) {
            department.resetIndexTree(department.getIndexTree().substring(1,department.getIndexTree().length()-1));
        }

        List<String> result = new ArrayList<>();
        for (DepartmentEntity departmentEntity : departList) {
            if (isMaster || deptAdminIds.contains(departmentEntity.getId())) {
                result.add(departmentEntity.getIndexTree());
            }
        }
        return result;
    }

    /**
     * description: 获取部门管理员管理的所有部门顶级treeCode
     * author: linchunpeng
     * date:  2023-05-16 14:33
     */
    private Set<String> getDeptAdminTopTreeCodes(String organizationId) {
        Set<String> treeCodes = null;
        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        if (userContext.get().isDepartAdmin()) {
            //部门管理员，需要汇集根节点，提高查询速度
            Collection<DepartmentAdminEntity> departmentAdmins = departmentAdminRepository.getByEmployeeIds(organizationId, Collections.singletonList(userContext.get().getEmployeeId()));
            if (!CollectionUtils.isEmpty(departmentAdmins)) {
                treeCodes = new HashSet<>();
                Map<String, String> treeCodeMap = new HashMap<>();
                String split = "-";
                for (DepartmentAdminEntity departmentAdmin : departmentAdmins) {
                    if (org.springframework.util.StringUtils.hasLength(departmentAdmin.getDepartmentIndexTree())) {
                        treeCodeMap.put(departmentAdmin.getDepartmentIndexTree(), departmentAdmin.getDepartmentIndexTree());
                    }
                }

                for (DepartmentAdminEntity departmentAdmin : departmentAdmins) {
                    if(org.springframework.util.StringUtils.hasLength(departmentAdmin.getDepartmentIndexTree())) {
                        String[] codes = departmentAdmin.getDepartmentIndexTree().split(split);
                        StringBuffer parentCode = new StringBuffer();
                        for (int i = 0; i < codes.length; i++) {
                            if (i > 0) {
                                parentCode.append(split);
                            }
                            parentCode.append(codes[i]);

                            if (treeCodeMap.get(parentCode.toString()) != null) {
                                //有父节点
                                treeCodes.add(parentCode.toString());
                                break;
                            }
                        }
                    }
                }
            }
        }
        return treeCodes;
    }
}
