package com.bcxin.identity.domains.services.impls;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.bcxin.Infrastructures.IdWorker;
import com.bcxin.Infrastructures.SystemConstant;
import com.bcxin.Infrastructures.TenantConfigProperty;
import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.exceptions.*;
import com.bcxin.api.interfaces.commons.OperateLogRpcProvider;
import com.bcxin.api.interfaces.enums.UpdatePasswordType;
import com.bcxin.identity.domains.components.FaceIdentityForgetPasswordExecutorImpl;
import com.bcxin.identity.domains.components.ForgetPasswordExecutorImpl;
import com.bcxin.identity.domains.components.PasswordEncoder;
import com.bcxin.identity.domains.components.SmsForgetPasswordExecutor;
import com.bcxin.identity.domains.components.commands.ForgetPasswordActionExecuteCommand;
import com.bcxin.identity.domains.configs.EnvConfig;
import com.bcxin.identity.domains.entities.*;
import com.bcxin.identity.domains.enums.COAuthType;
import com.bcxin.identity.domains.enums.ForgetPasswordAction;
import com.bcxin.identity.domains.exceptions.IdentityExceptionConverter;
import com.bcxin.identity.domains.exceptions.IdentityNotFoundException;
import com.bcxin.identity.domains.readers.IdentityDbReader;
import com.bcxin.identity.domains.readers.dtos.UserReaderDto;
import com.bcxin.identity.domains.repositories.IdentityUserRepository;
import com.bcxin.identity.domains.repositories.ToCOAuthSubjectPrincipalRepository;
import com.bcxin.identity.domains.repositories.UserNamePasswordPrincipalRepository;
import com.bcxin.identity.domains.repositories.WechatPrincipalRepository;
import com.bcxin.identity.domains.repositories.dtos.IdentityUserNamePasswordDTO;
import com.bcxin.identity.domains.services.IdentityUserService;
import com.bcxin.identity.domains.services.commandResult.PrepareResetPasswordCommandResult;
import com.bcxin.identity.domains.services.commandResult.WechatCgiBinSignInCommandResult;
import com.bcxin.identity.domains.services.commands.*;
import com.bcxin.identity.domains.utils.JwtUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

@Slf4j
@Service
public class IdentityUserServiceImpl implements IdentityUserService {
    private final IdentityUserRepository identityUserRepository;
    private final UserNamePasswordPrincipalRepository userNamePasswordPrincipalRepository;
    private final WechatPrincipalRepository wechatPrincipalRepository;
    private final PasswordEncoder passwordEncoder;
    private final UnitWork unitWork;
    private final IdentityDbReader dbReader;
    private final SmsForgetPasswordExecutor smsForgetPasswordExecutor;
    private final FaceIdentityForgetPasswordExecutorImpl faceIdentityForgetPasswordExecutor;
    private final TenantConfigProperty configProperty;
    private final JsonProvider jsonProvider;
    private final EnvConfig envConfig;
    private final OperateLogRpcProvider operateLogRpcProvider;

    private final ToCOAuthSubjectPrincipalRepository toCOAuthSubjectPrincipalRepository;

    private final IdWorker idWorker;


    public IdentityUserServiceImpl(IdentityUserRepository identityUserRepository,
                                   UserNamePasswordPrincipalRepository userNamePasswordPrincipalRepository,
                                   WechatPrincipalRepository wechatPrincipalRepository,
                                   PasswordEncoder passwordEncoder,
                                   UnitWork unitWork, IdentityDbReader dbReader,
                                   SmsForgetPasswordExecutor smsForgetPasswordExecutor,
                                   FaceIdentityForgetPasswordExecutorImpl faceIdentityForgetPasswordExecutor,
                                   TenantConfigProperty configProperty,
                                   JsonProvider jsonProvider,
                                   EnvConfig envConfig,
                                   OperateLogRpcProvider operateLogRpcProvider,
                                   ToCOAuthSubjectPrincipalRepository toCOAuthSubjectPrincipalRepository,
                                   IdWorker idWorker){
        this.identityUserRepository = identityUserRepository;
        this.userNamePasswordPrincipalRepository = userNamePasswordPrincipalRepository;
        this.wechatPrincipalRepository = wechatPrincipalRepository;
        this.passwordEncoder = passwordEncoder;
        this.unitWork = unitWork;
        this.dbReader = dbReader;
        this.smsForgetPasswordExecutor = smsForgetPasswordExecutor;
        this.faceIdentityForgetPasswordExecutor = faceIdentityForgetPasswordExecutor;
        this.configProperty = configProperty;
        this.jsonProvider = jsonProvider;
        this.envConfig = envConfig;
        this.operateLogRpcProvider = operateLogRpcProvider;
        this.toCOAuthSubjectPrincipalRepository = toCOAuthSubjectPrincipalRepository;
        this.idWorker = idWorker;
    }


    @Override
    public SignInCommand.SignInCommandResult signIn(SignInCommand command) {
        IdentityUserNamePasswordDTO identityUserNamePasswordDTO =
                this.identityUserRepository.findOne(command.getUserName());
        if (identityUserNamePasswordDTO == null) {
            throw new BadTenantException("账号/密码不正确!");
        }

        UserNamePasswordPrincipalEntity principal = this.userNamePasswordPrincipalRepository.getByUserName(command.getUserName());
        //判断账号是否锁定30分钟
        if (principal.isLock()) {
            throw new AccountLockedException(String.format("您的登录密码已连续输错5次，账号已被安全锁定，请于%s分钟后重新登录。", principal.getLockRemainingTime()));
        }

        if (command.isValidPassword() &&
                !this.passwordEncoder.isMatched(identityUserNamePasswordDTO.getPassword(), command.getPassword())) {
            AtomicInteger count = new AtomicInteger();
            //登录失败增加计数
            this.unitWork.executeTran(() -> {
                count.set(principal.loginFailAddCount());
                this.userNamePasswordPrincipalRepository.save(principal);
            });
            if (count.get() >= 5) {
                throw new AccountLockedException(String.format("您的登录密码已连续输错5次，账号已被安全锁定，请于%s分钟后重新登录。", principal.getLockRemainingTime()));
            }
            throw new BadTenantException("账号/密码不正确!");
        }

        if(!StringUtils.hasLength(identityUserNamePasswordDTO.getTenantUserId()) ) {
            throw new BadTenantException("账号异常, 找不到租户信息, 请联系管理员!");
        }

        try {
            this.unitWork.executeTran(() -> {
                //登录成功，清除登录失败计数
                if (principal.getLoginFailCount() != null && principal.getLoginFailCount() > 0) {
                    principal.loginFailClearCount();
                }
                principal.recordLastLoginTime();
                this.userNamePasswordPrincipalRepository.save(principal);
            });

            //记录登录日志
            operateLogRpcProvider.loginLog(identityUserNamePasswordDTO.getTenantUserId(), command.getUserName(), identityUserNamePasswordDTO.getName(), command.getIpAddress());
        } catch (Exception e) {
            log.error("清除登录失败计数、记录操作日志异常：{}", e.getMessage(), e);
        }

        UpdatePasswordType updatePasswordType = null;
        if (command.isValidPassword()) {
            //只有需要验证密码的，才需要提示修改密码的逻辑
            updatePasswordType = principal.getUpdatePasswordType();
        }

        return SignInCommand.SignInCommandResult.create(
                identityUserNamePasswordDTO.getIdentityUserId(),
                identityUserNamePasswordDTO.getTenantUserId(),
                identityUserNamePasswordDTO.getName(),
                identityUserNamePasswordDTO.getIdNum(),
                command.isFromMobile(),
                updatePasswordType);
    }

    /**
     * description：登录失败增加计数
     * author：linchunpeng
     * date：2025/3/31
     */
    public int loginFailAddCount(UserNamePasswordPrincipalEntity principal) {
        AtomicInteger count = new AtomicInteger();
        //登录失败增加计数
        this.unitWork.executeTran(() -> {
            count.set(principal.loginFailAddCount());
            this.userNamePasswordPrincipalRepository.save(principal);
        });
        return count.get();
    }

    /**
     * description：登录成功，清除登录失败计数
     * author：linchunpeng
     * date：2025/3/31
     */
    public void loginFailClearCount(UserNamePasswordPrincipalEntity principal) {
        this.unitWork.executeTran(() -> {
            //登录成功，清除登录失败计数
            if (principal.getLoginFailCount() != null && principal.getLoginFailCount() > 0) {
                principal.loginFailClearCount();
            }
            principal.recordLastLoginTime();
            this.userNamePasswordPrincipalRepository.save(principal);
        });
    }

    @Override
    public ChangePasswordCommand.ChangePasswordCommandResult changePassword(ChangePasswordCommand command) {
        IdentityUserEntity identityUser =
                this.identityUserRepository.getByUserName(command.getIdentityUserId(), command.getUserName());

        if (identityUser == null) {
            throw new IdentityNotFoundException();
        }

        this.unitWork.executeTran(() -> {
            identityUser.changePassword(
                    command.getUserName(),
                    passwordEncoder, command.getPassword(),
                    command.getNewPassword(),
                    command.getConfirmedPassword());

            this.identityUserRepository.save(identityUser);
        });

        return ChangePasswordCommand.ChangePasswordCommandResult.create(true);
    }

    /**
     * 如果用户已经存在则忽略提交
     *
     * @param command
     */
    @Override
    public void signUp(SignUpCommand command) {
        try {
            IdentityUserEntity identityUser = IdentityUserEntity.create(command.getTenantUserId(), command.getName(), command.getTelephone(),command.getIdNum());
            String password = command.getIdNum();
            String username = envConfig.isRequiredPhoneAsLoginName() ? command.getTelephone():command.getIdNum();

            //密码默认为123456
            if (envConfig.isRequiredPhoneAsLoginName()){
                password = "123456";
            }else {
                if (command.getIdNum().length() > 6) {
                    password = command.getIdNum().substring(command.getIdNum().length() - 6);
                }
                if(!StringUtils.isEmpty(envConfig.getPassPrefix())){
                    //Abcd@身份证后六位
                    password = envConfig.getPassPrefix() + password;
                }
            }
            identityUser.addUsernamePasswordPrincipal(username, password, this.passwordEncoder);
            identityUser.changeNote(this.jsonProvider.getJson(command));
            unitWork.executeTran(() -> {
                this.identityUserRepository.save(identityUser);
            });

        } catch (Exception ex) {
            unitWork.detachAll();

            TenantExceptionAbstract tenantException = IdentityExceptionConverter.cast(ex);
            if (tenantException instanceof ConflictTenantException) {
                //可忽略的冲突异常
                log.error("预期-可忽略的异常: 用户(%s)已经存在!",command.getIdNum(),ex);
            } else {
                ex.printStackTrace();
                throw ex;
            }
        }
    }

    @Override
    public void dispatch(WechatBindCommand command) {
        command.validate();

        IdentityUserEntity identityUserEntity = this.identityUserRepository.findByTenantUserId(command.getTenantId())
                .stream().findFirst().orElse(null);
        if (identityUserEntity == null) {
            throw new NotFoundTenantException("当前用户无效");
        }

        WechatPrincipalEntity wechatPrincipalEntity = identityUserRepository.findByOpenId(command.getOpenId());
        if (wechatPrincipalEntity != null && !wechatPrincipalEntity.getIdentityUser().getTenantUserId().equals(command.getTenantId())) {
            throw new NotFoundTenantException("当前微信已经绑定过用户了，不能再次绑定，如果需要更换用户，请先解绑旧用户");
        }

        this.unitWork.executeTran(() -> {
            identityUserEntity.bindWechat(jsonProvider, command.getOpenId(), command.getNicky(), command.getUnionId());

            this.identityUserRepository.save(identityUserEntity);

            Collection<ToCOAuthSubjectPrincipalEntity> toCOAuthSubjectPrincipals =
                    toCOAuthSubjectPrincipalRepository.findSubjectsByUnionId(COAuthType.WechatCgi, command.getUnionId());
            if(toCOAuthSubjectPrincipals!=null) {
                Collection<ToCOAuthSubjectPrincipalEntity> unBindCOAuthSubjectPrincipals
                        = toCOAuthSubjectPrincipals
                        .stream().filter(ii -> SystemConstant.isEmpty(ii.getTenantUserId())).collect(Collectors.toList());

                for (ToCOAuthSubjectPrincipalEntity toCOAuthSubjectPrincipal : unBindCOAuthSubjectPrincipals) {
                    toCOAuthSubjectPrincipal.assignAssign(identityUserEntity.getTenantUserId(), identityUserEntity.getId());
                    this.toCOAuthSubjectPrincipalRepository.save(toCOAuthSubjectPrincipal);

                    log.error("绑定微信的时候同时绑定了C端登入用户的个人信息:{}", toCOAuthSubjectPrincipal.getId(), toCOAuthSubjectPrincipal.getUnionId());
                }
            }
        });
    }

    @Override
    public SignInCommand.SignInCommandResult dispatch(WechatSignInCommand command) {
        command.validate();

        WechatPrincipalEntity wechatPrincipalEntity =
                this.identityUserRepository.findByOpenId(command.getOpenId());
        if (wechatPrincipalEntity == null) {
            throw new NotFoundTenantException("账号无效");
        }

        IdentityUserEntity identityUser = wechatPrincipalEntity.getIdentityUser();
        //记录登录日志
        try {
            this.unitWork.executeTran(() -> {
                wechatPrincipalEntity.recordLastLoginTime();
                this.wechatPrincipalRepository.save(wechatPrincipalEntity);
            });
            operateLogRpcProvider.loginLog(identityUser.getTenantUserId(), command.getOpenId(), identityUser.getName(), command.getIpAddress());
        } catch (Exception e) {
            log.error("记录操作日志异常：{}", e.getMessage(), e);
        }

        return SignInCommand.SignInCommandResult.create(
                identityUser.getId(),
                identityUser.getTenantUserId(),
                identityUser.getName(),
                identityUser.getIdNum(),
                true,
                null);
    }

    @Override
    public void dispatch(UpdateIdentityBasicCommand command) {
        command.validate();
        IdentityUserEntity identityUser = this.identityUserRepository.findByTenantUserId(command.getTenantUserId())
                .stream().findFirst().orElse(null);
        if (identityUser == null) {
            throw new NotFoundTenantException("找不到用户信息");
        }
        identityUser.change(command.getName(), command.getTelephone(), command.getIdNum(), envConfig.isRequiredPhoneAsLoginName());

        this.unitWork.executeTran(() -> {
            this.identityUserRepository.save(identityUser);
        });
    }

    @Override
    public void dispatch(WechatUnBindCommand command) {
        IdentityUserEntity identityUser = this.identityUserRepository.findByTenantUserId(command.getId())
                .stream().findFirst().orElse(null);
        if (identityUser == null) {
            throw new NotFoundTenantException("账号/密码不正确!");
        }

        this.unitWork.executeTran(() -> {
            identityUser.unbindWechat();
            this.identityUserRepository.save(identityUser);
        });
    }


    /**
     * description：判断是否绑定微信unionid
     * author：linchunpeng
     * date：2023/10/12
     */
    @Override
    public boolean dispatch(WechatCheckUnionIdCommand command) {
        IdentityUserEntity identityUser = this.identityUserRepository
                .findByTenantUserId(command.getId()).stream().findFirst().orElse(null);
        if (identityUser == null) {
            return false;
        }
        if (identityUser.getPrincipals() == null || identityUser.getPrincipals().size()==0) {
            return false;
        }

        Optional<PrincipalAbstract> principalOptional =
                identityUser.getPrincipals().stream().filter(ii -> ii.getClass().isAssignableFrom(WechatPrincipalEntity.class)).findFirst();
        if (!principalOptional.isPresent()) {
            return false;
        }

        WechatPrincipalEntity wechatPrincipalEntity = (WechatPrincipalEntity) principalOptional.get();
        return StringUtils.hasLength(wechatPrincipalEntity.getUnionId());
    }

    @Override
    public PrepareResetPasswordCommandResult dispatch(PrepareResetPasswordCommand command) {
        command.validate();
        UserReaderDto userReaderDto = this.dbReader.getByUserName(command.getLoginName());
        if (userReaderDto == null) {
            throw new NotFoundTenantException("找不到用户信息");
        }

        String faceImage = userReaderDto.getHeadPhoto();

        if (StringUtils.isEmpty(faceImage) && command.getAction().equals(ForgetPasswordAction.FaceIdentity)) {
            throw new NotFoundTenantException("该用户未上传身份证头像，无法使用人脸认证找回密码");
        }

        if (StringUtils.hasLength(faceImage) && !faceImage.startsWith("http")) {
            faceImage = String.format("%s%s", configProperty.getSite(), faceImage);
        }

        return getByMessageType(command.getAction()).execute(ForgetPasswordActionExecuteCommand.create(
                userReaderDto.getStatus(),
                command.getLoginName(),
                userReaderDto.getTelephone(),
                userReaderDto.getName(),
                faceImage,
                command.getData()
        ));
    }
    @Override
    public PrepareResetPasswordCommandResult getUserMobile(PrepareResetPasswordCommand command) {
        UserReaderDto userReaderDto = this.dbReader.getByUserName(command.getLoginName());
        if (userReaderDto == null) {
            throw new NotFoundTenantException("找不到用户信息");
        }
        return PrepareResetPasswordCommandResult.create(null,true,null,userReaderDto.getTelephone(), StrUtil.isNotEmpty(userReaderDto.getHeadPhoto()));
    }
    @Override
    public void dispatch(ResetPasswordCommandAdmin command){
        IdentityUserEntity identityUser =
                this.identityUserRepository.getDirectlyByUserName(command.getLoginName());

        if (identityUser == null) {
            throw new IdentityNotFoundException();
        }
        //判断是否有新密码 没有的话默认用后六位 如果账号小于六位则密码跟账号一样
        if (!StringUtils.hasLength(command.getNewPassword())){
            String password = command.getLoginName();
            if (password.length() > 6){
                password = command.getLoginName().substring(command.getLoginName().length() - 6);
            }
            if(!StringUtils.isEmpty(envConfig.getPassPrefix())){
                //Abcd@身份证后六位
                password = envConfig.getPassPrefix() + password;
            }
            command.setNewPassword(password);
        }

        this.unitWork.executeTran(() -> {
            identityUser.resetPassword(
                    command.getLoginName(),
                    passwordEncoder,
                    command.getNewPassword(),
                    command.getNewPassword(),
                    true);

            this.identityUserRepository.save(identityUser);
        });

    };


    @Override
    public void resetPasswordCheckCode(ResetPasswordCommand command) {
        command.validate();
        String json = JwtUtil.parseJwt(command.getJwt());
        Map<String, Object> map = (Map<String, Object>) JSON.parse(json);
        String time = (String) map.get("time");
        String GetPasswordType = (String) map.get("GetPasswordType");
        String lName = (String) map.get("loginName");
        String sendCode = (String) map.get("code");

        boolean flg = compareDate(time + "");
        if (!lName.equals(command.getLoginName())) {
            throw new BadTenantException("登录账号认证出错请检查");
        }
        if (flg) {
            if ("1".equals(GetPasswordType)) {
                //短信修改密码
                if (StringUtils.hasLength(command.getCode()) && command.getCode().equals(sendCode)) {
                    //验证通过
                } else {
                    throw new BadTenantException("短信验证,验证码出错");
                }
            } else {
                throw new BadTenantException("非短信找回密码，无法校验验证码！");
            }
        } else {
            throw new BadTenantException("token超时,请重新获取");
        }
    }

    @Override
    public void changeUserName(ChangeUserNameCommand command) {
        if (StrUtil.isEmpty(command.getUserName())){
            throw new BadTenantException("修改登录名失败,登录名不能为空！");
        }
        String newPassword = command.getUserName();

        //密码默认为123456
        if (envConfig.isRequiredPhoneAsLoginName()){
            newPassword = "123456";
        }else {
            if (command.getUserName().length() > 6) {
                newPassword = command.getUserName().substring(command.getUserName().length() - 6);
            }
            if(!StringUtils.isEmpty(envConfig.getPassPrefix())){
                //Abcd@身份证后六位
                newPassword = envConfig.getPassPrefix() + newPassword;
            }
        }
//        String newPassword = command.getUserName().length()<= 6?command.getUserName():command.getUserName().substring(command.getUserName().length()-6);
        IdentityUserEntity identityUserEntity = this.identityUserRepository.findByTenantUserId(command.getUserId())
                .stream().findFirst().orElse(null);
        if (identityUserEntity == null) {
            if (identityUserEntity == null) {
                throw new IdentityNotFoundException();
            }
        }

        try {
            String finalNewPassword = newPassword;
            this.unitWork.executeTran(()->{
                Optional<UserNamePasswordPrincipalEntity> userNamePasswordOptional =
                        identityUserEntity.getPrincipals().stream().filter(ii -> ii instanceof UserNamePasswordPrincipalEntity)
                                .map(ii -> (UserNamePasswordPrincipalEntity) ii)
                                .findFirst();
                if (userNamePasswordOptional.isPresent()) {
                    userNamePasswordOptional.get().changeUserName(command.getUserName());
                    userNamePasswordOptional.get().resetPassword(passwordEncoder, finalNewPassword, finalNewPassword, false);
                }

                this.identityUserRepository.save(identityUserEntity);
            });
        }catch (Exception e) {
            if (e.getMessage().contains("identity_username_password_username")) {
                throw new BadTenantException("修改失败,登录名已存在");
            }
            throw new BadTenantException("修改登录名失败！");

        }
    }

    @Override
    public WechatCgiBinSignInCommandResult dispatch(WechatCgiBinSignInCommand command) {
        ToCOAuthSubjectPrincipalEntity principal =
                this.toCOAuthSubjectPrincipalRepository.getByUnionIdAndOpenId(
                        COAuthType.WechatCgi,
                        command.getUnionId(),
                        command.getOpenId()
                );

        /**
         * 针对C端的用户; 第一次使用的时候, 不需要创建真正的用户
         * 系统在做身份核验的时候，我们将更换和找到对应的identityUser和tenantUsers;
         */
        if (principal == null) {
            IdentityUserEntity identityUser = this.identityUserRepository.findIdentityUserByUnionId(command.getUnionId());
            principal = ToCOAuthSubjectPrincipalEntity.create(
                    COAuthType.WechatCgi,
                    String.valueOf(idWorker.nextId()),
                    command.getOpenId(),
                    command.getUnionId(),
                    command.getNicky(),
                    command.getSysCode(),
                    identityUser
            );

            ToCOAuthSubjectPrincipalEntity finalPrincipal = principal;
            this.unitWork.executeTran(() -> {
                this.toCOAuthSubjectPrincipalRepository.save(finalPrincipal);
            });
        }

        return WechatCgiBinSignInCommandResult.create(
                principal.getOpenId(),
                principal.getUnionId(),
                principal.getIdentityUserId(),
                principal.getTenantUserId(),
                principal.getNicky()
        );
    }

    @Override
    public void dispatch(TocUserAssignIdentityCommand command) {
        ToCOAuthSubjectPrincipalEntity subjectPrincipal =
                this.toCOAuthSubjectPrincipalRepository.getByUnionIdAndOpenId(
                        COAuthType.WechatCgi, command.getUnionId(), command.getOpenId()
                );
        if (subjectPrincipal == null) {
            throw new NotFoundTenantException();
        }


        String tranId = this.unitWork.beginTransaction();
        try {
            Collection<IdentityUserEntity> identityUsers =
                    this.identityUserRepository.findByTenantUserId(command.getTenantUserId());
            IdentityUserEntity identityUser = identityUsers.stream().findFirst().orElse(null);
            if (identityUser == null) {
                identityUser = IdentityUserEntity.create(command.getTenantUserId(),
                        command.getName(), command.getPhone(),
                        command.getTenantUserId());

                this.identityUserRepository.save(identityUser);
            }else {
                if (identityUsers.stream().count() > 1) {
                    log.error("TocUserAssignIdentityCommand: 当前找到的用户信息:{};count={}", command.getTenantUserId(), identityUsers.stream().count());
                }
            }

            subjectPrincipal.assignAssign(identityUser.getTenantUserId(), identityUser.getId());

            this.toCOAuthSubjectPrincipalRepository.save(subjectPrincipal);

            this.unitWork.commit(tranId);
        }
        catch (Exception ex) {
            this.unitWork.rollback(tranId);

            throw new BadTenantException(String.format("关联用户异常:%s",ex.getMessage()), ex);
        }

    }

    @Override
    public void dispatch(ResetPasswordCommand command) {
        command.validate();
        IdentityUserEntity identityUser =
                this.identityUserRepository.getDirectlyByUserName(command.getLoginName());

        if (identityUser == null) {
            throw new IdentityNotFoundException();
        }

        String json = JwtUtil.parseJwt(command.getJwt());
        if(!StringUtils.hasLength(json)) {
            throw new ArgumentTenantException(String.format("参数解析异常:%s", command.getJwt()));
        }

        Map<String, Object> map = (Map<String, Object>) JSON.parse(json);
        String time = (String) map.get("time");
        String GetPasswordType = (String) map.get("GetPasswordType");
        String lName = (String) map.get("loginName");
        String sendCode = (String) map.get("code");

        boolean flg = compareDate(time + "");
        if (!lName.equals(command.getLoginName())) {
            throw new BadTenantException("登录账号认证出错请检查");
            // return new ResultMsg(false, "-2", "登录账号认证出错请检查");
        }
        if (flg) {
            if ("1".equals(GetPasswordType)) {
                //短信修改密码
                this.unitWork.executeTran(() -> {
                    identityUser.resetPassword(
                            command.getLoginName(),
                            passwordEncoder,
                            command.getNewPassword(),
                            command.getNewPassword(),
                            false);

                    this.identityUserRepository.save(identityUser);
                });
            } else {
                if ("0".equals(GetPasswordType)) {
                    //人脸认证修改密码
                    //修改密码
                    this.unitWork.executeTran(() -> {
                        identityUser.resetPassword(
                                command.getLoginName(),
                                passwordEncoder,
                                command.getNewPassword(),
                                command.getNewPassword(),
                                false);

                        this.identityUserRepository.save(identityUser);
                    });
                }
            }
        } else {
            throw new BadTenantException("token超时,请重新获取");
        }
    }

    //判断两个日期是否大于3分钟
    public static boolean compareDate(String time) {
        Calendar dateOne = Calendar.getInstance();
        Calendar dateTwo = Calendar.getInstance();
        dateOne.setTime(new Date());    //设置为当前系统时间

        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Date date = null;
        try {
            date = simpleDateFormat.parse(time);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        dateTwo.setTime(date);
        long timeOne = dateOne.getTimeInMillis();
        long timeTwo = dateTwo.getTimeInMillis();
        long minute = (timeOne - timeTwo) / (1000 * 60);//转化minute
        if (minute <= 3) {
            //小于3分钟
            return true;
        }
        return false;
    }

    private ForgetPasswordExecutorImpl getByMessageType(ForgetPasswordAction action) {
        if (this == null) {
            throw new BadTenantException("忘记密码的方式不能为空");
        }

        switch (action) {
            case SMS:
                return this.smsForgetPasswordExecutor;
            case FaceIdentity:
                return this.faceIdentityForgetPasswordExecutor;
        }

        throw new NotSupportTenantException("不支持该忘记密码的实现方案");
    }
}
