package com.bcxin.tenant.domain.entities;

import com.bcxin.Infrastructures.enums.CredentialType;
import com.bcxin.Infrastructures.enums.TrueFalseStatus;
import com.bcxin.Infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.utils.UUIDUtil;
import com.bcxin.Infrastructures.validations.CredentialValidator;
import com.bcxin.tenant.domain.DomainConstraint;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.springframework.util.StringUtils;

import javax.persistence.*;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.Date;

/**
 * 证件类型
 */
@Getter
@Setter(AccessLevel.PROTECTED)
@Table(name = "tenant_user_credentials",uniqueConstraints =
@UniqueConstraint(name = DomainConstraint.UNIQUE_TENANT_USER_CREDENTIAL_NUMBER,columnNames = {"credential_type", "number"}))
@Entity
public class TenantUserCredentialsEntity {
    @Id
    private String id;

    /**
     * 证件姓名
     */
    @Column(name = "name",length = 50,nullable = false)
    private String name;

    @Enumerated(EnumType.ORDINAL)
    @Column(name = "credential_type", nullable = false)
    private CredentialType credentialType;

    @Column(name = "number")
    private String number;

    @Temporal(TemporalType.DATE)
    @Column(name = "valid_date_from", nullable = true)
    private Date validDateFrom;

    @Temporal(TemporalType.DATE)
    @Column(name = "valid_date_to", nullable = true)
    private Date validDateTo;
    /**
     * 证件头像
     */
    private String headPhoto;

    /**
     * 正面照：比如: 身份证人像面
     */
    @Column(name = "front_photo", length = 500)
    private String frontPhoto;

    /**
     * 反面照：比如: 身份证国徽面
     */
    @Column(name = "reverse_photo", length = 500)
    private String reversePhoto;

    /**
     * 证件地址: 比如: 身份证住址
     */
    @Column(name = "address", length = 500)
    private String address;

    @Column(name = "created_time", nullable = false)
    private Timestamp CreatedTime;

    @ManyToOne
    @JoinColumn(name = "tenant_user_id", referencedColumnName = "id")
    private TenantUserEntity tenantUser;

    @Column(name = "selected",nullable = false)
    private TrueFalseStatus selected;

    public void changeDetail(String headPhoto,
                             String frontPhoto,
                             String reversePhoto,
                             String address,
                             Date validDateFrom,
                             Date validDateTo) {
        if(!StringUtils.hasLength(headPhoto)){
            throw new ArgumentTenantException("身份证头像照片不能为空");
        }
        if(!StringUtils.hasLength(frontPhoto)){
            throw new ArgumentTenantException("身份证人像面照片不能为空");
        }
        if(!StringUtils.hasLength(reversePhoto)){
            throw new ArgumentTenantException("身份证国徽面照片不能为空");
        }
        this.setHeadPhoto(headPhoto);
        this.setFrontPhoto(frontPhoto);
        this.setReversePhoto(reversePhoto);
        this.setAddress(address);
        this.setValidDateFrom(validDateFrom);
        this.setValidDateTo(validDateTo);
    }

    public void changeDetail(
            CredentialType credentialType,
            String name,
            String number,
            String headPhoto,String frontPhoto,
            String reversePhoto, String address,
            Date validDateFrom, Date validDateTo) {
        if (credentialType == null) {
            throw new ArgumentTenantException("证件类型不能为空");
        }

        if (!StringUtils.hasLength(name)) {
            throw new ArgumentTenantException("证件名不能为空");
        }
        if (!StringUtils.hasLength(number)) {
            throw new ArgumentTenantException("证件号不能为空");
        }

        if (!CredentialValidator.validate(credentialType, number)) {
            throw new ArgumentTenantException(String.format("证件号码(%s)格式不正确", number));
        }

        this.setCredentialType(credentialType);
        this.setName(name);
        this.setNumber(number);
        this.setHeadPhoto(headPhoto);
        this.setFrontPhoto(frontPhoto);
        this.setReversePhoto(reversePhoto);
        this.setAddress(address);
        this.setValidDateFrom(validDateFrom);
        this.setValidDateTo(validDateTo);
        this.markSelected(true);
    }

    public void changeName(String name) {
        this.setName(name);
    }

    protected TenantUserCredentialsEntity() {
        this.markSelected(true);
        this.setCreatedTime(Timestamp.from(Instant.now()));
    }

    public void markSelected(boolean selected) {
        this.setSelected(selected ? TrueFalseStatus.True : TrueFalseStatus.False);
    }

    public static TenantUserCredentialsEntity create(
            TenantUserEntity tenantUser,CredentialType credentialType,String number) {
        TenantUserCredentialsEntity credential = new TenantUserCredentialsEntity();
        if (!StringUtils.hasLength(number)) {
            throw new BadTenantException("证件号不能为空");
        }

        if (!CredentialValidator.validate(credentialType, number)) {
            throw new BadTenantException(String.format("证件编号(%s)不正确", number));
        }

        credential.setTenantUser(tenantUser);
        credential.setName(tenantUser.getName());
        credential.setCredentialType(credentialType);
        credential.setNumber(number);
        credential.setId(UUIDUtil.getShortUuid());

        return credential;
    }
}
