/**
 * Copyright (c) 2005-2012
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 */
package com.bcxin.ars.util.spring.util;

import com.abcxin.smart.validator.annotation.DataSyncAnnotation;
import com.abcxin.smart.validator.annotation.DataSyncOutAnnotation;
import com.abcxin.smart.validator.enums.SyncType;
import org.springframework.aop.framework.Advised;
import org.springframework.aop.target.SingletonTargetSource;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import java.lang.annotation.Annotation;
import java.util.*;
import java.util.stream.Collectors;

/**
 * spring工具类
 *
 * @author xuerdong
 * @date 2017/1/8 12:10
 */
@Component
public final class SpringUtils implements ApplicationContextAware,BeanFactoryPostProcessor {

    private static ConfigurableListableBeanFactory beanFactory; // Spring应用上下文环境

    private static ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        SpringUtils.applicationContext = applicationContext;
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        SpringUtils.beanFactory = beanFactory;
    }

    /**
     * 获取对象
     *
     * @param name
     * @return Object 一个以所给名字注册的bean的实例
     * @throws BeansException
     */
    @SuppressWarnings("unchecked")
    public static <T> T getBean(String name) throws BeansException {
        return (T) beanFactory.getBean(name);
    }

    /**
     * 获取类型为requiredType的对象
     *
     * @param clz
     * @return
     * @throws BeansException
     */
    public static <T> T getBean(Class<T> clz) throws BeansException {
        @SuppressWarnings("unchecked")
        T result = (T) beanFactory.getBean(clz);
        return result;
    }

    /**
     * 如果BeanFactory包含一个与所给名称匹配的bean定义，则返回true
     *
     * @param name
     * @return boolean
     */
    public static boolean containsBean(String name) {
        return beanFactory.containsBean(name);
    }

    /**
     * 判断以给定名字注册的bean定义是一个singleton还是一个prototype。 如果与给定名字相应的bean定义没有被找到，将会抛出一个异常（NoSuchBeanDefinitionException）
     *
     * @param name
     * @return boolean
     * @throws NoSuchBeanDefinitionException
     */
    public static boolean isSingleton(String name) throws NoSuchBeanDefinitionException {
        return beanFactory.isSingleton(name);
    }

    /**
     * @param name
     * @return Class 注册对象的类型
     * @throws NoSuchBeanDefinitionException
     */
    public static Class<?> getType(String name) throws NoSuchBeanDefinitionException {
        return beanFactory.getType(name);
    }

    /**
     * 如果给定的bean名字在bean定义中有别名，则返回这些别名
     *
     * @param name
     * @return
     * @throws NoSuchBeanDefinitionException
     */
    public static String[] getAliases(String name) throws NoSuchBeanDefinitionException {
        return beanFactory.getAliases(name);
    }


    /**
     * 获取同步执行排序后的数据同步Service Bean 列表
     * @param clazz
     * @return
     */
    public static List<Map.Entry<String,Object>> getDataSyncBeanList_sync(Class clazz){
        Map<String, Object> beans = getBeansByAnnotation(clazz);

        List<Map.Entry<String,Object>> list = new ArrayList<Map.Entry<String,Object>>(beans.entrySet());

        //获取同步执行的数据同步Service Bean 列表
        List<Map.Entry<String,Object>> maps = list.stream().filter((map)-> SyncType.ASYNC.equals(getSyncType(map.getValue(),clazz))).collect(Collectors.toList());

        //排序
        Collections.sort(maps,new Comparator<Map.Entry<String,Object>>() {
            //升序排序 比较器
            @Override
            public int compare(Map.Entry<String, Object> map1, Map.Entry<String, Object> map2) {
                //compare（a,b）方法:根据第一个参数小于、等于或大于第二个参数分别返回负整数、零或正整数。
                Object o1 = map1.getValue();
                Object o2 = map2.getValue();
                //根据seq排序
                return getDataSyncSeq(map1.getValue(),clazz) - getDataSyncSeq(map2.getValue(),clazz);
            }
        });
        return maps;
    }


    /**
     * 获取同步执行排序后的数据同步Service Bean 列表
     * @param clazz
     * @return
     */
    public static List<Map.Entry<String,Object>> getDataSyncBeanList_assync(Class clazz){
        Map<String, Object> beans = getBeansByAnnotation(clazz);

        List<Map.Entry<String,Object>> list = new ArrayList<Map.Entry<String,Object>>(beans.entrySet());

        //获取同步执行的数据同步Service Bean 列表
        List<Map.Entry<String,Object>> maps = list.stream().filter((map)-> SyncType.SYNC.equals(getSyncType(map.getValue(),clazz))).collect(Collectors.toList());

        //排序
        Collections.sort(maps,new Comparator<Map.Entry<String,Object>>() {
            //升序排序 比较器
            @Override
            public int compare(Map.Entry<String, Object> map1, Map.Entry<String, Object> map2) {
                //compare（a,b）方法:根据第一个参数小于、等于或大于第二个参数分别返回负整数、零或正整数。
                Object o1 = map1.getValue();
                Object o2 = map2.getValue();
                //根据seq排序
                return getDataSyncSeq(map1.getValue(),clazz) - getDataSyncSeq(map2.getValue(),clazz);
            }
        });
        return maps;
    }
    /**
     * 获取 数据同步Service Bean 的seq
     * @param entityService
     * @return
     */
    public static int getDataSyncSeq(Object entityService, Class clazz){
        Class entityServiceClazz = entityService.getClass();
        Annotation annotation = entityServiceClazz.getAnnotation(clazz);

        if(annotation instanceof DataSyncAnnotation){
            DataSyncAnnotation dataSyncAnnotation = (DataSyncAnnotation) annotation;
            return dataSyncAnnotation.getSeq();
        }else if(annotation instanceof DataSyncOutAnnotation){
            DataSyncOutAnnotation dataSyncAnnotation = (DataSyncOutAnnotation) annotation;
            return dataSyncAnnotation.getSeq();
        }
        return 0;
    }
    /**
     * 获取数据同步执行方式
     * @param entityService
     * @param clazz
     * @return
     */
    public static SyncType getSyncType(Object entityService,Class clazz){
        Class entityServiceClazz = entityService.getClass();
        Annotation annotation = entityServiceClazz.getAnnotation(clazz);

        if(annotation instanceof DataSyncAnnotation){
            DataSyncAnnotation dataSyncAnnotation = (DataSyncAnnotation) annotation;
            return dataSyncAnnotation.getDataSyncType();
        }else if(annotation instanceof DataSyncOutAnnotation){
            DataSyncOutAnnotation dataSyncAnnotation = (DataSyncOutAnnotation) annotation;
            return dataSyncAnnotation.getDataSyncType();
        }
        return null;
    }

    public static Map<String, Object> getBeansByAnnotation(Class clazz){
        Map<String, Object> beans = applicationContext.getBeansWithAnnotation(clazz);
        Map<String, Object> result = new HashMap<>();
        for (String key : beans.keySet()) {
            Object entityService = beans.get(key);
            Class beanClass = entityService.getClass();
            //使用@Transactional注解取到的是代理类
            if("Proxy".equals(beanClass.getSuperclass().getSimpleName())){
                //代理类取到对应的service bean
                Advised advised = (Advised)entityService;
                SingletonTargetSource singTarget = (SingletonTargetSource) advised.getTargetSource();
                Object target = singTarget.getTarget();
                result.put(key,target);
            }else{
                //非代理类
                result.put(key,entityService);
            }
        }
        return result;
    }
}
