package com.bcxin.ars.timer.approval;

import cn.hutool.core.thread.ThreadUtil;
import com.bcxin.ars.model.Config;
import com.bcxin.ars.model.sys.Approval;
import com.bcxin.ars.model.sys.JobRunLog;
import com.bcxin.ars.service.ConfigService;
import com.bcxin.ars.service.sys.ApprovalService;
import com.bcxin.ars.service.sys.JobRunLogService;
import com.bcxin.ars.util.Constants;
import com.bcxin.ars.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadPoolExecutor;

/***
 * 自动将未启动流程启动
 * @author subh
 */
@Service
public class AutoApprovalProcess {
    /***
     * 日志
     */
    private Logger logger = LoggerFactory.getLogger(AutoApprovalProcess.class);
    /***
     * 审批记录
     */
    @Autowired
    private ApprovalService approvalService;
    @Value("${timeFlag}")
    private String timeFlag = "";
    @Autowired
    private JobRunLogService jobRunLogService;
    /***
     * 配置信息
     */
    @Autowired
    private ConfigService configService;
    public void run() {
        Config config = configService.findByKey(Constants.POLICE);
        if (config.getValue().equals(Constants.POLICE_IN) && timeFlag.equals(Constants.TIMEFLAG_VALUE)) {
            //获取没有启动的审批列表（保安员资格证与年报除外）
            List<Approval> approvalList = approvalService.findUnStartProcess();
            if (approvalList != null && approvalList.size() > 0) {
                JobRunLog log = new JobRunLog();
                log.setCreateTime(new Date());
                log.setActive(true);
                log.setUpdateBy("jobSystem");
                //开始时间
                long startTime = System.currentTimeMillis();
                    //背景筛查
                   startCensorStatus(approvalList,log);
                   //启动流程
                   startApproval(approvalList,log);
                //日志信息
                long endTime = System.currentTimeMillis();
                log.setUpdateTime(new Date());
                log.setJobName(AutoApprovalProcess.class.getName());
                log.setRunTime(new Date());
                log.setRunTimeLength((endTime - startTime) + "ms");
                jobRunLogService.insert(log);
            }
        }
    }
    /****
     * 启动流程
     * @param approvalList 启动列表
     */
    public void startCensorStatus(List<Approval> approvalList,JobRunLog jobRunLog){
        //使用线程池
        CountDownLatch countDownLatch = ThreadUtil.newCountDownLatch(approvalList.size());
        ThreadPoolExecutor threadPoolExecutor = ThreadUtil.newExecutor(15, 15);
        for (Approval approval : approvalList) {
            threadPoolExecutor.execute(new Runnable(){
                @Override
                public void run() {
                    try {
                        approvalService.startCensorStatus(approval);
                        //背景筛查无异常
                        approval.setCensorStatusError(false);
                    } catch (Exception e) {
                        //背景筛查异常
                        approval.setCensorStatusError(true);
                        //日志
                        jobRunLog.setExceptionMsg(jobRunLog.getExceptionMsg()!=null?jobRunLog.getExceptionMsg():""+approval.getId()+"背景筛查失败");
                        logger.error(e.getMessage(),e);
                    }finally {
                        countDownLatch.countDown();
                    }
                }
            });
        }
        try {
            //线程等待全部完成
            countDownLatch.await();
        } catch (InterruptedException e) {
            logger.error(e.getMessage(), e);
            Thread.currentThread().interrupt();
        }finally {
            threadPoolExecutor.shutdown();
        }
    }
    /****
     * 启动流程
     * @param approvalList 启动列表
     */
    public void startApproval(List<Approval> approvalList,JobRunLog jobRunLog){
        Long policeId = 0L;
        String testState = StringUtil.EMPTY;
        //使用线程池
        CountDownLatch countDownLatch = ThreadUtil.newCountDownLatch(approvalList.size());
        ThreadPoolExecutor threadPoolExecutor = ThreadUtil.newExecutor(15, 15);
        for (Approval approval : approvalList) {
            threadPoolExecutor.execute(new Runnable(){
                @Override
                public void run() {
                    try {
                        if(!approval.getCensorStatusError()) {
                            approvalService.startApproval(approval, policeId, testState);
                        }
                    } catch (Exception e) {
                        logger.error(e.getMessage(),e);
                        //日志
                        jobRunLog.setExceptionMsg(jobRunLog.getExceptionMsg()+approval.getId()+"流程启动失败");
                    }finally {
                        countDownLatch.countDown();
                    }
                }
            });
        }
        try {
            //线程等待全部完成
            countDownLatch.await();
        } catch (InterruptedException e) {
            logger.error(e.getMessage(), e);
            Thread.currentThread().interrupt();
        }finally {
            threadPoolExecutor.shutdown();
        }
    }
}
