package com.bcxin.rest.web.apis.controllers;

import cn.hutool.core.util.StrUtil;
import com.bcxin.Infrastructures.components.CacheV2Provider;
import com.bcxin.Infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.api.interfaces.tenants.UserRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.TenantUserRegionCriteria;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.*;
import com.bcxin.api.interfaces.tenants.responses.UserAppealsResponse;
import com.bcxin.api.interfaces.tenants.responses.UserCredentialResponse;
import com.bcxin.api.interfaces.tenants.responses.UserRegionGetResponse;
import com.bcxin.rest.web.apis.utils.CacheUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Collection;
import java.util.List;

@Api(value = "用户管理api", tags = "用户管理api")
@RestController
@RequestMapping("/tenant/users")
public class TenantUserController extends ControllerAbstract{
    private final UserRpcProvider userRpcProvider;
    private final CacheV2Provider cacheV2Provider;

    public TenantUserController(UserRpcProvider userRpcProvider,CacheV2Provider cacheV2Provider) {
        this.userRpcProvider = userRpcProvider;
        this.cacheV2Provider = cacheV2Provider;
    }

    @ApiOperation(value = "更改用户其他基本", response = Void.class)
    @PutMapping("/{id}/detail")
    public ResponseEntity put(@PathVariable String id, @RequestBody UpdateTenantUserRequest request) {
        this.userRpcProvider.update(id, request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }

    @ApiOperation(value = "变更用户姓名/手机信息", response = Void.class)
    @PutMapping("/{id}/basic")
    public ResponseEntity put_basic(@PathVariable String id, @RequestBody UpdateBasicTenantUserRequest request) {
        this.userRpcProvider.updateBasic(id, request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }


    @ApiOperation(value = "查询用户cid", response = Void.class)
    @GetMapping("/get-cid/{phone}")
    public ResponseEntity put_basic(@PathVariable String phone) {
        String cid = this.userRpcProvider.getCidByPhone(phone);
        return this.ok(cid);
    }


    @ApiOperation(value = "变更用户证件信息", response = Void.class)
    @PostMapping("/{id}/credential")
    public ResponseEntity put_credential(@PathVariable String id, @RequestBody UpdateCredentialRequest request) {
        this.userRpcProvider.updateCredential(id, request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }

    @ApiOperation(value = "根据监管地区查询用户信息", response = Void.class)
    @PostMapping("/get-user/region")
    public ResponseEntity getUserByRegion(@RequestBody TenantUserRegionCriteria criteria) {
        if (StrUtil.isEmpty(criteria.getRegion())) {
            throw new ArgumentTenantException("缺少参数:region");
        }
        List<UserRegionGetResponse> userData = this.userRpcProvider.getUsersByRegion(criteria);
        return this.ok(userData);
    }


    @ApiOperation(value = "修改用户实名结果", response = Void.class)
    @PostMapping("/update-user/real-name")
    public ResponseEntity updateRealName(@RequestBody List<UpdateTenantUserRealNameRequest> userRealNameRequests) {
        this.userRpcProvider.updateUserRealName(userRealNameRequests);
        return this.ok();
    }

    @ApiOperation(value = "根据tenanuserid，证件编号，证件类型返回有效的证件信息", response = UserCredentialResponse.class)
    @GetMapping("/{id}/{credentialType}/{credentialNumber}")
    public ResponseEntity getUserCredential(@PathVariable String id,@PathVariable Integer credentialType, @PathVariable String credentialNumber) {
        UserCredentialResponse result = this.userRpcProvider.getUserCredential(id,credentialType,credentialNumber);
        return this.ok(result);
    }

    @ApiOperation(value = "根据用户Id和事件Id重复短信")
    @PostMapping("/{id}/message/{eventId}")
    public ResponseEntity reTriggerMessageWithEventId(@PathVariable String id,@PathVariable String eventId) {
        this.userRpcProvider.notifyMessage(id, eventId);

        return this.ok();
    }

    @ApiOperation(value = "新增实名认证申诉")
    @PostMapping("/{id}/authentication/appeals")
    public ResponseEntity createAuthAppeal(@PathVariable String id, @RequestBody CreateAuthAppealRequest createAppealRequest) {
        this.userRpcProvider.createAuthAppeal(id, createAppealRequest);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileCacheKey());
        return this.ok();
    }

    @ApiOperation(value = "查询人员申诉记录")
    @PostMapping ("/authentication/appeals/search")
    public ResponseEntity getAuthAppeals(@RequestBody GetAuthAppealsRequest request) {
        Collection<UserAppealsResponse> result = this.userRpcProvider.getUserAppeals(request);
        return this.ok(result);
    }

    @ApiOperation(value = "保存用户拓展信息", response = Void.class)
    @PostMapping("/{id}/save/extension")
    public ResponseEntity saveExtension(@PathVariable String id, @Valid @RequestBody TenantUserExtensionListRequest request) {
        this.userRpcProvider.saveTenantUserExtension(id, request);
        this.cacheV2Provider.del(CacheUtil.getCurrentProfileEmployeeDeletedCachedKeys());
        return this.ok();
    }
}
