package com.bcxin.rest.web.apis.controllers;

import com.bcxin.Infrastructures.enums.RelationshipsStatus;
import com.bcxin.api.interfaces.tenants.OrganizationRelationshipRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.organizationRelationship.OrganizationRelationshipCreateRequest;
import com.bcxin.api.interfaces.tenants.requests.organizationRelationship.OrganizationRelationshipInfoUpdateRequest;
import com.bcxin.api.interfaces.tenants.requests.organizationRelationship.OrganizationRelationshipStatusUpdateRequest;
import com.bcxin.api.interfaces.tenants.requests.organizationRelationship.OrganizationStructureQueryRequest;
import com.bcxin.api.interfaces.tenants.responses.OrgRelationResponse;
import com.bcxin.api.interfaces.tenants.responses.OrganizationRelationshipResponse;
import com.bcxin.api.interfaces.tenants.responses.OrganizationStructureResponse;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.v3.oas.annotations.Parameter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import javax.validation.Valid;
import java.util.Collection;
import java.util.List;

@RestController
@RequestMapping("/tenant/organizations/{organizationId}/organizationRelationship")
@Api(value = "集团组织关系api", tags = "集团组织关系api")
public class OrganizationRelationshipController extends ControllerAbstract{

    private final Logger logger = LoggerFactory.getLogger(OrganizationRelationshipController.class);

    private final OrganizationRelationshipRpcProvider organizationRelationshipRpcProvider;

    public OrganizationRelationshipController(OrganizationRelationshipRpcProvider organizationRelationshipRpcProvider) {
        this.organizationRelationshipRpcProvider = organizationRelationshipRpcProvider;
    }

    @ApiOperation(value = "子公司/分公司申请加入集团")
    @PostMapping("/join")
    public ResponseEntity join(@Valid @RequestBody OrganizationRelationshipCreateRequest organizationRelationshipCreateRequest){
        organizationRelationshipRpcProvider.join(organizationRelationshipCreateRequest);
        return ok();
    }


    @ApiOperation(value = "更新集团组织状态")
    @PutMapping("/status/update")
    public ResponseEntity updateStatus(@PathVariable String organizationId,@Valid @RequestBody OrganizationRelationshipStatusUpdateRequest request){
        organizationRelationshipRpcProvider.updateStatus(organizationId,request);
        return ok();
    }

    @ApiOperation(value = "更新组织关系详情")
    @PutMapping("/info")
    public ResponseEntity updateOrganizationInfo(@PathVariable String organizationId, @Valid @RequestBody OrganizationRelationshipInfoUpdateRequest request){
        request.setOrganizationId(organizationId);
        organizationRelationshipRpcProvider.updateRelationshipInfo(request);
        return ok();
    }

    @ApiOperation(value = "查询集团组织关系详情")
    @GetMapping("/findById/{id}")
    public ResponseEntity findById(@PathVariable long id){
        OrganizationRelationshipResponse organizationRelationshipResponse = organizationRelationshipRpcProvider.findRelationshipDetailById(id);
        return ok(organizationRelationshipResponse);
    }

    @ApiOperation(value = "查询集团组织架构",response = OrganizationStructureResponse.class)
    @GetMapping("/structure")
    public ResponseEntity<OrganizationStructureResponse> getOrganizationStructure(@PathVariable String organizationId,
                                                                                  @RequestParam(required = false)List<RelationshipsStatus> statuses){
        OrganizationStructureQueryRequest request = OrganizationStructureQueryRequest.create(organizationId,statuses);
        OrganizationStructureResponse orgStructureResponse = organizationRelationshipRpcProvider.getOrganizationStructure(request);
        return ok(orgStructureResponse);
    }

    @ApiOperation(value = "删除组织集团关系")
    @DeleteMapping ("/delete")
    public ResponseEntity delete(@PathVariable String organizationId,@Valid @RequestBody Collection<Long> orIdS){
        organizationRelationshipRpcProvider.delete(orIdS,organizationId);
        return ok();
    }

    @ApiOperation(value = "获取上级组织列表")
    @GetMapping("/getParentList")
    public ResponseEntity getParentList(@PathVariable String organizationId) {
        List<OrgRelationResponse> response = this.organizationRelationshipRpcProvider.getParentList(organizationId);
        return this.ok(response);
    }

    @ApiOperation(value = "获取下级组织列表")
    @GetMapping("/getChildrenList")
    public ResponseEntity getChildrenList(@PathVariable String organizationId) {
        List<OrgRelationResponse> response = this.organizationRelationshipRpcProvider.getChildrenList(organizationId);
        return this.ok(response);
    }
}
