package com.bcxin.tenant.backend.configs;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.enums.CredentialType;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.backend.constants.KafkaConstants;
import com.bcxin.tenant.domain.services.TenantUserService;
import com.bcxin.tenant.domain.services.commands.BatchCreateTenantUserCommand;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 考试报名用户Kafka监听组件
 * 监听tms.binlog-cdc.topic.v2.bus_exam_enrollment_users主题，同步创建tenant_users
 */
@Component
@ConditionalOnProperty("spring.kafka.bootstrap-servers")
public class ExamEnrollmentUserKafkaComponent {
    private static final Logger logger = LoggerFactory.getLogger(ExamEnrollmentUserKafkaComponent.class);
    
    private final TenantUserService userService;
    private final JsonProvider jsonProvider;

    public ExamEnrollmentUserKafkaComponent(TenantUserService userService, JsonProvider jsonProvider) {
        this.userService = userService;
        this.jsonProvider = jsonProvider;
    }

    /**
     * 订阅考试报名用户信息
     * @param records Kafka消息记录
     * @param ack 确认对象
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-exam-enrollment-users",
            topics = {KafkaConstants.TMS_EXAM_ENROLLMENT_USERS_TOPIC},
            groupId = "${spring.kafka.consumer.group-id}-exam-enrollment-users")
    public void ackExamEnrollmentUsersListener(List<ConsumerRecord<String, String>> records,
                                                 Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.warn("收到空的Kafka消息记录");
            ack.acknowledge();
            return;
        }

        boolean allowed2CommitAtFinal = true;
        Exception lastException = null;
        
        try {
            // 过滤出新增操作的记录（before为null，after不为null）
            List<ConsumerRecord<String, String>> createRecords = records.stream()
                    .filter(record -> {
                        try {
                            JSONObject data = jsonProvider.toObject(JSONObject.class, record.value());
                            if (data == null) {
                                return false;
                            }
                            JSONObject before = data.getJSONObject("before");
                            JSONObject after = data.getJSONObject("after");
                            // 只处理新增操作
                            return before == null && after != null;
                        } catch (Exception e) {
                            logger.error("解析Kafka消息失败，offset={}, partition={}, key={}", 
                                    record.offset(), record.partition(), record.key(), e);
                            return false;
                        }
                    })
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(createRecords)) {
                logger.debug("没有需要处理的新增记录，跳过处理");
                // 即使没有需要处理的记录，也要确认消息，避免重复消费
                ack.acknowledge();
                return;
            }

            logger.info("开始处理{}条新增考试报名用户记录", createRecords.size());

            // 转换为BatchCreateTenantUserCommand.TenantUserCommandItem
            Collection<BatchCreateTenantUserCommand.TenantUserCommandItem> commandItems = 
                    createRecords.stream()
                            .map(this::convertToCommandItem)
                            .filter(item -> item != null)
                            .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(commandItems)) {
                logger.warn("转换后的命令项为空，跳过用户创建。原始记录数：{}，转换失败数：{}", 
                        createRecords.size(), createRecords.size() - commandItems.size());
                // 即使转换失败，也要确认消息，避免重复消费
                ack.acknowledge();
                return;
            }

            logger.info("准备创建{}个tenant_users，原始记录数：{}", commandItems.size(), createRecords.size());

            // 调用userService.create创建用户
            // 注意：userService.create会返回所有用户（包括已存在的和新创建的）
            Collection<?> tenantUsers = userService.create(BatchCreateTenantUserCommand.create(commandItems));
            
            int actualCreatedCount = tenantUsers != null ? tenantUsers.size() : 0;
            logger.info("userService.create返回{}个tenant_users（包括已存在的和新创建的），期望创建{}个", 
                    actualCreatedCount, commandItems.size());
            
            // 验证创建结果
            if (actualCreatedCount == 0) {
                logger.warn("userService.create返回空集合，可能创建失败。commandItems数量：{}", commandItems.size());
            } else if (actualCreatedCount < commandItems.size()) {
                logger.warn("userService.create返回的用户数({})少于期望数({})，可能部分用户已存在", 
                        actualCreatedCount, commandItems.size());
            }

        } catch (Exception e) {
            logger.error("处理考试报名用户Kafka消息时发生异常", e);
            allowed2CommitAtFinal = false;
            lastException = e;
        } finally {
            if (allowed2CommitAtFinal) {
                ack.acknowledge();
            } else {
                if (lastException != null) {
                    throw new BadTenantException(
                            String.format("Failed to consume topic=%s", KafkaConstants.TMS_EXAM_ENROLLMENT_USERS_TOPIC),
                            lastException);
                }
            }
        }
    }

    /**
     * 将Kafka消息记录转换为TenantUserCommandItem
     * @param record Kafka消息记录
     * @return TenantUserCommandItem，如果转换失败返回null
     */
    private BatchCreateTenantUserCommand.TenantUserCommandItem convertToCommandItem(
            ConsumerRecord<String, String> record) {
        try {
            JSONObject data = jsonProvider.toObject(JSONObject.class, record.value());
            if (data == null) {
                logger.warn("消息数据为空，offset={}, key={}", record.offset(), record.key());
                return null;
            }

            JSONObject after = data.getJSONObject("after");
            if (after == null) {
                logger.warn("after节点为空，offset={}, key={}", record.offset(), record.key());
                return null;
            }

            // 从after节点提取基础字段
            String name = after.getString("name");
            String telephone = after.getString("telephone");
            String credentialNumber = after.getString("credential_number");
            String sysDynamicJson = after.getString("sys_dynamic_json");

            // 验证必填字段
            if (!StringUtils.hasLength(name)) {
                logger.warn("姓名为空，跳过该记录，offset={}, key={}", record.offset(), record.key());
                return null;
            }
            if (!StringUtils.hasLength(telephone)) {
                logger.warn("手机号为空，跳过该记录，offset={}, key={}", record.offset(), record.key());
                return null;
            }
            if (!StringUtils.hasLength(credentialNumber)) {
                logger.warn("证件号为空，跳过该记录，offset={}, key={}", record.offset(), record.key());
                return null;
            }

            // 从sys_dynamic_json中提取credential_type
            CredentialType credentialType = extractCredentialType(sysDynamicJson);
            if (credentialType == null) {
                // 如果无法提取，默认使用IdCard
                credentialType = CredentialType.IdCard;
                logger.debug("无法从sys_dynamic_json提取credential_type，使用默认值IdCard，offset={}, key={}", 
                        record.offset(), record.key());
            }

            // nationality字段，如果数据中没有则使用null
            String nationality = null;

            logger.debug("转换用户数据：name={}, telephone={}, credentialNumber={}, credentialType={}, offset={}", 
                    name, telephone, credentialNumber, credentialType, record.offset());

            return BatchCreateTenantUserCommand.TenantUserCommandItem.create(
                    name,
                    telephone,
                    credentialType,
                    credentialNumber,
                    nationality
            );

        } catch (Exception e) {
            logger.error("转换Kafka消息为TenantUserCommandItem时发生异常，offset={}, key={}", 
                    record.offset(), record.key(), e);
            return null;
        }
    }

    /**
     * 从sys_dynamic_json中提取credential_type
     * sys_dynamic_json格式示例：
     * {
     *   "items": {
     *     "credential_type": "IdCard",
     *     ...
     *   },
     *   ...
     * }
     * @param sysDynamicJson sys_dynamic_json字符串
     * @return CredentialType枚举，如果无法提取则返回null
     */
    private CredentialType extractCredentialType(String sysDynamicJson) {
        if (!StringUtils.hasLength(sysDynamicJson)) {
            return null;
        }

        try {
            JSONObject dynamicJson = jsonProvider.toObject(JSONObject.class, sysDynamicJson);
            if (dynamicJson == null) {
                return null;
            }

            JSONObject items = dynamicJson.getJSONObject("items");
            if (items == null) {
                return null;
            }

            String credentialTypeStr = items.getString("credential_type");
            if (!StringUtils.hasLength(credentialTypeStr)) {
                return null;
            }

            // 将字符串转换为CredentialType枚举
            return convertStringToCredentialType(credentialTypeStr);

        } catch (Exception e) {
            logger.warn("解析sys_dynamic_json提取credential_type时发生异常，sysDynamicJson={}", sysDynamicJson, e);
            return null;
        }
    }

    /**
     * 将字符串转换为CredentialType枚举
     * @param credentialTypeStr 证件类型字符串，如"IdCard"
     * @return CredentialType枚举，如果无法匹配则返回null
     */
    private CredentialType convertStringToCredentialType(String credentialTypeStr) {
        if (!StringUtils.hasLength(credentialTypeStr)) {
            return null;
        }

        try {
            // 尝试直接通过名称匹配
            for (CredentialType type : CredentialType.values()) {
                if (type.name().equalsIgnoreCase(credentialTypeStr)) {
                    return type;
                }
            }

            // 如果直接匹配失败，尝试通过getTypeName匹配
            // 但根据示例数据，应该直接使用枚举名称，所以这里先不实现

            logger.warn("无法匹配credential_type字符串：{}，使用默认值IdCard", credentialTypeStr);
            return CredentialType.IdCard;

        } catch (Exception e) {
            logger.warn("转换credential_type字符串时发生异常，credentialTypeStr={}", credentialTypeStr, e);
            return CredentialType.IdCard;
        }
    }
}