package com.bcxin.tenant.backend.components.handlers.impls;

import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.components.RetryProvider;
import com.bcxin.Infrastructures.enums.CredentialType;
import com.bcxin.Infrastructures.enums.ProcessedStatus;
import com.bcxin.api.interfaces.identities.IdentityRpcProvider;
import com.bcxin.api.interfaces.identities.requests.SignUpRequest;
import com.bcxin.tenant.backend.components.handlers.SubscriberEventActionHandler;
import com.bcxin.tenant.domain.entities.*;
import com.bcxin.tenant.domain.entities.reports.EmployeeReportEntity;
import com.bcxin.tenant.domain.enums.EventAction;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.repositories.*;
import com.bcxin.tenant.domain.snapshots.TenantEmployeeSnapshot;
import com.bcxin.tenant.domain.utils.ExecuteUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 同步推送保安员的用户信息
 */
@Component
public class SyncSubscriberEventActionHandlerImpl implements SubscriberEventActionHandler {
    private final JsonProvider jsonProvider;
    private final TenantDbReader dbReader;
    private final UnitWork unitWork;
    private final EmployeeReportRepository employeeReportRepository;
    private final EventSubscriberRepository eventSubscriberRepository;
    private final TenantUserRepository tenantUserRepository;
    private final IdentityRpcProvider identityRpcProvider;
    private final TenantEventRepository tenantEventRepository;
    private final RetryProvider retryProvider;
    private final EventSubscribedActionLogRepository eventSubscribedActionLogRepository;

    public SyncSubscriberEventActionHandlerImpl(JsonProvider jsonProvider,
                                                TenantDbReader dbReader,
                                                UnitWork unitWork,
                                                EmployeeReportRepository employeeReportRepository,
                                                EventSubscriberRepository eventSubscriberRepository,
                                                TenantUserRepository tenantUserRepository,
                                                IdentityRpcProvider identityRpcProvider,
                                                TenantEventRepository tenantEventRepository,
                                                RetryProvider retryProvider,
                                                EventSubscribedActionLogRepository eventSubscribedActionLogRepository) {
        this.jsonProvider = jsonProvider;
        this.dbReader = dbReader;
        this.unitWork = unitWork;
        this.employeeReportRepository = employeeReportRepository;
        this.eventSubscriberRepository = eventSubscriberRepository;
        this.tenantUserRepository = tenantUserRepository;
        this.identityRpcProvider = identityRpcProvider;
        this.tenantEventRepository = tenantEventRepository;
        this.retryProvider = retryProvider;
        this.eventSubscribedActionLogRepository = eventSubscribedActionLogRepository;
    }

    @Override
    public void execute(EventSubscriberEntity subscriber,Collection<String> eventIds) {
        if (!StringUtils.hasLength(subscriber.getSelector())) {
            return;
        }

        Collection<String> identitySelectorMapKeys = Stream.of(EventAction.TenantUserAfterCreatedEventForIdentity.name()).collect(Collectors.toList());

        if (identitySelectorMapKeys.contains(subscriber.getSelector())) {
            syncIdentity(subscriber, identitySelectorMapKeys, eventIds);
        }
    }

    private void syncIdentity(EventSubscriberEntity subscriber,Collection<String> mapKeys,Collection<String> eventIds) {
        org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(SyncSubscriberEventActionHandlerImpl.class);
        logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 开始处理Identity同步, 订阅者: {}, 选择器: {}, 事件ID列表: {}", 
                subscriber.getName(), mapKeys, eventIds);
        
        Collection<TenantEventEntity> tenantEvents = getNBySelector(subscriber.getLastVersion(),mapKeys,eventIds);
        if (tenantEvents.size() == 0) {
            logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 未查询到事件, 订阅者: {}, 版本: {}", 
                    subscriber.getName(), subscriber.getLastVersion());
            return;
        }

        logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 查询到事件数量: {}, 事件ID: {}", 
                tenantEvents.size(), tenantEvents.stream().map(TenantEventEntity::getId).collect(Collectors.toList()));

        Collection<String> tenantIds =
                tenantEvents.stream().map(ii -> ii.getContent()).collect(Collectors.toList());

        logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 查询用户信息, 用户ID列表: {}", tenantIds);
        
        Collection<TenantUserEntity> tenantUsers = this.tenantUserRepository.getByIds(tenantIds);
        logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 查询到用户数量: {}, 用户ID: {}", 
                tenantUsers.size(), tenantUsers.stream().map(TenantUserEntity::getId).collect(Collectors.toList()));
        
        if (tenantUsers.size() != tenantIds.size()) {
            logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 用户数量不匹配, 期望: {}, 实际: {}", 
                    tenantIds.size(), tenantUsers.size());
        }
        
        tenantUsers.forEach(tenantUser -> {
            Optional<TenantEventEntity> tenantEventOptional = tenantEvents.stream().filter(ii -> ii.getContent().equals(tenantUser.getId()))
                    .findFirst();
            if (tenantEventOptional.isPresent()) {
                TenantEventEntity event = tenantEventOptional.get();
                logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 开始处理用户Identity同步, 用户ID: {}, 姓名: {}, 手机号: {}, 事件ID: {}", 
                        tenantUser.getId(), tenantUser.getName(), tenantUser.getTelephone(), event.getId());
                
                ExecuteUtils.executeTenantEventAction(event, subscriber, () -> {
                            Collection<SignUpRequest.CredentialRequest> credentialRequests = new ArrayList<>();

                            final Collection<CredentialType> validCredentialTypes =
                                    Arrays.stream(CredentialType.values()).filter(ii -> ii.isIdentityNo())
                                            .collect(Collectors.toList());
                            Collection<TenantUserCredentialsEntity> validTenantUserCredentials =
                                    tenantUser.getCredentials().stream()
                                            .filter(ii -> validCredentialTypes.contains(ii.getCredentialType())).collect(Collectors.toList());

                            logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 用户证件信息, 用户ID: {}, 有效证件数量: {}", 
                                    tenantUser.getId(), validTenantUserCredentials.size());

                            for (TenantUserCredentialsEntity credential : validTenantUserCredentials) {
                                credentialRequests.add(SignUpRequest.CredentialRequest.create(credential.getCredentialType(), credential.getNumber()));
                                logger.debug("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 添加证件请求, 用户ID: {}, 证件类型: {}, 证件号: {}", 
                                        tenantUser.getId(), credential.getCredentialType(), credential.getNumber());
                            }

                            String selectedCredentialNumber = tenantUser.getSelectedCredential() != null ? 
                                    tenantUser.getSelectedCredential().getNumber() : null;
                            
                            logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 构建SignUpRequest, 用户ID: {}, 姓名: {}, 手机号: {}, 选中证件号: {}, 证件请求数量: {}", 
                                    tenantUser.getId(), tenantUser.getName(), tenantUser.getTelephone(), 
                                    selectedCredentialNumber, credentialRequests.size());
                            
                            SignUpRequest request = SignUpRequest.create(
                                    tenantUser.getId(), tenantUser.getTelephone(), selectedCredentialNumber, tenantUser.getName(), credentialRequests);

                            logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 开始调用identityRpcProvider.signUp, 用户ID: {}", tenantUser.getId());
                            
                            try {
                                this.identityRpcProvider.signUp(request);
                                logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] identityRpcProvider.signUp调用成功, 用户ID: {}, 姓名: {}, 手机号: {}", 
                                        tenantUser.getId(), tenantUser.getName(), tenantUser.getTelephone());
                            } catch (Exception e) {
                                logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] identityRpcProvider.signUp调用失败, 用户ID: {}, 姓名: {}, 手机号: {}, 错误信息: {}", 
                                        tenantUser.getId(), tenantUser.getName(), tenantUser.getTelephone(), 
                                        com.bcxin.Infrastructures.utils.ExceptionUtil.getStackMessage(e), e);
                                throw e;
                            }
                        }, retryProvider, eventSubscribedActionLogRepository,
                        eventSubscriberRepository, tenantEventRepository, unitWork);
            } else {
                logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] 未找到对应事件, 用户ID: {}", tenantUser.getId());
            }
        });
        
        logger.error("[SyncSubscriberEventActionHandlerImpl.syncIdentity] Identity同步处理完成, 订阅者: {}, 处理用户数量: {}", 
                subscriber.getName(), tenantUsers.size());
    }

    private Collection<TenantEventEntity> getNBySelector(Long version, Collection<String> mapKeys,Collection<String> eventIds)
    {
        Collection<TenantEventEntity> tenantEvents = this.dbReader.getTopNTenantEvents(version, mapKeys, eventIds,100);

        return tenantEvents;
    }
}
