package com.bcxin.ferry.strategy.receive.impl;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.ferry.common.emus.FileTypeEnum;
import com.bcxin.ferry.dtos.FerryFileDto;
import com.bcxin.ferry.entity.FerryTaskFileEntity;
import com.bcxin.ferry.service.FerryReceiveTaskFileService;
import com.bcxin.ferry.service.FerryReceiveTaskService;
import com.bcxin.ferry.strategy.receive.ReceiveStrategy;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * description：接收任务文件策略
 * author：linchunpeng
 * date：2024/3/6
 */
@Slf4j
@Service
public class ReceiveTaskFileStrategy implements ReceiveStrategy {

    @Autowired
    private FerryReceiveTaskService ferryReceiveTaskService;
    @Autowired
    private RedissonClient redissonClient;

    @Async
    @Override
    public void handle(String data) {
        //摆渡任务id不存在，才需要处理
        //分布式锁key
        String lockKey = "REDISSON_LOCK_RECEIVE_TASK_FILE";
        //取锁
        log.info("接收摆渡任务文件， lockKey：{}，取锁中.....", lockKey);
        RLock lock = redissonClient.getLock(lockKey);
        //加锁，并设置过期时间 300s
        lock.lock(300, TimeUnit.SECONDS);
        log.info("取到锁");
        try {
            FerryFileDto ferryFileDto = JSONObject.parseObject(data, FerryFileDto.class);
            log.info("接收任务文件策略，基本参数");
            log.info("ferryTaskId：{}", ferryFileDto.getFerryTaskId());
            log.info("requestId：{}", ferryFileDto.getRequestId());
            log.info("fileId：{}", ferryFileDto.getFileId());
            log.info("fileType：{}", ferryFileDto.getFileType());
            //接收摆渡任务文件
            String packageUrl = ferryReceiveTaskService.receiveRequest(ferryFileDto);
            if (ferryFileDto.getFileType() != FileTypeEnum.NORMAL.getCode()) {
                //文件类型不是普通文件
                log.info("接收摆渡任务文件，需要转文件的base64字符串，文件id：{}", ferryFileDto.getFileId());
                ferryFileDto.putFileContent(data);
            }
            //解析任务文件
            List<FerryTaskFileEntity> ferryTaskFileEntityList = ferryReceiveTaskService.parseTaskFile(ferryFileDto);
            //初始化任务信息
            ferryReceiveTaskService.initTaskInfo(ferryFileDto, ferryTaskFileEntityList, packageUrl);
            //任务文件回调请求端
            ferryReceiveTaskService.taskFileCallback(ferryFileDto);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("接收摆渡任务文件异常，{}", e.getMessage(), e);
        } finally {
            if (lock.isLocked()) {
                lock.unlock();
            }
            log.info("接收摆渡任务文件，lockKey：{}，解锁", lockKey);
        }
    }
}
